import { IInput } from './input';
/**
 * Expression for events in Detector Model state.
 * @see https://docs.aws.amazon.com/iotevents/latest/developerguide/iotevents-expressions.html
 */
export declare abstract class Expression {
    /**
     * Create a expression from the given string.
     */
    static fromString(value: string): Expression;
    /**
     * Create a expression for function `currentInput()`.
     * It is evaluated to true if the specified input message was received.
     */
    static currentInput(input: IInput): Expression;
    /**
     * Create a expression for function `timeout("timer-name")`.
     * It is evaluated to true if the specified timer has elapsed.
     * You can define a timer only using the `setTimer` action.
     */
    static timeout(timerName: string): Expression;
    /**
     * Create a expression for get an input attribute as `$input.TemperatureInput.temperatures[2]`.
     */
    static inputAttribute(input: IInput, path: string): Expression;
    /**
     * Create a expression for the Addition operator.
     */
    static add(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Subtraction operator.
     */
    static subtract(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Division operator.
     */
    static divide(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Multiplication operator.
     */
    static multiply(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the String Concatenation operator.
     */
    static concat(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Bitwise OR operator.
     */
    static bitwiseOr(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Bitwise AND operator.
     */
    static bitwiseAnd(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Bitwise XOR operator.
     */
    static bitwiseXor(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Equal operator.
     */
    static eq(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Not Equal operator.
     */
    static neq(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Less Than operator.
     */
    static lt(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Less Than Or Equal operator.
     */
    static lte(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Greater Than operator.
     */
    static gt(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the Greater Than Or Equal operator.
     */
    static gte(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the AND operator.
     */
    static and(left: Expression, right: Expression): Expression;
    /**
     * Create a expression for the OR operator.
     */
    static or(left: Expression, right: Expression): Expression;
    constructor();
    /**
     * This is called to evaluate the expression.
     *
     * @param parentPriority priority of the parent of this expression,
     *   used for determining whether or not to add parenthesis around the expression.
     *   This is intended to be set according to MDN rules, see
     *   https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Operator_Precedence#table
     *   for details
     */
    abstract evaluate(parentPriority?: number): string;
}
