import { Resource } from 'aws-cdk-lib';
import { IVpc, SubnetSelection, InstanceType, IMachineImage } from 'aws-cdk-lib/aws-ec2';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
/**
 * Properties for NAT instances
 */
export interface SimpleNATProps {
    /**
     * The VPC the NAT instances will reside
     */
    readonly vpc: IVpc;
    /**
     * The subnet selection for NAT instances, one NAT instance will be placed in the selected subnets.
     *
     * NOTE: must select the public subnet
     *
     * @default - subnetType is SubnetType.PUBLIC and onePerAZ is true.
     */
    readonly natSubnetsSelection?: SubnetSelection;
    /**
     * The subnet selection for updating route tables for selected subnets.
     *
     * @default - subnetType is SubnetType.PRIVATE_WITH_NAT.
     */
    readonly privateSubnetsSelection?: SubnetSelection;
    /**
     * The instance type of NAT instances
     *
     * @default - t3.MICRO.
     */
    readonly instanceType?: InstanceType;
    /**
     * The AMI of NAT instances
     *
     * @default - Amazon Linux 2 for x86_64.
     */
    readonly machineImage?: IMachineImage;
    /**
     * The key name of ssh key of NAT instances.
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * The custom script when provisioning the NAT instances.
     *
     * @default - no custom script.
     */
    readonly customScripts?: string;
    /**
     * The IAM role attached to NAT instances.
     *
     * @default - an IAM role is created.
     */
    readonly role?: IRole;
}
/**
 * Properties for how adding IPs to route
 */
export interface RouteProps {
    /**
     * If excluding IPv6 when creating route
     *
     * @default - false
     */
    readonly excludeIPv6?: boolean;
}
/**
 * Simple NAT instaces construct.
 */
export declare class SimpleNAT extends Resource {
    static readonly Ipv6Regex = "^s*((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]d|1dd|[1-9]?d)(.(25[0-5]|2[0-4]d|1dd|[1-9]?d)){3}))|:)))(%.+)?s*(/([0-9]|[1-9][0-9]|1[0-1][0-9]|12[0-8]))?$";
    private gateways;
    private _securityGroup;
    private _routeMappingSubnets;
    private _routeTablesLimit;
    private readonly _defaultRoutesPerTable;
    private readonly _ipV6Regex;
    constructor(scope: Construct, id: string, props: SimpleNATProps);
    addV4Route(v4CIDR: string): SimpleNAT;
    addV6Route(v6CIDR: string): SimpleNAT;
    /**
     * Add Github IPs to route table
     */
    withGithubRoute(props?: RouteProps): SimpleNAT;
    /**
     * Add Google IPs to route table
     */
    withGoogleRoute(props?: RouteProps): SimpleNAT;
    /**
     * Add Cloudflare IPs to route table
     *
     * See https://www.cloudflare.com/ips/ for details
     */
    withCloudflareRoute(props?: RouteProps): SimpleNAT;
    private _configureSubnet;
    private _addRoute;
}
