# -*- coding: utf-8 -*-
#
#   Ming-Ke-Ming : Decentralized User Identity Authentication
#
#                                Written in 2019 by Moky <albert.moky@gmail.com>
#
# ==============================================================================
# MIT License
#
# Copyright (c) 2019 Albert Moky
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# ==============================================================================

from abc import ABCMeta, abstractmethod
from typing import Optional

from .crypto import PrivateKey
from .identifier import ID
from .meta import Meta
from .profile import Profile


class EntityDataSource(metaclass=ABCMeta):
    """This interface is for getting information for entity(user/group)

        Entity Data Source
        ~~~~~~~~~~~~~~~~~~

        1. meta for user, which is generated by the user's private key
        2. meta for group, which is generated by the founder's private key
        3. meta key, which can verify message sent by this user(or group founder)
        4. profile key, which can encrypt message for the receiver(user)
    """

    @abstractmethod
    def meta(self, identifier: ID) -> Optional[Meta]:
        """
        Get meta for entity ID

        :param identifier: entity ID
        :return:           meta info
        """
        pass

    @abstractmethod
    def profile(self, identifier: ID) -> Optional[Profile]:
        """
        Get profile for entity ID

        :param identifier: entity ID
        :return:           profile info
        """
        pass


class UserDataSource(EntityDataSource):
    """This interface is for getting private information for local user

        Local User Data Source
        ~~~~~~~~~~~~~~~~~~~~~~

        1. private key for signature, is the key matching with meta.key;
        2. private key for decryption, is the key matching with profile.key,
           if profile.key not exists, means it is the same key pair with meta.key
    """

    @abstractmethod
    def private_key_for_signature(self, identifier: ID) -> Optional[PrivateKey]:
        """
        Get user's private key for signature

        :param identifier: user ID
        :return: private key
        """
        pass

    @abstractmethod
    def private_keys_for_decryption(self, identifier: ID) -> Optional[list]:
        """
        Get user's private keys for decryption

        :param identifier: user ID
        :return: private keys
        """
        pass

    @abstractmethod
    def contacts(self, identifier: ID) -> Optional[list]:
        """
        Get user's contacts list

        :param identifier: user ID
        :return: contacts list (ID)
        """
        pass


class GroupDataSource(EntityDataSource):
    """This interface is for getting information for group

        Group Data Source
        ~~~~~~~~~~~~~~~~~

        1. founder has the same public key with the group's meta.key
        2. owner and members should be set complying with the consensus algorithm
    """

    @abstractmethod
    def founder(self, identifier: ID) -> Optional[ID]:
        """ Get founder of the group """
        pass

    @abstractmethod
    def owner(self, identifier: ID) -> Optional[ID]:
        """ Get current owner of the group """
        pass

    @abstractmethod
    def members(self, identifier: ID) -> Optional[list]:
        """ Get all members in the group """
        pass
