"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk = require("../lib");
const lib_1 = require("../lib");
function createModernApp() {
    return new cdk.App({
        context: {
            [cxapi.DISABLE_VERSION_REPORTING]: 'true',
        }
    });
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.synth();
        // THEN
        test.same(app.synth(), session); // same session if we synth() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {});
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.synth();
        // THEN
        test.deepEqual(list(session.directory), [
            'cdk.out',
            'manifest.json',
            'one-stack.template.json'
        ]);
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.assembly.outdir, 'foo.json', { bar: 123 });
                s.assembly.addArtifact('my-random-construct', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json'
                    }
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.synth();
        // THEN
        test.deepEqual(list(session.directory), [
            'cdk.out',
            'foo.json',
            'manifest.json',
            'one-stack.template.json'
        ]);
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: '0.36.0',
            artifacts: {
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' }
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                }
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends lib_1.Construct {
            constructor() {
                super(undefined, 'id');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.assembly.addArtifact('art', {
                    type: cxapi.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    properties: {
                        templateFile: 'hey.json',
                        parameters: {
                            paramId: 'paramValue',
                            paramId2: 'paramValue2'
                        }
                    },
                    environment: 'aws://unknown-account/us-east-1'
                });
                writeJson(session.assembly.outdir, 'hey.json', { hello: 123 });
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
        }
        const root = new SynthesizeMe();
        const assembly = lib_1.ConstructNode.synth(root.node, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStack('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.templateFile, 'hey.json');
        test.deepEqual(stack.parameters, { paramId: 'paramValue', paramId2: 'paramValue2' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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