"""Basic ADC models."""

import numpy as np


def dac(samples, nbits=8, vref=1):
    """Digital to analog converter."""
    quants = 2**nbits
    dv = vref / quants
    return samples * dv


class ADC:
    """
    Generic ADC Class.

    Parameters
    ----------
    nbits : int, default=8
        Number of bits for the ADC.
    fs : int or float, default=1
        Sample rate for the ADC in Hz.
    vref : int or float, default=1
        Reference level of the ADC in Volts ([0, +vref] conversion range).
    seed : int, default=1
        Seed for random variable generation.
    **kwargs
        Extra arguments.

    Attributes
    -------
    vin : float
        Sets or returns the current input voltage level. Assumed +/-vref/2 input
    vlsb : float
        LSB voltage of the converter. vref/2^nbits
    noise : float, default=0
        Sets or returns the stdev of the noise generated by the converter.
    mismatch : float, default=0
        Sets or returns the stdev of the mismatch of the converter.
    offset : tuple of float, default=(0, 0)
        Sets the (mean, stdev) of the offset of the converter.
    gain_error : tuple of float, default=(0, 0)
        Sets the (mean, stdev) of the gain error of the converter.
    distortion : list of float, default=[1]
        Sets the harmonic distortion values with index=0 corresponding to HD1.
        Example: For unity gain and only -30dB of HD3, input is [1, 0, 0.032]
    dout : int
        Digital output code for current vin value.

    Methods
    -------
    run_step

    """

    def __init__(self, nbits=8, fs=1, vref=1, seed=1, **kwargs):
        """Initialization function for Generic ADC."""
        np.random.seed(seed)
        self.nbits = nbits
        self.fs = fs
        self.vref = vref
        self.seed = seed
        self.err = {"noise": 0, "gain": 0, "dist": [1], "offset": 0, "mismatch": 0}
        self.dbits = np.zeros(nbits)
        self.dval = 0

    @property
    def vin(self):
        """Return input value."""
        return self._vin

    @vin.setter
    def vin(self, x):
        """Set the input value."""
        x += self.vref / 2
        x = max(0, min(x, self.vref))
        self._vin = x

    @property
    def vlsb(self):
        """Return the LSB voltage."""
        return self.vref / 2**self.nbits

    @property
    def noise(self):
        """Return noise status."""
        return self.err["noise"]

    @noise.setter
    def noise(self, stdev):
        """Set noise stdev in Vrms."""
        self.err["noise"] = stdev

    @property
    def mismatch(self):
        """Return noise stdev."""
        print("WARNING: 'mismatch' feature not implemented for this class.")
        return False

    @mismatch.setter
    def mismatch(self, stdev):
        """Set mismatch stdev."""
        print("WARNING: 'mismatch' feature not implemented for this class.")
        pass

    @property
    def offset(self):
        """Return offset value."""
        return self.err["offset"]

    @offset.setter
    def offset(self, values):
        """Set offset mean and stdev."""
        self.err["offset"] = np.random.normal(values[0], values[1])

    @property
    def gain_error(self):
        """Return gain error status."""
        return self.err["gain"]

    @gain_error.setter
    def gain_error(self, values):
        """Set gain error mean and stdev."""
        self.err["gain"] = np.random.normal(values[0], values[1])

    @property
    def distortion(self):
        """Return distortion gains (1st-order indexed)."""
        return self.err["dist"]

    @distortion.setter
    def distortion(self, gains):
        """Set distortion gains (1st-order indexed)."""
        self.err["dist"] = gains

    @property
    def dout(self):
        """Return digital output code."""
        return int(self.dval)

    def run_step(self):
        """Run a single ADC step."""
        vinx = self.vin
        dval = int(
            min(max(int((2**self.nbits) * vinx / self.vref), 0), 2**self.nbits - 1)
        )
        bits = [int(x) for x in bin(dval)[2:]]

        while len(bits) < self.nbits:
            bits.insert(0, 0)
        self.dbits = bits
        self.dval = dval
