from typing import Optional, Dict, Any, Mapping, Sequence, Union, List

from clutch.network.rpc.message import Request
from clutch.network.rpc.torrent.mutator import TorrentMutator, TorrentMutatorArguments
from clutch.network.rpc.typing import FlatTrackerReplaceArg, TrackerReplace

hyphenated_arguments: Sequence[str] = [
    'files-wanted',
    'files-unwanted',
    'peer-limit',
    'priority-high',
    'priority-low',
    'priority-normal'
]


def _convert(key: str) -> str:
    hyphenated = key.replace("_", "-")
    if hyphenated in hyphenated_arguments:
        words = hyphenated.split("-")
        for word in words[1:]:
            word.capitalize()
        return "-".join(words)
    else:
        words = hyphenated.split("-")
        for word in words[1:]:
            word.capitalize()
        return "".join(words)


def _clone_and_convert_keys(arguments: Mapping[str, Any]) -> Dict[str, Any]:
    result = dict(arguments)
    for (k, v) in arguments.items():
        result[_convert(k)] = v
    return result


def convert_mutator(mutator: TorrentMutator) -> Optional[Request]:
    def flatten(tuples: Sequence[TrackerReplace]) -> FlatTrackerReplaceArg:
        result: List[Union[int, str]] = []
        pair: TrackerReplace
        for pair in tuples:
            result.append(pair.trackerId)
            result.append(pair.announceUrl)
        return result

    def process_arguments(args: TorrentMutatorArguments) -> Mapping[str, str]:
        result = _clone_and_convert_keys(args)
        try:
            result['tracker-replace'] = flatten(result["tracker-replace"])
        except KeyError:
            pass
        for (k, v) in result.items():
            result[k] = str(v)
        return result

    request = Request(method=mutator['method'])
    if len(arguments := mutator["arguments"]) > 0:
        request['arguments'] = process_arguments(arguments)

    return request
