from __future__ import annotations

import typing

import toolsql

from ctc import config
from ctc import db
from ctc import spec


async def async_intake_default_erc20s(
    context: spec.Context = None,
    verbose: bool = True,
) -> None:

    network = config.get_context_chain_id(context)
    if network not in ['ethereum', 1]:
        print('No default tokens for network: ' + str(network))
        return

    # load data
    data = load_default_erc20s(context=context)

    # write to db
    db_config = config.get_context_db_config(
        schema_name='erc20_metadata',
        context=context,
    )
    async with toolsql.async_connect(db_config) as conn:
        await db.async_upsert_erc20s_metadata(
            erc20s_metadata=data,
            conn=conn,
            context=context,
        )

    # print summary
    if verbose:
        import toolstr
        from ctc import cli

        styles = cli.get_cli_styles()
        toolstr.print(
            'Added metadata of',
            toolstr.add_style(str(len(data)), styles['description'] + ' bold'),
            'default ERC20 tokens to db',
        )


def load_default_erc20s(
    context: spec.Context = None,
) -> typing.Sequence[spec.ERC20Metadata]:

    network = config.get_context_chain_id(context)

    if network in ('ethereum', 1):
        raw_csv_data = etheruem_erc20s_csv
    else:
        raise NotImplementedError(
            'no erc20 defaults provided for network ' + str(network)
        )

    erc20_metadatas = []
    lines = raw_csv_data.split('\n')
    for line in lines[1:]:
        address, decimals_str, name, symbol = line.split(',')
        decimals = int(decimals_str)
        erc20_metadata: spec.ERC20Metadata = {
            'address': address,
            'decimals': decimals,
            'name': name,
            'symbol': symbol,
        }
        erc20_metadatas.append(erc20_metadata)

    return erc20_metadatas


# from 1inch default token list
etheruem_erc20s_csv = """address,decimals,name,symbol
0x006bea43baa3f7a6f765f14f10a1a1b08334ef45,18,Stox,STX
0x0327112423f3a68efdf1fcf402f6c5cb9f7c33fd,18,PieDAOBTC++,BTC
0x0417912b3a7af768051765040a55bb0925d4ddcf,18,LiquidityDividendsProtocol,LID
0x04fa0d235c4abf4bcf4787af4cf447de572ef828,18,UMAVotingTokenv1,UMA
0x07597255910a51509ca469568b048f2597e72504,18,Uptrennd,1UP
0x08d967bb0134f2d07f7cfb6e246680c53927dd30,18,MATHToken,MATH
0x0a913bead80f321e7ac35285ee10d9d922659cb7,18,DOSNetworkToken,DOS
0x0ae055097c6d159879521c384f1d2123d1f195e6,18,STAKE,STAKE
0x88df592f8eb5d7bd38bfef7deb0fbc02cf3778a0,18,TellorTributes,TRB
0x0bc529c00c6401aef6d220be8c6ea1667f6ad93e,18,yearn.finance,YFI
0x0cf0ee63788a0849fe5297f3407f701e122cc023,18,Streamr,DATA
0x0d438f3b5175bebc262bf23753c1e53d03432bde,18,WrappedNXM,wNXM
0x0d8775f648430679a709e98d2b0cb6250d2887ef,18,BasicAttentionToken,BAT
0x0e8d6b471e332f140e7d9dbb99e5e3822f728da6,18,Abyss,ABYSS
0x0f5d2fb29fb7d3cfee444a200298f468908cc942,18,Mana,MANA
0x0f7f961648ae6db43c75663ac7e5414eb79b5704,18,XIONetwork,XIO
0x12b19d3e2ccc14da04fae33e63652ce469b3f2fd,12,GRID,GRID
0x12f649a9e821f90bb143089a6e56846945892ffb,18,uDOO,uDOO
0x13339fd07934cd674269726edf3b5ccee9dd93de,18,CurToken,CUR
0x1453dbb8a29551ade11d89825ca812e05317eaeb,18,TendiesToken,TEND
0x1776e1f26f98b1a5df9cd347953a26dd3cb46671,18,Numeraire,NMR
0x178c820f862b14f316509ec36b13123da19a6054,18,EnergyWebTokenBridged,EWTB
0x1a5f9352af8af974bfc03399e3767df6370d82e4,18,OWLToken,OWL
0x1beef31946fbbb40b877a72e4ae04a8d1a5cee06,18,Parachute,PAR
0x196f4727526ea7fb1e17b2071b3d8eaa38486988,18,Reserve,RSV
0x1f573d6fb3f13d689ff844b4ce37794d79a7ff1c,18,Bancor,BNT
0x20f7a3ddf244dc9299975b4da1c39f8d5d75f05a,6,SapienNetwork,SPN
0x221657776846890989a759ba2973e427dff5c9bb,18,ReputationV2,REPv2
0x2260fac5e5542a773aa44fbcfedf7c193bc2c599,8,WrappedBTC,WBTC
0x255aa6df07540cb5d3d297f0d0d4d84cb52bc8e6,18,RaidenNetworkToken,RDN
0x26b3038a7fc10b36c426846a9086ef87328da702,18,YieldFarmingToken,YFT
0x27054b13b1b798b345b591a4d22e6562d47ea75a,4,AirSwap,AST
0x28cb7e841ee97947a86b06fa4090c8451f64c0be,18,YFLink,YFL
0x28dee01d53fed0edf5f6e310bf8ef9311513ae40,18,BlitzPredict,XBP
0x2b591e99afe9f32eaa6214f7b7629768c40eeb39,8,HEX,HEX
0x2ba592f78db6436527729929aaf6c908497cb200,18,Cream,CREAM
0x2c4e8f2d746113d0696ce89b35f0d8bf88e0aeca,18,OpenSimpleToken,OST
0x2c537e5624e4af88a7ae4060c022609376c8d0eb,6,BiLira,TRYB
0x2c974b2d0ba1716e644c1fc59982a89ddd2ff724,18,Viberate,VIB
0x301c755ba0fca00b1923768fffb3df7f4e63af31,18,GlobalDigitalContent,GDC
0x309627af60f0926daa6041b8279484312f2bf060,18,USDB,USDB
0x30f271c9e86d2b7d00a6376cd96a1cfbd5f0b9b3,18,Decentr,DEC
0x3166c570935a7d8554c8f4ea792ff965d2efe1f2,18,QDAO,QDAO
0x340d2bde5eb28c1eed91b2f790723e3b160613b7,18,BLOCKv,VEE
0x37e8789bb9996cac9156cd5f5fd32599e6b91289,18,AidCoin,AID
0x3a92bd396aef82af98ebc0aa9030d25a23b11c6b,18,Tokenbox,TBX
0x3c6ff50c9ec362efa359317009428d52115fe643,18,PeerExNetwork,PERX
0x3d1ba9be9f66b8ee101911bc36d3fb562eac2244,18,Rivetz,RVT
0x408e41876cccdc0f92210600ef50372656052a38,18,Republic,REN
0x40fd72257597aa14c7231a7b1aaa29fce868f677,18,SoraToken,XOR
0x419d0d8bdd9af5e606ae2232ed285aff190e711b,8,FunFair,FUN
0x41ab1b6fcbb2fa9dced81acbdec13ea6315f2bf2,18,XinFin,XDCE
0x41e5560054824ea6b0732e656e3ad64e20e94e45,8,Civic,CVC
0x42d6622dece394b54999fbd73d108123806f6a18,18,SPANK,SPANK
0x43044f861ec040db59a7e324c40507addb673142,18,Cap,CAP
0xfef4185594457050cc9c23980d301908fe057bb1,18,VIDTDatalink,VIDT
0x4470bb87d77b963a013db939be332f927f2b992e,4,AdExNetworkV1,ADXv1
0x456ae45c0ce901e2e7c99c0718031cec0a7a59ff,18,VisionNetwork,VSN
0x49184e6dae8c8ecd89d8bdc1b950c597b8167c90,2,LIBERTAS,LIBERTAS
0x4946fcea7c692606e8908002e55a582af44ac121,18,FOAMToken,FOAM
0x4954db6391f4feb5468b6b943d4935353596aec9,18,USDQ,USDQ
0x4a220e6096b25eadb88358cb44068a3248254675,18,Quant,QNT
0x4a57e687b9126435a9b19e4a802113e266adebde,18,Flexacoin,FXC
0x4aac461c86abfa71e9d00d9a2cde8d74e4e1aeea,18,ZINC,ZINC
0x4c327471c44b2dacd6e90525f9d629bd2e4f662c,18,GHOST,GHOST
0x4cc19356f2d37338b9802aa8e8fc58b0373296e7,18,SelfKey,KEY
0x4d953cf077c0c95ba090226e59a18fcf97db44ec,18,MINISWAP,MINI
0x4da9b813057d04baef4e5800e36083717b4a0341,18,AaveInterestbearingTUSD,aTUSDv1
0x4e352cf164e64adcbad318c3a1e222e9eba4ce42,18,MCDEXToken,MCB
0x4f3afec4e5a3f2a6a1a411def7d7dfe50ee057bf,9,DigixGoldToken,DGX
0x4fabb145d64652a948d72533023f6e7a623c7c53,18,BinanceUSD,BUSD
0x5102791ca02fc3595398400bfe0e33d7b6c82267,18,LeadCoin,LDC
0x514910771af9ca656af840dff83e8264ecf986ca,18,ChainLink,LINK
0x543ff227f64aa17ea132bf9886cab5db55dcaddf,18,DAOStack,GEN
0x56d811088235f11c8920698a204a5010a788f4b3,18,bZxProtocolToken,BZRX
0x5732046a883704404f284ce41ffadd5b007fd668,18,Bluzelle,BLZ
0x57700244b20f84799a31c6c96dadff373ca9d6c5,18,TRUSTDAO,TRUST
0x58b6a8a3302369daec383334672404ee733ab239,18,LivepeerToken,LPT
0x595832f8fc6bf59c85c527fec3740a1b7a361269,6,PowerLedger,POWR
0x5adc961d6ac3f7062d2ea45fefb8d8167d44b190,18,Dether,DTH
0x5c872500c00565505f3624ab435c222e558e9ff8,18,CoTrader,COT
0x5caf454ba92e6f2c929df14667ee360ed9fd5b26,18,Dev,DEV
0x5d3a536e4d6dbd6114cc1ead35777bab948e3643,8,CompoundDai,cDAI
0x5d60d8d7ef6d37e16ebabc324de3be57f135e0bc,18,MyBit,MYB
0x607c794cda77efb21f8848b7910ecf27451ae842,18,DeFiPIEToken,PIE
0x607f4c5bb672230e8672085532f7e901544a7375,9,iExecRLC,RLC
0x6226caa1857afbc6dfb6ca66071eb241228031a1,18,Linkart,LAR
0x6251e725cd45fb1af99354035a414a2c0890b929,18,MixTrust,MXT
0x625ae63000f46200499120b906716420bd059240,18,AaveInterestbearingSUSD,aSUSDv1
0x667088b212ce3d06a1b553a7221e1fd19000d9af,18,Wings,WINGS
0x6710c63432a2de02954fc0f851db07146a6c0312,18,SmartMFG,MFG
0x6758b7d441a9739b98552b373703d8d3d14f9e62,18,POA,POA20
0x6810e776880c02933d47db1b9fc05908e5386b96,18,Gnosis,GNO
0x68d57c9a1c35f63e2c83ee8e49a64e9d70528d25,18,SirinLabs,SRN
0x6b785a0322126826d8226d77e173d75dafb84d11,18,BankrollVault,VLT
0x6b9f031d718dded0d681c20cb754f97b3bb81b78,18,Geeq,GEEQ
0x6ba460ab75cd2c56343b3517ffeba60748654d26,8,UpToken,UP
0x6c6ee5e31d828de241282b9606c8e98ea48526e2,18,HoloToken,HOT
0x6f87d756daf0503d08eb8993686c7fc01dc44fb1,18,UniTrade,TRADE
0x6fe56c0bcdd471359019fcbc48863d6c3e9d4f41,18,PropsToken,PROPS
0x71fc860f7d3a592a4a98740e39db31d25db65ae8,6,AaveInterestbearingUSDT,aUSDTv1
0x737f98ac8ca59f2c68ad658e3c3d8c8963e40a4c,18,Amon,AMN
0x744d70fdbe2ba4cf95131626614a1763df805b9e,18,Status,SNT
0x763186eb8d4856d536ed4478302971214febc6a9,18,BetterBetting,BETR
0x780116d91e5592e58a3b3c76a351571b39abcec6,15,Blockparty,BOXX
0x7b0c06043468469967dba22d1af33d77d44056c8,4,MorpheusNetwork,MRPH
0x7b123f53421b1bf8533339bfbdc7c98aa94163db,18,dfohub,buidl1
0x7c5a0ce9267ed19b22f8cae653f198e3e8daf098,18,Santiment,SAN
0x7de91b204c1c737bcee6f000aaa6569cf7061cb7,9,Robonomics,XRT
0x80fb784b7ed66730e8b1dbd9820afd29931aab03,18,EthLend,LEND
0x814e0908b12a99fecf5bc101bb5d0b8b5cdf7d26,18,MeasurableDataToken,MDT1
0x8207c1ffc5b6804f6024322ccf34f29c3541ae26,18,OriginProtocol,OGN
0x821144518dfe9e7b44fcf4d0824e15e8390d4637,18,ATISToken,ATIS
0x83984d6142934bb535793a82adb0a46ef0f66b6d,4,Remme,REM
0x83cee9e086a77e492ee0bb93c2b0437ad6fdeccc,18,Goldmint,MNTP
0x8400d94a5cb0fa0d041a3788e395285d61c9ee5e,8,UniBright,UBT
0x84ca8bc7997272c7cfb4d0cd3d55cd942b3c9419,18,DIAToken,DIA
0x8762db106b2c2a0bccb3a80d1ed41273552616e8,18,ReserveRights,RSR
0x89ab32156e46f46d02ade3fecbe5fc4243b9aaed,18,pNetworkToken,PNT
0x8ab7404063ec4dbcfd4598215992dc3f8ec853d7,18,Akropolis,AKRO
0x7b6f71c8b123b38aa8099e0098bec7fbc35b8a13,8,NerveNetwork,NVT
0x8ce9137d39326ad0cd6491fb5cc0cba0e089b6a9,18,Swipe,SXP
0x8e870d67f660d95d5be530380d0ec0bd388289e1,18,PaxDollar,USDP
0x8eb24319393716668d768dcec29356ae9cffe285,8,SingularityNETToken,AGI
0x8f8221afbb33998d8584a2b05749ba73c37a938a,18,Request,REQ
0x0258f474786ddfd37abce6df6bbb1dd5dfc4434a,8,OrionProtocol,ORN
0x9214ec02cb71cba0ada6896b8da260736a67ab10,18,REAL,REAL
0x93ed3fbe21207ec2e8f2d3c3de6e058cb73bc04d,18,Kleros,PNK
0x95172ccbe8344fecd73d0a30f54123652981bd6f,18,MeridianNetwork,LOCK
0x960b236a07cf122663c4303350609a66a7b288c0,18,AragonNetworkToken,ANTv1
0x967da4048cd07ab37855c090aaf366e4ce1b9f48,18,OceanToken,OCEAN
0x990f341946a3fdb507ae7e52d17851b87168017c,18,Strong,STRONG
0x9992ec3cf6a55b00978cddf2b27bc6882d88d1ec,18,Polymath,POLY
0x9ba00d6856a4edf4665bca2c2309936572473b7e,6,AaveInterestbearingUSDC,aUSDCv1
0x9cb2f26a23b8d89973f08c957c4d7cdf75cd341c,6,DigitalRand,DZAR
0x9f49ed43c90a540d1cf12f6170ace8d0b88a14e6,18,ETHRSI6040YieldII,ETHRSIAPY
0x9f8f72aa9304c8b593d555f12ef6589cc3a579a2,18,Maker,MKR
0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48,6,USDCoin,USDC
0xa15c7ebe1f07caf6bff097d8a589fb8ac49ae5b3,18,PundiX,NPXS
0xa1d0e215a23d7030842fc67ce582a6afa3ccab83,18,YFII.finance,YFII
0xa1d65e8fb6e87b60feccbc582f7f97804b725521,18,DXdao,DXD
0xa3bed4e1c75d00fa6f4e5e6922db7261b5e9acd2,18,Meta,MTA
0xa3d58c4e56fedcae3a7c43a725aee9a71f0ece4e,18,Metronome,MET
0xa462d0e6bb788c7807b1b1c96992ce1f7069e195,18,EQUUSMiningToken,EQMT
0xa4bdb11dc0a2bec88d24a3aa1e6bb17201112ebe,6,StableUSD,USDS
0xa4e8c3ec456107ea67d3075bf9e3df3a75823db0,18,LoomNetwork,LOOM
0xa64bd6c70cb9051f6a9ba1f163fdc07e0dfb5f84,18,AaveInterestbearingLINK,aLINKv1
0xa704fce7b309ec09df16e2f5ab8caf6fe8a4baa9,18,AgriChain,AGRI
0xaaaebe6fe48e54f431b0c390cfaf0b017d09d42d,4,Celsius,CEL
0xaaaf91d9b90df800df4f55c205fd6989c977e73a,8,Monolith,TKN
0xb0280743b44bf7db4b6be482b2ba7b75e5da096c,18,Transcodium,TNS
0xb056c38f6b7dc4064367403e26424cd2c60655e1,18,CEEKVR,CEEK
0xb3319f5d18bc0d84dd1b4825dcde5d5f7266d407,8,Compound0x,cZRX
0xb4272071ecadd69d933adcd19ca99fe80664fc08,18,CryptoFranc,XCHF
0xb4efd85c19999d84251304bda99e90b92300bd93,18,RocketPool,RPL
0xeca82185adce47f39c684352b0439f030f860318,18,Perlin,PERL
0xb4058411967d5046f3510943103805be61f0600e,18,STONK,STONK
0xb683d83a532e2cb7dfa5275eed3698436371cc9f,18,BTUProtocol,BTU
0xb6c4267c4877bb0d6b1685cfd85b0fbe82f105ec,18,Relevant,REL
0xba100000625a3754423978a60c9317c58a424e3d,18,Balancer,BAL
0xba11d00c5f74255f56a5e366f4f77f5a186d7f55,18,BandProtocol,BAND
0xbb1fa4fdeb3459733bf67ebc6f893003fa976a82,18,Bitnation,XPAT
0xbbbbca6a901c926f240b89eacb641d8aec7aeafd,18,Loopring,LRC
0xbc86727e770de68b1060c91f6bb6945c73e10388,18,InkProtocol,XNK
0xbd2949f67dcdc549c6ebe98696449fa79d988a9f,18,MeterGovernancemappedbyMeter.io,eMTRG
0xd9ec3ff1f8be459bb9369b4e79e9ebcf7141c093,18,KardiaChainToken,KAI
0xbe9375c6a420d2eeb258962efb95551a5b722803,18,KyberStormXToken,STMX
0xbf2179859fc6d5bee9bf9158632dc51678a4100e,18,Aelf,ELF
0xc00e94cb662c3520282e6f5717214004a7f26888,18,Compound,COMP
0xc011a73ee8576fb46f5e1c5751ca3b9fe0af2a6f,18,SynthetixNetworkToken,SNX
0x27702a26126e0b3702af63ee09ac4d1a084ef628,18,aleph.imv2,ALEPH
0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2,18,WrappedEther,WETH
0xc11b1268c1a384e55c48c2391d8d480264a3a7f4,8,CompoundWrappedBTC,cWBTC
0xc12d099be31567add4e4e4d0d45691c3f58f5663,18,Auctus,AUC
0xc27a2f05fa577a83ba0fdb4c38443c0718356501,18,LamdenTau,TAU
0xc28e931814725bbeb9e670676fabbcb694fe7df2,18,QuadrantProtocol,EQUAD
0xc3dd23a0a854b4f9ae80670f528094e9eb607ccb,18,Trendering,TRND
0xc5bbae50781be1669306b9e001eff57a2957b09d,5,Gifto,GTO
0xc75f15ada581219c95485c578e124df3985e4ce0,18,zzz.finance,ZZZ
0xcc4304a31d09258b0029ea7fe63d032f52e44efe,18,TrustSwapToken,SWAP
0xcc80c051057b774cd75067dc48f8987c4eb97a5e,18,EthfinexNectarToken,NEC
0xcd62b1c403fa761baadfc74c525ce2b51780b184,18,AragonNetworkJuror,ANJ
0xcee1d3c3a02267e37e6b373060f79d5d7b9e1669,18,yffi.finance,YFFI
0xcf8f9555d55ce45a3a33a81d6ef99a2a2e71dee2,18,CBIIndex7,CBIX7
0xd15ecdcf5ea68e3995b2d0527a0ae0a3258302f8,18,MachiXToken,MCX
0xd26114cd6ee289accf82350c8d8487fedb8a0c07,18,OmiseGO,OMG
0xd341d1680eeee3255b8c4c75bcce7eb57f144dae,18,onG.social,ONG
0xd46ba6d942050d489dbd938a2c909a5d5039a161,9,Ampleforth,AMPL
0xd4c435f5b09f855c3317c8524cb1f586e42795fa,18,Cindicator,CND
0xd559f20296ff4895da39b5bd9add54b442596a61,18,FintruX,FTX
0xd6f0bb2a45110f819e908a915237d652ac7c5aa8,18,DFOHub,BUIDL2
0xd7631787b4dcc87b1254cfd1e5ce48e96823dee8,8,Sociall,SCL
0xd8912c10681d8b21fd3742244f44658dba12264e,18,Pluton,PLU
0xdac17f958d2ee523a2206206994597c13d831ec7,6,TetherUSD,USDT
0xdb25f211ab05b1c97d595516f45794528a807ad8,2,STASISEURS,EURS
0xdd974d5c2e2928dea5f71b9825b8b646686bd200,18,KyberNetworkLegacy,KNCL
0xdf2c7238198ad8b389666574f2d8bc411a4b7428,18,Mainframe,MFT
0xdf574c24545e5ffecb9a659c229253d4111d87e1,8,HUSD,HUSD
0xdfe691f37b6264a90ff507eb359c45d55037951c,4,Karma,KARMA
0xe17f017475a709de58e976081eb916081ff4c9d5,9,RMPL,RMPL
0xe25b0bba01dc5630312b6a21927e578061a13f55,18,ShipChainSHIP,SHIP
0xe3818504c1b32bf1557b16c238b2e01fd3149c17,18,Pillar,PLR
0xe41d2489571d322189246dafa5ebde1f4699f498,18,0xProtocol,ZRX
0xe48972fcd82a274411c01834e2f031d4377fa2c0,18,2key.network,2KEY
0xeb4c2781e4eba804ce9a9803c67d0893436bb27d,8,renBTC,renBTC
0xec67005c4e498ec7f55e092bd1d35cbc47c91892,18,MelonToken,MLN
0xedd7c94fd7b4971b916d15067bc454b9e1bad980,18,Zippie,ZIPT
0xeeee2a622330e6d2036691e983dee87330588603,18,AskobarNetwork,ASKO
0xeeeeeeeee2af8d0e1940679860398308e0ef24d6,18,EthverseToken,ETHVEthverse
0xef9cd7882c067686691b6ff49e650b43afbbcc6b,18,FinNexus,FNX
0xefbd6d7def37ffae990503ecdb1291b2f7e38788,18,Evolution,EVO
0xf04a8ac553fcedb5ba99a64799155826c136b0be,18,Flixxo,FLIXX
0xf0fac7104aac544e4a7ce1a55adf2b5a25c65bd1,18,PampNetwork,PAMP
0xf1290473e210b2108a85237fbcd7b6eb42cc654f,18,HedgeTrade,HEDG
0xf29992d7b589a0a6bd2de7be29a97a6eb73eaf85,18,DMScript,DMST
0xf29e46887ffae92f1ff87dfe39713875da541373,18,UniCrypt,UNC
0xf2f9a7e93f845b3ce154efbeb64fb9346fcce509,18,UniPower,POWER
0xf433089366899d83a9f26a773d59ec7ecf30355e,8,Metal,MTL
0xf629cbd94d3791c9250152bd8dfbdf380e2a3b9c,18,EnjinCoin,ENJ
0xf650c3d88d12db855b8bf7d11be6c55a4e07dcc9,8,CompoundUSDT,cUSDT
0xf8e386eda857484f5a12e4b5daa9984e06e73705,18,Indorse,IND
0xf911a7ec46a2c6fa49193212fe4a2a9b95851c27,9,Antiample,XAMP
0xf970b8e36e23f7fc3fd752eea86f8be8d83375a6,18,Ripio,RCN
0xfc1e690f61efd961294b3e1ce3313fbd8aa4f85d,18,AaveInterestbearingDAI,aDAIv1
0x00006100f7090010005f1bd7ae6122c3c2cf0090,18,TrueAUD,TAUD
0x00000000441378008ea67f4284a57932b1c000a5,18,TrueGBP,TGBP
0x00000100f2a2bd000715001920eb70d229700085,18,TrueCAD,TCAD
0x0000852600ceb001e08e00bc008be620d60031f2,18,TrueHKD,THKD
0x0000000000b3f879cb30fe243b4dfee438691c04,2,Gastoken.io2,GST2
0x06af07097c9eeb7fd685c692751d5c66db49c215,18,ChaiToken,CHAI
0x6c8c6b02e7b2be14d4fa6022dfd6d75921d90e4e,8,CompoundBasicAttentionToken,cBAT
0xf5dce57282a584d2746faf1593d3121fcac444dc,8,CompoundSai,cSAI
0x4ddc2d193948926d02f9b1fe9e1daa0718270ed5,8,CompoundETH,cETH
0x39aa39c021dfbae8fac545936693ac917d5e7563,8,CompoundUSDCoin,cUSDC
0x158079ee67fce2f58472a96584a73c7ab9ac95c1,8,CompoundAugur,cREP
0x2af5d2ad76741191d15dfe7bf6ac92d4bd912ca3,18,BitfinexLEOToken,LEO
0x6f259637dcd74c767781e37bc6133cd6a68aa161,18,HuobiToken,HT
0x7d1afa7b718fb893db30a3abc0cfc608aacfebb0,18,MaticToken,MATIC
0x6b175474e89094c44da98b954eedeac495271d0f,18,DaiStablecoin,DAI
0x89d24a6b4ccb1b6faa2625fe562bdd9a23260359,18,SaiStablecoin,SAI
0x57ab1ec28d129707052df4df418d58a2d46d5f51,18,SynthsUSD,sUSD
0xeb269732ab75a6fd61ea60b06fe994cd32a83549,18,dForce,USDx
0x0000000000085d4780b73119b644ae5ecd22b376,18,TrueUSD,TUSD
0x679131f591b4f369acb8cd8c51e68596806c3916,18,TrustlinesNetworkToken,TLN
0x3a3a65aab0dd2a17e3f1947ba16138cd37d08c04,18,AaveInterestbearingETH,aETHv1
0xe1ba0fb44ccb0d11b80f92f4f8ed94ca3ff51d00,18,AaveInterestbearingBAT,aBATv1
0x9d91be44c06d373a8a226e1f3b146956083803eb,18,AaveInterestbearingKNC,aKNCv1
0x7d2d3688df45ce7c552e19c27e007673da9204b8,18,AaveInterestbearingLEND,aLENDv1
0x6fce4a401b6b80ace52baaefe4421bd188e76f6f,18,AaveInterestbearingMANA,aMANAv1
0x7deb5e830be29f91e298ba5ff1356bb7f8146998,18,AaveInterestbearingMKR,aMKRv1
0x71010a9d003445ac60c4e6a7017c1e89a477b438,18,AaveInterestbearingREP,aREPv1
0x328c4c80bc7aca0834db37e6600a6c49e12da4de,18,AaveInterestbearingSNX,aSNXv1
0xfc4b8ed459e00e5400be803a9bb3954234fd50e3,8,AaveInterestbearingWBTC,aWBTCv1
0x6fb0855c404e09c47c3fbca25f08d4e41f9f062f,18,AaveInterestbearingZRX,aZRXv1
0x66fd97a78d8854fec445cd1c80a07896b0b4851f,18,LunchMoney,LMY
0x16de59092dae5ccf4a1e6439d611fd0653f0bd01,18,iearnDAIv2,yDAIv2
0xc2cb1040220768554cf699b0d863a3cd4324ce32,18,iearnDAIv3,yDAIv3
0x04bc0ab673d88ae9dbc9da2380cb6b79c4bca9ae,18,iearnBUSD,yBUSD
0x04aa51bbcb46541455ccf1b8bef2ebc5d3787ec9,8,iearnWBTC,yBTC
0x73a052500105205d34daf004eab301916da8190f,18,iearnTUSD,yTUSDiearn
0x83f798e925bcd4017eb265844fddabb448f1707d,6,iearnUSDTv2,yUSDTv2
0xe6354ed5bc4b393a5aad09f21c46e101e692d447,6,iearnUSDTv3,yUSDTv3
0xd6ad7a6750a7593e092a9b218d66c0a814a3436e,6,iearnUSDCv2,yUSDCv2
0x26ea744e5b887e5205727f55dfbe8685e3b21951,6,iearnUSDCv3,yUSDCv3
0xf61718057901f84c4eec4339ef8f0d86d2b45600,18,iearnSUSD,ySUSD
0xacfa209fb73bf3dd5bbfb1101b9bc999c49062a5,18,BlockchainCertifiedDataToken,BCDT
0x4de2573e27e648607b50e1cfff921a33e4a34405,18,LendroidSupportToken,LST
0x630d98424efe0ea27fb1b3ab7741907dffeaad78,8,PEAKDEFI,PEAK
0xd56dac73a4d6766464b38ec6d91eb45ce7457c44,18,Panvalapan,PAN
0x056fd409e1d7a124bd7017459dfea2f387b6d5cd,2,Geminidollar,GUSD
0x6ee0f7bb50a54ab5253da0667b0dc2ee526c30a8,18,AaveInterestbearingBinanceUSD,aBUSDv1
0xd7efb00d12c2c13131fd319336fdf952525da2af,4,Proton,XPR
0x85eee30c52b0b379b046fb0f85f4f3dc3009afec,18,KEEPToken,KEEP
0x1c5db575e2ff833e46a2e9864c22f4b22e0b37c2,8,renZEC,renZEC
0x459086f2376525bdceba5bdda135e4e9d3fef5bf,8,renBCH,renBCH
0x8daebade922df735c38c80c7ebd708af50815faa,18,tBTC,tBTC
0x0316eb71485b0ab14103307bf65a021042c6d380,18,HuobiBTC,HBTC
0x3a9fff453d50d4ac52a6890647b823379ba36b9e,18,Shuffle.MonsterV3,SHUF
0xc0f9bd5fa5698b6505f643900ffa515ea5df54a9,18,DONUT,DONUT
0x0000000000004946c0e9f43f4dee607b0ef1fa1c,0,ChiGastokenby1inch,CHI
0xd6a55c63865affd67e2fb9f284f87b7a9e5ff3bd,18,Switch,ESH
0x1fc31488f28ac846588ffa201cde0669168471bd,2,UAX,UAX
0x45804880de22913dafe09f4980848ece6ecbaf78,18,PaxosGold,PAXG
0x035df12e0f3ac6671126525f1015e47d79dfeddf,18,0xMonero,0xMR
0xb64ef51c888972c908cfacf59b47c1afbc0ab8ac,8,Storj,STORJ
0x4156d3342d5c385a87d264f90653733592000581,8,Salt,SALT
0xdf5e0e81dff6faf3a7e52ba697820c5e32d806a8,18,Curve.fiiearnpooltoken,yCurve
0xfca59cd816ab1ead66534d82bc21e7515ce441cf,18,Rarible,RARI
0xea5f88e54d982cbb0c441cde4e79bc305e5b43bc,18,ParetoNetworkToken,PARETO
0x9fbfed658919a896b5dc7b00456ce22d780f9b65,18,PlutusDeFi,PLT
0x5228a22e72ccc52d415ecfd199f99d0665e7733b,18,pTokensBTC,pBTC
0x476c5e26a75bd202a9683ffd34359c0cc15be0ff,6,Serum,SRM
0xc813ea5e3b48bebeedb796ab42a30c5599b01740,4,Autonio,NIOX
0xa7de087329bfcda5639247f96140f9dabe3deed1,18,Statera,STA
0xdc5864ede28bd4405aa04d93e05a0531797d9d59,6,Falcon,FNT
0x0aacfbec6a24756c20d41914f2caba817c0d8521,18,YAM,YAM
0xade00c28244d5ce17d72e40330b1c318cd12b7c3,18,AdExNetwork,ADX
0xd533a949740bb3306d119cc777fa900ba034cd52,18,CurveDAOToken,CRV
0x9469d013805bffb7d3debe5e7839237e535ec483,18,DarwiniaNetworkNativeToken,RING
0x2baecdf43734f22fd5c152db08e3c27233f0c7d2,18,OMToken,OMv1
0x491604c0fdf08347dd1fa4ee062a822a5dd06b5d,18,CartesiToken,CTSI
0x0ff6ffcfda92c53f615a4a75d982f399c989366b,18,Unilayer,LAYER
0xd5525d397898e5502075ea5e830d8914f6f0affe,8,MEME,MEME
0x68a118ef45063051eac49c7e647ce5ace48a68a5,18,BASED,BASED
0xaa7a9ca87d3694b5755f213b5d04094b8d0f0a6f,18,Trace,TRAC
0x4dfd148b532e934a2a26ea65689cf6268753e130,18,GovernancetokenMonolithosDAO,MDT2
0xaba8cac6866b83ae4eec97dd07ed254282f6ad8a,24,YAMv2,YAMv2
0x8a9c67fee641579deba04928c4bc45f66e26343a,18,JarvisRewardToken,JRT
0x45f24baeef268bb6d63aee5129015d69702bcdfa,18,YFValue,YFV
0x674c6ad92fd080e4004b2312b45f796a192d27a0,18,Neutrino,USDN
0x362bc847a3a9637d3af6624eec853618a43ed7d2,18,ParsiqToken,PRQ
0x0e29e5abbb5fd88e28b2d355774e73bd47de3bcd,18,HakkaFinance,HAKKA
0x08ad83d779bdf2bbe1ad9cc0f78aa0d24ab97802,18,RobonomicsWebServicesV1,RWS
0x09e64c2b61a5f1690ee6fbed9baf5d6990f8dfd0,18,Growth,GRO
0x38e4adb44ef08f22f5b5b76a8f0c2d0dcbe7dca1,18,ConcentratedVotingPower,CVP
0x4fe5851c9af07df9e5ad8217afae1ea72737ebda,18,OpenPredictToken,OPT
0x6b3595068778dd592e39a122f4f5a5cf09c90fe2,18,SushiToken,SUSHI
0x3affcca64c2a6f4e3b6bd9c64cd2c969efd1ecbe,18,DSLA,DSLA
0xba21ef4c9f433ede00badefcc2754b8e74bd538a,18,Swapfolio,SWFL
0xfffffffff15abf397da76f1dcc1a1604f45126db,18,FalconSwapToken,FSW
0x94d863173ee77439e4292284ff13fad54b3ba182,18,AkropolisDelphi,ADEL
0xb8baa0e4287890a5f79863ab62b7f175cecbd433,18,SwerveDAOToken,SWRV
0x8a6f3bf52a26a21531514e23016eeae8ba7e7018,8,Multiplier,MXX
0x103c3a209da59d3e7c4a89307e66521e081cfdf0,18,GenesisVision,GVT
0x892b14321a4fcba80669ae30bd0cd99a7ecf6ac0,8,CreamCream,crCream
0xe1237aa7f535b0cc33fd973d66cbf830354d16c7,18,yearnWrappedEther,yWETH
0x5dbcf33d8c2e976c6b560249878e6f1491bca25c,18,yearnCurve.fiyDAIyUSDCyUSDTyTUSD,yUSD
0x50026ad58b338cf3eccc2b422deb8faa725f377f,8,1Step.finance,STEP
0x1aa61c196e76805fcbe394ea00e4ffced24fc469,18,yieldfarming.insure,SAFE
0x556148562d5ddeb72545d7ec4b3ec8edc8f55ba7,18,PredixNetwork,PRDX
0x1494ca1f11d487c2bbe4543e90080aeba4ba3c2b,18,DefiPulseIndex,DPI
0x3f382dbd960e3a9bbceae22651e88158d2791550,18,AavegotchiGHSTToken,GHST
0x1f9840a85d5af5bf1d1762f925bdaddc4201f984,18,Uniswap,UNI
0x2e93fe8d550a7b7e7b2e561cd45cebccbaa79358,5,GXChainCoreAsset,GXC
0x429881672b9ae42b8eba0e26cd9c73711b891ca5,18,PickleToken,PICKLE
0xf8c3527cc04340b208c854e985240c02f7b7793f,18,FrontierToken,FRONT
0xca1207647ff814039530d7d35df0e1dd2e91fa84,18,dHedgeDAOToken,DHT
0xa0246c9032bc3a600820415ae600c6388619a14d,18,FARMRewardToken,FARM
0x488e0369f9bc5c40c002ea7c1fe4fd01a198801c,18,Golff.finance,GOF
0xecbf566944250dde88322581024e611419715f7a,9,xBTC,xBTC
0x2a8e1e676ec238d8a992307b495b45b3feaa5e86,18,OriginDollar,OUSD
0x1d09144f3479bb805cb7c92346987420bcbdc10c,18,CreamYUSD,cyUSD
0x7fc66500c84a76ad7e9c93437bfc5ac33e2ddae9,18,AaveToken,AAVE
0x43dfc4159d86f3a37a5a4b3d4580b888ad7d4ddd,18,DODObird,DODO
0x250a3500f48666561386832f1f1f1019b89a2699,18,SAFE2,SAFE2
0x6006fc2a849fedaba8330ce36f5133de01f96189,18,SHAKEtokenbySpaceSwapv2,SHAKE
0x80c8c3dcfb854f9542567c8dac3f44d709ebc1de,18,MilkyWayTokenbySpaceSwapv2,MILK2
0x62359ed7505efc61ff1d56fef82158ccaffa23d7,18,cVault.finance,CORE
0xbc396689893d065f41bc2c6ecbee5e0085233447,18,Perpetual,PERP
0x49e833337ece7afe375e44f4e3e8481029218e5c,18,ValueLiquidity,VALUE
0x4b7ad3a56810032782afce12d7d27122bdb96eff,8,Sparkle,SPRKL
0x1c48f86ae57291f7686349f12601910bd8d470bb,18,USDK,USDK
0x87edffde3e14c7a66c9b9724747a1c5696b742e6,18,SwagToken,SWAG
0xad6a626ae2b43dcb1b39430ce496d2fa0365ba9c,18,PieDAODEFISmallCap,DEFIS
0xad32a8e6220741182940c5abf610bde99e737b2d,18,PieDAODOUGHv2,DOUGH
0x3f09400313e83d53366147e3ea0e4e2279d80850,18,KUSH.FINANCE,kSEED
0x70e36f6bf80a52b3b46b3af8e106cc0ed743e8e4,8,CompoundCollateral,cCOMP
0x35a18000230da775cac24873d00ff85bccded550,8,CompoundUniswap,cUNI
0x70da48f4b7e83c386ef983d4cef4e58c2c09d8ac,8,QurasToken,XQC
0xe2f2a5c287993345a840db3b0845fbc70f5935a5,18,mStableUSD,mUSD
0x2edf094db69d6dcd487f1b3db9febe2eec0dd4c5,18,ZeroSwapToken,ZEE
0x584bc13c7d411c00c01a62e8019472de68768430,18,Hegic,HEGIC
0x054f76beed60ab6dbeb23502178c52d6c5debe40,18,DeFiner,FIN
0xcbd55d4ffc43467142761a764763652b48b969ff,18,AstroTools.io,ASTRO
0xff20817765cb7f73d4bde2e66e067e58d11095c2,18,Amp,AMP
0x0391d2021f89dc339f60fff84546ea23e337750f,18,BarnBridgeGovernanceToken,BOND
0xa117000000f279d81a1d3cc75430faa017fa5a2e,18,AragonNetworkToken,ANT
0x970b9bb2c0444f5e81e9d0efb84c8ccdcdcaf84d,18,FuseToken,FUSE
0x36f3fd68e7325a35eb768f1aedaae9ea0689d723,18,EmptySetDollar,ESD
0xa9af25a2d43eb04723a6ec0749913519cf347842,6,TealCoin,TEAC
0x1ceb5cb57c4d4e2b2433641b95dd330a33185a44,18,Keep3rV1,KP3R
0x5bc25f649fc4e26069ddf4cf4010f9f706c23831,18,DefiDollar,DUSD
0xb705268213d593b8fd88d3fdeff93aff5cbdcfae,18,IDEXToken,IDEX
0x2e2364966267b5d7d2ce6cd9a9b5bd19d9c7c6a9,18,VoiceToken,VOICE
0xeef9f339514298c6a857efcfc1a762af84438dee,18,HermezNetworkToken,HEZ
0xea319e87cf06203dae107dd8e5672175e3ee976c,18,SURF.Finance,SURF
0x3383c5a8969dc413bfddc9656eb80a1408e4ba20,18,WrappedANATHA,wANATHA
0xa665fed1b0c9da00e91ca582f77df36e325048c5,18,yfarm.finance,YFM
0xdacd69347de42babfaecd09dc88958378780fb62,0,AtariToken,ATRI
0x0954906da0bf32d5479e25f46056d22f08464cab,18,Index,INDEX
0xc57d533c50bc22247d49a368880fb49a1caa39f7,18,PowerTradeFuelToken,PTF
0x20c36f062a31865bed8a5b1e512d9a1a20aa333a,18,DefiDollarDAO,DFD
0x95a4492f028aa1fd432ea71146b433e7b4446611,18,APYGovernanceToken,APY
0xbea98c05eeae2f3bc8c3565db7551eb738c8ccab,18,Geyser,GYSR
0xa89ac6e529acf391cfbbd377f3ac9d93eae9664e,18,Keep4r,KP4R
0xbb0e17ef65f82ab018d8edd776e8dd940327b28b,18,AxieInfinityShard,AXS
0x297e4e5e59ad72b1b0a2fd446929e76117be0e0a,18,ValorToken,VALOR
0x00a8b738e453ffd858a7edf03bccfe20412f0eb0,18,AllianceBlockToken,ALBT
0x5beabaebb3146685dd74176f68a0721f91297d37,18,BounceToken,BOT
0x05d3606d5c81eb9b7b18530995ec9b29da05faba,18,TomoChain,TOMOE
0xb1f66997a5760428d3a87d68b90bfe0ae64121cc,18,LuaToken,LUA
0x39eae99e685906ff1c11a962a743440d0a1a6e09,18,Holyheld,HOLY
0x9d942bd31169ed25a1ca78c776dab92de104e50e,18,DeFISocks2020,DEFISOCKS
0xf4cd3d3fda8d7fd6c5a500203e38640a70bf9577,18,YfDAI.finance,YfDAI
0x83e6f1e41cdd28eaceb20cb649155049fac3d5aa,18,PolkastarterToken,POLS
0xaf9f549774ecedbd0966c52f250acc548d3f36e5,18,RioFuelToken,RFuel
0x0202be363b8a4820f3f4de7faf5224ff05943ab1,18,UniLendFinanceToken,UFT
0xae697f994fc5ebc000f8e22ebffee04612f98a0d,18,LGCYNetwork,LGCY
0x9d47894f8becb68b9cf3428d256311affe8b068b,18,ROPE,ROPE
0x72f020f8f3e8fd9382705723cd26380f8d0c66bb,18,PLOT,PLOT
0x6a7ef4998eb9d0f706238756949f311a59e05745,18,KenysiansNetwork,KEN
0x20945ca1df56d237fd40036d47e866c7dccd2114,18,NsureNetworkToken,Nsure
0x485d17a6f1b8780392d53d64751824253011a260,8,ChronoTechToken,TIME
0x12e51e77daaa58aa0e9247db7510ea4b46f9bead,18,AaveInterestbearingYFI,aYFIv1
0xba3d9687cf50fe253cd2e1cfeede1d6787344ed5,18,AaveInterestbearingAaveToken,aAAVEv1
0xb124541127a0a657f056d9dd06188c4f1b0e5aab,18,AaveInterestbearingUniswap,aUNIv1
0x712db54daa836b53ef1ecbb9c6ba3b9efb073f40,18,AaveInterestbearingENJ,aENJv1
0xb753428af26e81097e7fd17f40c88aaa3e04902c,18,Spice,SFI
0x8888801af4d980682e47f1a9036e589479e835c5,18,88mph.app,MPH
0x5d8d9f5b96f4438195be9b99eee6118ed4304286,18,CoverProtocol,COVERv1
0xc3eb2622190c57429aac3901808994443b64b466,18,OROToken,ORO
0x6468e79a80c0eab0f9a2b574c8d5bc374af59414,18,E-RADIX,eXRD
0x3e780920601d61cedb860fe9c4a90c9ea6a35e78,18,BoostedFinance,BOOST
0x431ad2ff6a9c365805ebad47ee021148d6f7dbe0,18,dForce,DF
0x1695936d6a953df699c38ca21c2140d497c08bd9,18,SynLev,SYN
0x3db6ba6ab6f95efed1a6e794cad492faaabf294d,8,LTONetworkToken,LTO
0xf5238462e7235c7b62811567e63dd17d12c2eaa0,8,CACHEGold,CGT
0x4fe83213d56308330ec302a8bd641f1d0113a4cc,18,NuCypher,NU
0x9e70240d2a8a30a592d3f076441c4f303b26de12,8,WrappedOCT,OCT
0x6e0dade58d2d89ebbe7afc384e3e4f15b70b14d8,18,QuiverX,QRX
0x05079687d35b93538cbd59fe5596380cae9054a9,18,BitSong,BTSG
0x355c665e101b9da58704a8fddb5feef210ef20c0,18,dForceGOLDx,GOLDx
0x69948cc03f478b95283f7dbf1ce764d0fc7ec54c,18,AaveInterestbearingREN,aRENv1
0xe88f8313e61a97cec1871ee37fbbe2a8bf3ed1e4,18,SoraValidatorToken,VAL
0x0b38210ea11411557c13457d4da7dc6ea731b88a,18,API3,API3
0x3449fc1cd036255ba1eb19d65ff4ba2b8903a69a,18,BAC,BAC
0xa7ed29b253d8b4e3109ce07c80fc570f81b63696,18,BAS,BAS
0x26607ac599266b21d13c7acf7942c7701a8b699c,18,PowerIndexPoolToken,PIPT
0x3218a02f8f8b5c3894ce30eb255f10bcba13e654,18,MegaCryptoPolisMEGATokenMEGA,MEGA
0x07150e919b4de5fd6a63de1f9384828396f25fdc,9,BaseProtocol,BASE
0x91dfbee3965baaee32784c2d546b7a0c62f268c9,18,Bondly,BONDLY
0x9d79d5b61de59d882ce90125b18f74af650acb93,6,NeutrinoSystemBaseToken,NSBT
0xb62132e35a6c13ee1ee0f84dc5d40bad8d815206,18,Nexo,NEXO
0xffc97d72e13e01096502cb8eb52dee56f74dad7b,18,AaveinterestbearingAAVE,aAAVE
0x05ec93c0365baaeabf7aeffb0972ea7ecdd39cf1,18,AaveinterestbearingBAT,aBAT
0xa361718326c15715591c299427c62086f69923d9,18,AaveinterestbearingBUSD,aBUSD
0x028171bca77440897b824ca71d1c56cac55b68a3,18,AaveinterestbearingDAI,aDAI
0xac6df26a590f08dcc95d5a4705ae8abbc88509ef,18,AaveinterestbearingENJ,aENJ
0x39c6b3e42d6a679d7d776778fe880bc9487c2eda,18,AaveinterestbearingKNC,aKNC
0xa06bc25b5805d5f8d82847d191cb4af5a3e873e0,18,AaveinterestbearingLINK,aLINK
0xa685a61171bb30d4072b338c80cb7b2c865c873e,18,AaveinterestbearingMANA,aMANA
0xc713e5e149d5d0715dcd1c156a020976e7e56b88,18,AaveinterestbearingMKR,aMKR
0xcc12abe4ff81c9378d670de1b57f8e0dd228d77a,18,AaveInterestbearingREN,aREN
0x35f6b052c598d933d69a4eec4d04c73a191fe6c2,18,AaveinterestbearingSNX,aSNX
0x6c5024cd4f8a59110119c56f8933403a539555eb,18,AaveinterestbearingSUSD,aSUSD
0x101cc05f4a51c0319f570d5e146a8c625198e636,18,AaveinterestbearingTUSD,aTUSD
0xb9d7cb55f463405cdfbe4e90a6d2df01c2b92bf1,18,AaveinterestbearingUNI,aUNI
0xbcca60bb61934080951369a648fb03df4f96263c,6,AaveinterestbearingUSDC,aUSDC
0x3ed3b47dd13ec9a98b44e6204a523e766b225811,6,AaveinterestbearingUSDT,aUSDT
0x9ff58f4ffb29fa2266ab25e75e2a8b3503311656,8,AaveinterestbearingWBTC,aWBTC
0x030ba81f1c18d280636f32af80b9aad02cf0854e,18,AaveinterestbearingWETH,aWETH
0x5165d24277cd063f5ac44efd447b27025e888f37,18,AaveinterestbearingYFI,aYFI
0xdf7ff54aacacbff42dfe29dd6144a69b629f8c9e,18,AaveinterestbearingZRX,aZRX
0xf644d4477cd8db7791cea3013cb053b3fec4beb3,8,CryptoenterLIONtoken,LION
0xa8e7ad77c60ee6f30bac54e2e7c0617bd7b5a03e,18,zLOT,zLOT
0x191557728e4d8caa4ac94f86af842148c0fa8f7e,8,ECOTOKEN,ECO
0xdc9ac3c20d1ed0b540df9b1fedc10039df13f99c,18,UtrustToken,UTK
0x3472a5a71965499acd81997a54bba8d852c6e53d,18,Badger,BADGER
0xc944e90c64b2c07662a292be6244bdf05cda44a7,18,GraphToken,GRT
0x5a98fcbea516cf06857215779fd812ca3bef1b32,18,LidoDAOToken,LDO
0x77777feddddffc19ff86db637967013e6c6a116c,18,TornadoCash,TORN
0xae7ab96520de3a18e5e111b5eaab095312d7fe84,18,stETH,stETH
0xb4d930279552397bba2ee473229f89ec245bc365,18,MahaDAO,MAHA
0x57b946008913b82e4df85f501cbaed910e58d26c,18,MarlinPOND,POND
0x3593d125a4f7849a1b059e64f4517a86dd60c95d,18,MANTRADAO,OMv2
0x3432b6a60d23ca0dfca7761b7ab56459d9c964d0,18,FraxShare,FXS
0x1fdab294eda5112b7d066ed8f2e4e562d5bcc664,18,Spice,SPICE
0x111111111117dc0aa78b770fa6a738034120c302,18,1INCHToken,1INCH
0x054d64b73d3d8a21af3d764efd76bcaa774f3bb2,18,Plasma,PPAY
0x368b3a58b5f49392e5c9e4c998cb0bb966752e51,18,MIC,MIC
0x4b4d2e899658fb59b1d518b68fe836b100ee8958,18,MIS,MIS
0x03066da434e5264ef0b32f787923f974a5726fdc,18,BCS,BCS
0xee573a945b01b788b9287ce062a0cfc15be9fd86,18,Exeedme,XED
0x34950ff2b487d9e5282c5ab342d08a2f712eb79f,18,EFFORCEIEO,WOZX
0xbd2f0cd039e0bfcf88901c98c0bfac5ab27566e3,18,DynamicSetDollar,DSD
0xc770eefad204b5180df6a14ee197d99d808ee52d,18,FOX,FOX
0x4688a8b1f292fdab17e9a90c8bc379dc1dbd8713,18,CoverProtocolGovernanceToken,COVER
0x66a0f676479cee1d7373f3dc2e2952778bff5bd6,18,WiseToken,WISE
0x539f3615c1dbafa0d008d87504667458acbd16fa,18,FERA,FERA
0xffffffff2ba8f66d4e51811c5190992176930278,18,Furucombo,COMBO
0x2b4200a8d373d484993c37d63ee14aee0096cd12,18,USDFreeLiquidity,USDFL
0xaea46a60368a7bd060eec7df8cba43b7ef41ad85,18,Fetch,FET
0x6c5ba91642f10282b576d91922ae6448c9d52f4e,18,Phala,PHA
0xa8b12cc90abf65191532a12bb5394a714a46d358,18,POWBTC-35WT,pBTC35A
0x853d955acef822db058eb8505911ed77f175b99e,18,Frax,FRAX
0xe28b3b32b6c345a34ff64674606124dd5aceca30,18,InjectiveToken,INJ
0x0a50c93c762fdd6e56d86215c24aaad43ab629aa,8,LGOToken,LGO
0xf94b5c5651c888d928439ab6514b93944eee6f48,18,Yield,YLDAPP
0x63b4f3e3fa4e438698ce330e365e831f7ccd1ef4,18,CyberFiToken,CFi
0xd291e7a03283640fdc51b121ac401383a46cc623,18,RariGovernanceToken,RGT
0xfa5047c9c78b8877af97bdcb85db743fd7313d4a,18,ROOK,ROOK
0xae1eaae3f627aaca434127644371b67b18444051,8,YOP,YOP
0x87d73e916d7057945c9bcd8cdd94e42a6f47f776,18,NFTX,NFTX
0xa4eed63db85311e22df4473f87ccfc3dadcfa3e3,18,Rubic,RBC
0x73968b9a57c6e53d41345fd57a6e6ae27d6cdb2f,18,StakeDAOToken,SDT
0x3c4b6e6e1ea3d4863700d7f76b36b7f3d3f13e3d,8,VoyagerToken,VGX
0x9248c485b0b80f76da451f167a8db30f33c70907,18,Debase,DEBASE
0x8290333cef9e6d528dd5618fb97a76f268f3edd4,18,AnkrNetwork,ANKR
0xc719d010b63e5bbf2c0551872cd5316ed26acd83,18,DecentralizedInsuranceProtocol,DIPInsurance
0x3155ba85d5f96b2d030a4966af206230e46849cb,18,THORChainETH.RUNE,RUNE
0x374cb8c27130e2c9e04f44303f3c8351b9de61c1,18,BaoToken,BAO
0xfe3e6a25e6b192a42a44ecddcd13796471735acf,18,Reef.finance,REEF
0x4c19596f5aaff459fa38b0f7ed92f11ae6543784,8,TrustToken,TRU
0x86772b1409b61c639eaac9ba0acfbb6e238e5f83,18,Indexed,NDX
0x67c597624b17b16fb77959217360b7cd18284253,9,Benchmark,MARK
0xf0939011a9bb95c3b791f0cb546377ed2693a574,18,Zero.ExchangeToken,ZERO
0x7240ac91f01233baaf8b064248e80feaa5912ba3,18,Octo.fi,OCTO
0x4c11249814f11b9346808179cf06e71ac328c1b5,18,OraichainToken,ORAI
0xfbeea1c75e4c4465cb2fccc9c6d6afe984558e20,18,DuckDaoDime,DDIM
0x70401dfd142a16dc7031c56e862fc88cb9537ce0,18,Bird.Money,BIRD
0xcae72a7a0fd9046cf6b165ca54c9e3a3872109e0,18,AnRKeyX,ANRX
0x9b02dd390a603add5c07f9fd9175b7dabe8d63b7,18,Shopping.io,SPIShopping
0x86ed939b500e121c0c5f493f399084db596dad20,18,SpaceChainV2,SPC
0x33d0568941c0c64ff7e0fb4fba0b11bd37deed9f,18,RAMPDEFI,RAMP
0xb987d48ed8f2c468d52d6405624eadba5e76d723,18,StabilizeToken,STBZ
0x159751323a9e0415dd3d6d42a1212fe9f4a0848c,18,INFI,INFI
0xa0b73e1ff0b80914ab6fe0444e65848c4c34450b,8,CRO,CRO
0xa8b919680258d369114910511cc87595aec0be6d,18,LUKSOToken,LYXe
0xd084b83c305dafd76ae3e1b4e1f1fe2ecccb3988,18,TerraVirtuaKolect,TVK
0x798d1be841a82a273720ce31c822c61a67a601c3,9,Digg,DIGG
0xffed56a180f23fd32bc6a1d8d3c09c283ab594a8,18,FL,FL
0x5580ab97f226c324c671746a1787524aef42e415,18,JUL,JUL
0xa1faa113cbe53436df28ff0aee54275c13b40975,18,AlphaToken,ALPHA
0x817bbdbc3e8a1204f3691d14bb44992841e3db35,18,CudosToken,CUDOS
0xde4ee8057785a7e8e800db58f9784845a5c2cbd6,18,Dexe,DEXE
0x3845badade8e6dff049820680d1f14bd3903a5d0,18,SAND,SAND
0x3c03b4ec9477809072ff9cc9292c9b25d4a8e6c6,18,PolkaCover,CVR
0x79256db1bdb6259315a1a3d7dd237f69cadfd8fc,18,Typhoon,PHOON
0xe5caef4af8780e59df925470b050fb23c43ca68c,6,FerrumNetworkToken,FRM
0x298d492e8c1d909d3f63bc4a36c66c64acb3d695,18,PolkaBridge,PBR
0xfe9a29ab92522d14fc65880d817214261d8479ae,18,SnowSwap,SNOW
0x220b71671b649c03714da9c621285943f3cbcdc6,18,TosDis,DIS
0x69a95185ee2a045cdc4bcd1b1df10710395e4e23,18,PoolzFinance,POOLZ
0xe4815ae53b124e7263f08dcdbbb757d41ed658c6,18,Zks,ZKS
0x1337def16f9b486faed0293eb623dc8395dfe46a,18,Armor,ARMOR
0x1337def18c680af1f9f45cbcab6309562975b1dd,18,ArmorNXM,arNXM
0x888888888889c00c67689029d7856aac1065ec11,18,OpiumGovernanceToken,OPIUM
0x3fa729b4548becbad4eab6ef18413470e6d5324c,18,Holyheld,HH
0xb4bebd34f6daafd808f73de0d10235a92fbb6c3d,18,YearnEcosystemTokenIndex,YETI
0xfa2562da1bba7b954f26c74725df51fb62646313,18,ASSYIndex,ASSY
0x961c8c0b1aad0c0b10a51fef6a867e3091bcef17,18,DeFiYieldProtocol,DYP
0x7a3eb5727e33a8000030718e82481f33bc20c4e3,8,OpiumcDAIJuniorTranche,OPIUMLPa55e
0x2994529c0652d127b7842094103715ec5299bbed,18,yearnCurve.fiyDAIyUSDCyUSDTyBUSD,ybCRV
0xacd43e627e64355f1861cec6d3a6688b31a6f952,18,yearnDaiStablecoin,yDAI
0x597ad1e0c13bfe8025993d9e79c69e1c0233522e,6,yearnUSDC,yUSDC
0x2f08119c6f07c006695e079aafc638b8789faf18,6,yearnTetherUSD,yUSDT
0x37d19d1c4e1fa9dc47bd1ea12f742a0887eda74a,18,yearnTrueUSD,yTUSDyearn
0x7ff566e1d69deff32a7b244ae7276b9f90e9d0f6,18,yearnCurve.firenBTCwBTCsBTC,ycrvRenWSBTC
0x9aa8f427a17d6b0d91b6262989edc7d45d6aedf8,18,FARM_crvRenWBTC,FcrvRenWBTC
0x898bad2774eb97cf6b94605677f43b41871410b1,18,validator-Eth2,vETH2
0xe95a203b1a91a908f9b9ce46459d101078c2c3cb,18,AnkrEth2RewardBearingBond,aETHc
0xcbc1065255cbc3ab41a6868c22d1f1c573ab89fd,18,CreamETH2,CRETH2
0x4e15361fd6b4bb609fa63c81a2be19d873717870,18,FantomToken,FTM
0x8642a849d0dcb7a15a974794668adcfbe4794b56,18,Prosper,PROS
0xc888a0ab4831a29e6ca432babf52e353d23db3c2,18,FastSwapToken,FAST
0xa1afffe3f4d611d252010e3eaf6f4d77088b0cd7,9,reflect.finance,RFI
0xa47c8bf37f92abed4a126bda807a7b7498661acd,18,WrappedUSTToken,UST
0x3832d2f059e55934220881f831be501d180671a7,8,renDOGE,renDOGE
0x09a3ecafa817268f77be1283176b946c4ff2e608,18,WrappedMIRToken,MIR
0x0d7dea5922535087078dd3d7c554ea9f2655d4cb,18,BerezkaFLEX,FLEX
0xdc76450fd7e6352733fe8550efabff750b2de0e3,18,BerezkaDynamic,DYNA
0x7a8d51b82b36fa5b50fb77001d6d189e920d2f75,18,WrappedOpiumGovernanceToken,wOPIUM
0xefc1c73a3d8728dc4cf2a18ac5705fe93e5914ac,18,Metric.exchange,METRIC
0x1d37986f252d0e349522ea6c3b98cb935495e63e,18,ChartEx,CHART
0x725c263e32c72ddc3a19bea12c5a0479a81ee688,18,BridgeMutual,BMI
0xc666081073e8dff8d3d1c2292a29ae1a2153ec09,18,DigitexFutures,DGTX
0xa283aa7cfbb27ef0cfbcb2493dd9f4330e0fd304,18,MMToken,MM1
0x0000000000095413afc295d19edeb1ad7b71c952,18,Tokenlon,LON
0x1f3f9d3068568f8040775be2e8c03c103c61f3af,18,ArcherDAOGovernanceToken,ARCH
0xbb0a009ba1eb20c5062c790432f080f6597662af,18,BitBotV1,BBP
0x99fe3b1391503a1bc1788051347a1324bff41452,18,SportX,SX
0xa393473d64d2f9f026b60b6df7859a689715d092,8,LatticeToken,LTX
0x6fa0952355607dfb2d399138b7fe10eb90f245e4,18,ClashToken,SCT
0xbbff34e47e559ef680067a6b1c980639eeb64d24,18,LeverjGluon,L2
0xe0ad1806fd3e7edf6ff52fdb822432e847411033,18,OnX.finance,ONX
0x2791bfd60d232150bff86b39b7146c0eaaa2ba81,18,BiFi,BiFi
0xef3a930e1ffffacd2fc13434ac81bd278b0ecc8d,18,StaFi,FIS
0x3e9bc21c9b189c09df3ef1b824798658d5011937,18,LinearToken,LINA
0xb1f871ae9462f1b2c6826e88a7827e76f86751d4,18,GNYerc20,GNYerc20
0x9aeb50f542050172359a0e1a25a9933bc8c01259,8,oinfinance,OIN
0x6b33f15360cedbfb8f60539ec828ef52910aca9b,18,xINCH,xINCHb
0x8f6a193c8b3c949e1046f1547c3a3f0836944e4b,18,xINCH,xINCHa
0xac0104cca91d167873b8601d2e71eb3d4d8c33e0,18,Crowns,CWS
0x95ad61b0a150d79219dcf64e1e6cc01f0b64c4ce,18,SHIBAINU,SHIB
0x6d0f5149c502faf215c89ab306ec3e50b15e2892,18,PortionToken,PRT
0x8a9c4dfe8b9d8962b31e4e16f8321c44d48e246e,18,NameChangeToken,NCT
0xb6ca7399b4f9ca56fc27cbff44f4d2e4eef1fc81,18,Muse,MUSE
0x2b915b505c017abb1547aa5ab355fbe69865cc6d,6,Maps.meToken,MAPS
0x6e36556b3ee5aa28def2a8ec3dae30ec2b208739,18,BUILDFinance,BUILD
0x4f5fa8f2d12e5eb780f6082dd656c565c48e0f24,18,GourmetGalaxy,GUM
0xfa6de2697d59e88ed7fc4dfe5a33dac43565ea41,18,DEFITop5TokensIndex,DEFI5
0x17ac188e09a7890a1844e5e65471fe8b0ccfadf3,18,CryptocurrencyTop10TokensIndex,CC10
0x8eef5a82e6aa222a60f009ac18c24ee12dbf4b41,18,TixlToken,TXL
0x50de6856358cc35f3a9a57eaaa34bd4cb707d2cd,18,RAZOR,RAZOR
0x297d33e17e61c2ddd812389c2105193f8348188a,18,StrudelFinance,TRDL
0xec681f28f4561c2a9534799aa38e0d36a83cf478,18,YVS.Finance,YVS
0x8d3e855f3f55109d473735ab76f753218400fe96,18,Bundles,BUND
0xc28e27870558cf22add83540d2126da2e4b464c2,18,SashimiToken,SASHIMI
0x7968bc6a03017ea2de509aaa816f163db0f35148,6,Hedget,HGET
0xa0afaa285ce85974c3c881256cb7f225e3a1178a,18,WrappedCRES,wCRES
0x15d4c048f83bd7e37d49ea4c83a07267ec4203da,8,Gala,GALA
0x39795344cbcc76cc3fb94b9d1b15c23c2070c66d,9,SeigniorageShares,SHARE
0x9ed8e7c9604790f7ec589f99b94361d8aab64e5e,18,Unistake,UNISTAKE
0x910524678c0b1b23ffb9285a81f99c29c11cbaed,18,DokiDokiAzuki,AZUKI
0xe61fdaf474fac07063f2234fb9e60c1163cfa850,18,CoinUtilityToken,COIN
0x5f64ab1544d28732f0a24f4713c2c8ec0da089f0,18,DEXTFToken,DEXTF
0x018fb5af9d015af25592a014c4266a84143de7a0,18,mp3,MP3
0xb59490ab09a0f526cc7305822ac65f2ab12f9723,18,Litentry,LIT
0xd2877702675e6ceb975b4a1dff9fb7baf4c91ea9,18,WrappedLUNAToken,LUNA
0x00aba6fe5557de1a1d565658cbddddf7c710a1eb,18,EASYV2,EZ
0xb6ff96b8a8d214544ca0dbc9b33f7ad6503efd32,18,SYNC,SYNC
0x8a40c222996f9f3431f63bf80244c36822060f12,18,Finxflo,FXF
0xf9fbe825bfb2bf3e387af0dc18cac8d87f29dea8,18,BotOcean,BOTS
0x66c0dded8433c9ea86c8cf91237b14e10b4d70b7,18,MarsToken,Mars
0x7b3d36eb606f873a75a6ab68f8c999848b04f935,18,NFTLootBox.com,LOOT
0xc0ba369c8db6eb3924965e5c4fd0b4c1b91e305f,18,DLPDuckToken,DUCK
0xaef4f02e31cdbf007f8d98da4ae365188a0e9ecc,8,TheFamousToken,TFT
0xcca0c9c383076649604ee31b20248bc04fdf61ca,18,BitMaxtoken,BTMX
0xebd9d99a3982d547c5bb4db7e3b1f9f14b67eb83,18,EverestID,ID
0x00c83aecc790e8a4453e5dd3b0b4b3680501a7a7,18,SKALE,SKL
0x0f51bb10119727a7e5ea3538074fb341f56b09ad,18,DAOMaker,DAO
0x76c5449f4950f6338a393f53cda8b53b0cd3ca3a,18,BT.Finance,BT
0xb4467e8d621105312a914f1d42f10770c0ffe3c8,18,FlashToken,FLASH
0xf680429328caaacabee69b7a9fdb21a71419c063,18,ButterflyProtocolGovernanceToken,BFLY
0x6fc13eace26590b80cccab1ba5d51890577d83b2,18,Umbrella,UMB
0x903bef1736cddf2a537176cf3c64579c3867a881,9,ichi.farm,ICHI
0x1456688345527be1f37e9e627da0837d6f08c925,18,USDPStablecoin,USDP2
0x23b608675a2b2fb1890d3abbd85c5775c51691d5,18,UnisocksEdition0,SOCKS
0xaa4e3edb11afa93c41db59842b29de64b72e355b,18,MarginSwap,MFI
0x9af15d7b8776fa296019979e70a5be53c714a7ec,18,EvnToken,EVN
0xb97d5cf2864fb0d08b34a484ff48d5492b2324a0,18,KLON,KLON
0xe6c3502997f97f9bde34cb165fbce191065e068f,18,KBTC,KBTC
0xdcb01cc464238396e213a6fdd933e36796eaff9f,18,Yield,YLD
0x69e8b9528cabda89fe846c67675b5d73d463a916,18,BBB,OPEN
0x196c81385bc536467433014042788eb707703934,18,CTASKToken,CTASK
0xd7b7d3c0bda57723fb54ab95fd8f9ea033af37f2,18,PYLON,PYLON
0x89bd2e7e388fab44ae88bef4e1ad12b4f1e0911c,18,NUXPeanut.trade,NUX
0xa0bed124a09ac2bd941b10349d8d224fe3c955eb,18,DePay,DEPAY
0x6bff2fe249601ed0db3a87424a2e923118bb0312,18,FYOOZ,FYZ
0xfdc4a3fc36df16a78edcaf1b837d3acaaedb2cb4,18,ScifiToken,SCIFI
0xb9ef770b6a5e12e45983c5d80545258aa38f3b78,10,0chain,ZCN
0xadb2437e6f65682b85f814fbc12fec0508a7b1d0,18,UniCrypt,UNCX
0xed40834a13129509a89be39a9be9c0e96a0ddd71,18,WarpToken,WARP
0x875773784af8135ea0ef43b5a374aad105c5d39e,18,Idle,IDLE
0x0488401c3f535193fa8df029d9ffe615a06e74e6,18,SparkPoint,SRK
0x038a68ff68c393373ec894015816e33ad41bd564,18,Glitch,GLCH
0x10be9a8dae441d276a5027936c3aaded2d82bc15,18,unimexnetwork,UMX
0x5f0e628b693018f639d10e4a4f59bd4d8b2b6b44,18,WhiteheartToken,WHITE
0x3597bfd533a99c9aa083587b074434e61eb0a258,8,DENT,DENT
0xb1e9157c2fdcc5a856c8da8b2d89b6c32b3c1229,18,ZenfuseTradingPlatformToken,ZEFU
0x260e63d91fccc499606bae3fe945c4ed1cf56a56,18,MoonTools.io,MOONS
0x066798d9ef0833ccc719076dab77199ecbd178b0,18,SakeToken,SAKE
0x945facb997494cc2570096c74b5f66a3507330a1,18,mStableBTC,mBTC
0x1b40183efb4dd766f11bda7a7c3ad8982e998421,18,VesperToken,VSP
0x84810bcf08744d5862b8181f12d17bfd57d3b078,18,SharedStakeGovernanceToken,SGT
0xed0439eacf4c4965ae4613d77a5c2efe10e5f183,18,shroom.finance,SHROOM
0xaac41ec512808d64625576eddd580e7ea40ef8b2,18,gameswap.org,GSWAP
0x2688213fedd489762a281a67ae4f2295d8e17ecc,18,FUD.finance,FUD
0x03ab458634910aad20ef5f1c8ee96f1d6ac54919,18,RaiReflexIndex,RAI
0x95b3497bbcccc46a8f45f5cf54b0878b39f8d96c,18,UniDex,UNIDX
0x9ceb84f92a0561fa3cc4132ab9c0b76a59787544,18,DokiDokiFinance,DOKI
0xfc05987bd2be489accf0f509e44b0145d68240f7,18,ESSENTIA,ESS
0xc3771d47e2ab5a519e2917e61e23078d0c05ed7f,18,Gather,GTH
0xabe580e7ee158da464b51ee1a83ac0289622e6be,18,Offshift,XFT
0xca3fe04c7ee111f0bbb02c328c699226acf9fd33,18,seen.haus,SEEN
0x34612903db071e888a4dadcaa416d3ee263a87b9,18,ethart,arte
0x7ca4408137eb639570f8e647d9bd7b7e8717514a,18,AlpaToken,ALPA
0xa58a4f5c4bb043d2cc1e170613b74e767c94189b,18,UTUCoin,UTU
0x9a0aba393aac4dfbff4333b06c407458002c6183,18,ACoconut,AC
0x7eaf9c89037e4814dc0d9952ac7f888c784548db,18,Royale,ROYA
0x6399c842dd2be3de30bf99bc7d1bbf6fa3650e70,18,Premia,PREMIA
0x4ca0654f4fc1025cf1a17b7459c20ac0479522ad,18,RigelFinance,RIGEL
0x51e00a95748dbd2a3f47bc5c3b3e7b3f0fea666c,18,DVGToken,DVG
0x0cec1a9154ff802e7934fc916ed7ca50bde6844e,18,PoolTogether,POOL
0x72e9d9038ce484ee986fea183f8d8df93f9ada13,18,SMARTCREDITToken,SMARTCREDIT
0xcb5f72d37685c3d5ad0bb5f982443bc8fcdf570e,18,RootKit,ROOT
0x557b933a7c2c45672b610f8954a3deb39a51a8ca,18,REVV,REVV
0x38a2fdc11f526ddd5a607c1f251c065f40fbf2f7,18,PhoenixDAO,PHNX
0x7866e48c74cbfb8183cd1a929cd9b95a7a5cb4f4,18,DexKit,KIT
0x4691937a7508860f876c9c0a2a617e7d9e945d4b,18,WootradeNetwork,WOO
0xea1ea0972fa092dd463f2968f9bb51cc4c981d71,18,Modefi,MOD
0x946112efab61c3636cbd52de2e1392d7a75a6f01,18,Hydro,HYDRO
0x0fe629d1e84e171f8ff0c1ded2cc2221caa48a3f,18,Mask,MASK
0xf99d58e463a2e07e5692127302c20a191861b4d6,18,Anyswap,ANY
0xf1f955016ecbcd7321c7266bccfb96c68ea5e49b,18,Rally,RLY
0x16980b3b4a3f9d89e33311b5aa8f80303e5ca4f8,6,KIRANetwork,KEX
0xd13c7342e1ef687c5ad21b27c2b65d772cab5c8c,4,UltraToken,UOS
0x147faf8de9d8d8daae129b187f0d02d819126750,18,GeoDBCoin,GEO
0x8a854288a5976036a725879164ca3e91d30c6a1b,18,GET,GET
0x79c75e2e8720b39e258f41c37cc4f309e0b0ff80,8,PhantasmaStake,SOUL
0x9e78b8274e1d6a76a0dbbf90418894df27cbceb5,18,UniFi,UniFi
0x3a880652f47bfaa771908c07dd8673a787daed3a,18,DerivaDAO,DDX
0x6c28aef8977c9b773996d0e8376d2ee379446f2f,18,Quickswap,QUICK
0xa1d6df714f91debf4e0802a542e13067f31b8262,18,RFOX,RFOX
0x275f5ad03be0fa221b4c6649b8aee09a42d9412a,18,Monavale,MONA
0x9b53e429b0badd98ef7f01f03702986c516a5715,18,hybrixhydra,HY
0xe53ec727dbdeb9e2d5456c3be40cff031ab40a55,18,SuperFarm,SUPER
0x123151402076fc819b7564510989e475c9cd93ca,8,wrapped-DGLD,wDGLD
0xada0a1202462085999652dc5310a7a9e2bf3ed42,18,CoinSharesGoldandCryptoassetsIndexLite,CGI
0xea6412fb370e8d1605e6aeeaa21ad07c3c7e9f24,18,MUSH,MUSH
0x6149c26cd2f7b5ccdb32029af817123f6e37df5b,18,Launchpooltoken,LPOOL
0x69af81e73a73b40adf4f3d4223cd9b1ece623074,18,MaskNetwork,MASKNTWRK
0x7f3edcdd180dbe4819bd98fee8929b5cedb3adeb,18,xToken,XTK
0x7777777777697cfeecf846a76326da79cc606517,18,xSigma,SIG
0xf5581dfefd8fb0e4aec526be659cfab1f8c781da,18,HOPRToken,HOPR
0x6c972b70c533e2e045f333ee28b9ffb8d717be69,18,FoundryLogisticsToken,FRY
0x63f88a2298a5c4aee3c216aa6d926b184a4b2437,18,GAMECredits,GAME
0xc8d2ab2a6fdebc25432e54941cb85b55b9f152db,18,GRAP,GRAP
0x6de037ef9ad2725eb40118bb1702ebb27e4aeb24,18,RenderToken,RNDR
0x21bfbda47a0b4b5b1248c767ee49f7caa9b23697,18,OVR,OVR
0x2e1e15c44ffe4df6a0cb7371cd00d5028e571d14,18,Mettalex,MTLX
0xd478161c952357f05f0292b56012cd8457f1cfbf,18,Polkamarkets,POLK
0x48fb253446873234f2febbf9bdeaa72d9d387f94,18,BancorGovernanceToken,vBNT
0x011a014d5e8eb4771e575bb1000318d509230afa,8,CreamUniswapV2WBTC-ETH,crUNIV2WBTCETH
0x01da76dea59703578040012357b81ffe62015c2d,8,CreamyETH,crYETH
0x054b7ed3f45714d3091e82aad64a1588dc4096ed,8,CreamHuobiBTC,crHBTC
0x10a3da2bb0fae4d591476fd97d6636fd172923a8,8,CreamFTXToken,crHEGIC
0x10fdbd1e48ee2fd9336a482d746138ae19e649db,8,CreamFTXToken,crFTT
0x17107f40d70f4470d20cb3f138a052cae8ebd4be,8,CreamrenBTC,crRENBTC
0x197070723ce0d3810a0e47f06e935c30a480d4fc,8,CreamWrappedBTC,crWBTC
0x19d1666f543d42ef17f66e376944a22aea1a8e46,8,CreamCompound,crCOMP
0x1d0986fb43985c88ffa9ad959cc24e6a087c7e35,8,CreamAlphaToken,crALPHA
0x1ff8cdb51219a8838b52e9cac09b71e591bc998e,8,CreamBinanceUSD,crBUSD
0x21011bc93d9e515b9511a817a1ed1d6d468f49fc,8,CreamCoverProtocolGovernanceToken,crCOVER
0x228619cca194fbe3ebeb2f835ec1ea5080dafbb2,8,CreamSushiBar,crXSUSHI
0x22b243b96495c547598d9042b6f94b01c22b2e9e,8,CreamSwagToken,crSWAG
0x25555933a8246ab67cbf907ce3d1949884e82b55,8,CreamSynthsUSD,crSUSD
0x2a537fa9ffaea8c1a41d3c2b68a9cb791529366d,8,CreamDefiPulseIndex,crDPI
0x2db6c82ce72c8d7d770ba1b5f5ed0b6e075066d6,8,CreamAmp,crAMP
0x3225e3c669b39c7c8b3e204a8614bb218c5e31bc,8,CreamAAVEToken,crAAVE
0x338286c0bc081891a4bda39c7667ae150bf5d206,8,CreamSushiToken,crSUSHI
0x3623387773010d9214b10c551d6e7fc375d31f58,8,CreamMeta,crMTA
0x38f27c03d6609a86ff7716ad03038881320be4ad,8,CreamSushiSwapLPTokenDAI-ETH,crSLPDAIETH
0x3c6c553a95910f9fc81c98784736bd628636d296,8,CreamEmptySetDollar,crESD
0x44fbebd2f576670a6c33f6fc0b00aa8c5753b322,8,CreamUSDCoin,crUSDC
0x460ea730d204c822ce709f00a8e5959921715adc,8,CreamBasisCash,crBAC
0x4ee15f44c6f0d8d1136c83efd2e8e4ac768954c6,8,CreamyyCRV,crYYCRV
0x4fe11bc316b6d7a345493127fbe298b95adaad85,8,CreamUniswapV2USDC-ETH,crUNIV2USDCETH
0x59089279987dd76fc65bf94cb40e186b96e03cb3,8,CreamOriginToken,crOGN
0x5c291bc83d15f71fb37805878161718ea4b6aee9,8,CreamSushiSwapLPTokenETH-USDT,crSLPETHUSDT
0x5ecad8a75216cea7dff978525b2d523a251eea92,8,CreamSushiSwapLPTokenUSDC-ETH,crSLPUSDCETH
0x65883978ada0e707c3b2be2a6825b1c4bdf76a90,8,CreamAkropolis,crAKRO
0x697256caa3ccafd62bb6d3aa1c7c5671786a5fd9,8,CreamChainLinkToken,crLINK
0x6ba0c66c48641e220cf78177c144323b3838d375,8,CreamSushiSwapLPTokenSUSHI-ETH,crSLPSUSHIETH
0x73f6cba38922960b7092175c0add22ab8d0e81fc,8,CreamSushiSwapLPTokenWBTC-ETH,crSLPWBTCETH
0x797aab1ce7c01eb727ab980762ba88e7133d2157,8,CreamUSDT,crUSDT
0x7aaa323d7e398be4128c7042d197a2545f0f1fea,8,CreamOMGToken,crOMG
0x7ea9c63e216d5565c3940a2b3d150e59c2907db3,8,CreamBinanceWrappedBTC,crBBTC
0x85759961b116f1d36fd697855c57a6ae40793d9b,8,Cream1INCHToken,cr1INCH
0x8b3ff1ed4f36c2c2be675afb13cc3aa5d73685a5,8,CreamCelsius,crCEL
0x8b86e0598616a8d4f1fdae8b59e55fb5bc33d0d6,8,CreamEthLendToken,crLEND
0x8b950f43fcac4931d408f1fcda55c6cb6cbf3096,8,CreamBadgerSettBadger,crBBADGER
0x903560b1cce601794c584f58898da8a8b789fc5d,8,CreamKeep3rV1,crKP3R
0x92b767185fb3b04f881e3ac8e5b0662a027a1d9f,8,CreamDaiStablecoin,crDAI
0x9baf8a5236d44ac410c0186fe39178d5aad0bb87,8,CreamyCRV,crYCRV
0xb092b4601850e23903a42eacbc9d8a0eec26a4d5,8,CreamFrax,crFRAX
0xc25eae724f189ba9030b2556a1533e7c8a732e14,8,CreamSynthetixNetworkToken,crSNX
0xc68251421edda00a10815e273fa4b1191fac651b,8,CreamPickleToken,crPICKLE
0xc7fd8dcee4697ceef5a2fd4608a7bd6a94c77480,8,CreamCRV,crCRV
0xcbae0a83f4f9926997c8339545fb8ee32edc6b76,8,CreamYFI,crYFI
0xcd22c4110c12ac41acefa0091c432ef44efaafa0,8,CreamUniswapV2DAI-ETH,crUNIV2DAIETH
0xce4fe9b4b8ff61949dcfeb7e03bc9faca59d2eb3,8,CreamBalancer,crBAL
0xd06527d5e56a3495252a528c4987003b712860ee,8,CreamEther,crETH
0xd5103afcd0b3fa865997ef2984c66742c51b2a8b,8,CreamHuobiFIL,crHFIL
0xd532944df6dfd5dd629e8772f03d4fc861873abf,8,CreamSushiSwapLPTokenYFI-ETH,crSLPYFIETH
0xd692ac3245bb82319a31068d6b8412796ee85d2c,8,CreamHUSD,crHUSD
0xe6c3120f38f56deb38b69b65cc7dcaf916373963,8,CreamUniswapV2ETH-USDT,crUNIV2ETHUSDT
0xe89a6d0509faf730bd707bf868d9a2a744a363c7,8,CreamUniswap,crUNI
0xef58b2d5a1b8d3cde67b8ab054dc5c831e9bc025,8,CreamSerum,crSRM
0xeff039c3c1d668f408d09dd7b63008622a77532c,8,CreamWrappedNXM,crWNXM
0xf047d4be569fb770db143a6a90ef203fc1295922,8,CreamtBTC,crTBTC
0xf55bbe0255f7f4e70f63837ff72a577fbddbe924,8,CreamBarnBridgeGovernanceToken,crBOND
0xfd609a03b393f1a1cfcacedabf068cad09a924e2,8,CreamCreamETH2,crCRETH2
0x6e9730ecffbed43fd876a264c982e254ef05a0de,18,NordToken,NORD
0x661ab0ed68000491d98c796146bcf28c20d7c559,18,ShadowsNetwork,DOWS
0x0cdf9acd87e940837ff21bb40c9fd55f68bba059,18,PublicMint,MINT
0x126c121f99e1e211df2e5f8de2d96fa36647c855,18,DEGENIndex,DEGEN
0x0c7d5ae016f806603cb1782bea29ac69471cab9c,18,Bifrost,BFC
0xd23ac27148af6a2f339bd82d0e3cff380b5093de,18,SIREN,SI
0x4c25bdf026ea05f32713f00f73ca55857fbf6342,18,Font,FONT
0x68a3637ba6e75c0f66b61a42639c4e9fcd3d4824,18,MoonToken,MOON
0x4b1e80cac91e2216eeb63e29b957eb91ae9c2be8,18,Jupiter,JUP
0xa44e5137293e855b1b7bc7e2c6f8cd796ffcb037,8,SENTinel,SENT
0x88ef27e69108b2633f8e1c184cc37940a075cc02,18,dego.finance,DEGO
0xee06a81a695750e71a662b51066f2c74cf4478a0,18,decentral.games,DG
0x0dde6f6e345bfd23f3f419f0dfe04e93143b44fb,18,SOTA,SOTA
0x739763a258640919981f9ba610ae65492455be53,18,NodeRunners,NDR
0xd82bb924a1707950903e2c0a619824024e254cd1,18,DAOfi,DAOfi
0x31c8eacbffdd875c74b94b077895bd78cf1e64a3,18,Radicle,RAD
0xdbdb4d16eda451d0503b854cf79d55697f90c8df,18,Alchemix,ALCX
0x24a6a37576377f63f194caa5f518a60f45b42921,18,FloatBank,BANK
0x9b99cca871be05119b2012fd4474731dd653febe,18,Antimatter.FinanceGovernanceToken,MATTER
0xf4d861575ecc9493420a3f5a14f85b13f0b50eb3,18,FractalProtocolToken,FCL
0xf411903cbc70a74d22900a5de66a2dda66507255,18,VERA,VRA
0xe5feeac09d36b18b3fa757e5cf3f8da6b8e27f4c,18,NFTINDEX,NFTI
0x5affe41805a9e57fed3657d0e64d96aea0b77885,6,OpiumUSDTProtectionv2USDC,OPIUMLP34b7
0xf3dcbc6d72a4e1892f7917b7c43b74131df8480e,18,BDPToken,BDP
0xbbc2ae13b23d715c30720f079fcd9b4a74093505,18,EthernityChainERNToken,ERN
0x67b6d479c7bb412c54e03dca8e1bc6740ce6b99c,18,KylinNetwork,KYL
0xfb5453340c03db5ade474b27e68b6a9c6b2823eb,18,MetaFactory,ROBOT
0x59e9261255644c411afdd00bd89162d09d862e38,18,ETHA,ETHA
0x4c6ec08cf3fc987c6c4beb03184d335a2dfc4042,18,Paint,PAINT
0x2aeccb42482cc64e087b6d2e5da39f5a7a7001f8,18,RulerProtocol,RULER
0x892a6f9df0147e5f079b0993f486f9aca3c87881,9,unification.comxfund,xFUND
0x7a5ce6abd131ea6b148a022cb76fc180ae3315a6,18,bAlpha,bALPHA
0x80ab141f324c3d6f2b18b030f1c4e95d4d658778,18,DEA,DEA
0x3506424f91fd33084466f402d5d97f05f8e3b4af,18,chiliZ,CHZ
0x41d5d79431a913c4ae7d69a668ecdfe5ff9dfb68,18,InverseDAO,INV
0xeeaa40b28a2d1b0b08f6f97bb1dd4b75316c6107,18,GOVI,GOVI
0xcf3c8be2e2c42331da80ef210e9b1b307c03d36a,18,BetProtocolToken,BEPRO
0x1cbb83ebcd552d5ebf8131ef8c9cd9d9bab342bc,18,Non-FungibleYearn,NFY
0x2216e873ea4282ebef7a02ac5aea220be6391a7c,18,smol,SMOL
0x92e187a03b6cd19cb6af293ba17f2745fd2357d5,18,UnitProtocol,DUCKUNIT
0x888888435fde8e7d4c54cab67f206e4199454c60,18,DFXToken,DFX
0xc4de189abf94c57f396bd4c52ab13b954febefd8,18,B.20,B20
0x7f1f2d3dfa99678675ece1c243d3f7bc3746db5d,18,Tapmydata,TAP
0x00d1793d7c3aae506257ba985b34c76aaf642557,18,Tacos,TACO
0x19a2cf2a1b2f76e52e2b0c572bd80a95b4fa8643,18,FyoozNFT,FYZNFT
0x92ef4ffbfe0df030837b65d7fccfe1abd6549579,18,Swirge,SWG
0xed30dd7e50edf3581ad970efc5d9379ce2614adb,18,ARCGovernanceTokenOld,ARCXOLD
0xcd91538b91b4ba7797d39a2f66e63810b50a33d0,18,ARCSTABLEx,STABLEx
0x940a2db1b7008b6c776d4faaca729d6d4a4aa551,18,DuskNetwork,DUSK
0xaa6e8127831c9de45ae56bb1b0d4d4da6e5665bd,18,ETH2xFlexibleLeverageIndex,ETH2xFLI
0x50d1c9771902476076ecfc8b2a83ad6b9355a4c9,18,FTT,FTXToken
0x037a54aab062628c9bbae1fdb1583c195585fe41,18,LCX,LCX
0x544c42fbb96b39b21df61cf322b5edc285ee7429,18,InsurAce,INSUR
0x29239242a83479a4074cb1c9e2a3e6705a4a4455,18,TOZEX,TOZ
0xf938424f7210f31df2aee3011291b658f872e91e,18,VISOR,VISR
0x8185bc4757572da2a610f887561c32298f1a5748,18,Aluna,ALN
0x1fe24f25b1cf609b9c4e7e12d802e3640dfa5e43,18,ChainGuardiansGovernanceToken,CGG
0x32a7c02e79c4ea1008dd6564b35f131428673c41,18,CRUST,CRU
0xac51066d7bec65dc4589368da368b212745d63e8,6,ALICE,ALICE
0x1c9922314ed1415c95b9fd453c3818fd41867d0b,18,TOWER,TOWER
0xa8b61cff52564758a204f841e636265bebc8db9b,18,YieldProtocol,YIELD
0x850aab69f0e0171a9a49db8be3e71351c8247df4,18,Konomi,KONO
0xbae5f2d8a1299e5c4963eaff3312399253f27ccb,9,SOAR.FI,SOAR
0xac3211a5025414af2866ff09c23fc18bc97e79b1,18,DOVU,DOV
0x3f4e02741b155f5ce8d6190d294d4f916125b896,18,CircleEx,CX
0x73374ea518de7addd4c2b624c0e8b113955ee041,18,JuggernautDeFi,JGN
0xfad45e47083e4607302aa43c65fb3106f1cd7607,9,hoge.finance,HOGE
0xe9a95d175a5f4c9369f3b74222402eb1b837693b,8,ChangeNOW,NOW
0xeabb8996ea1662cad2f7fb715127852cd3262ae9,18,ConnectFinancial,CNFI
0xdea67845a51e24461d5fed8084e69b426af3d5db,18,HodlTree,HTRE
0x8b39b70e39aa811b69365398e0aace9bee238aeb,18,PolkaFoundry,PKF
0xb20043f149817bff5322f1b928e89abfc65a9925,8,EXRT,EXRT
0xa487bf43cf3b10dffc97a9a744cbb7036965d3b9,18,Deri,DERI
0x14da7b27b2e0fedefe0a664118b0c9bc68e2e9af,18,BlockchainCutiesUniverseGovernanceToken,BCUG
0x8b0e42f366ba502d787bb134478adfae966c8798,18,LABSGroup,LABS
0xcbfef8fdd706cde6f208460f2bf39aa9c785f05d,18,KineGovernanceToken,KINE
0xd9c2d319cd7e6177336b0a9c93c21cb48d84fb54,18,HAPI,HAPI
0xc5bddf9843308380375a611c18b50fb9341f502a,18,veCRV-DAOyVault,yveCRVDAO
0xb9d99c33ea2d86ec5ec6b8a4dd816ebba64404af,18,k21.kanon.art,K21
0x226f7b842e0f0120b7e194d05432b3fd14773a9d,18,UNIONProtocolGovernanceToken,UNN
0x0ace32f6e87ac1457a5385f8eb0208f37263b415,10,HabitatToken,HBT
0xa42f266684ac2ad6ecb00df95b1c76efbb6f136c,18,CashTech,CATE
0xbc4171f45ef0ef66e76f979df021a34b46dcc81d,18,Dorayaki,DORA
0x07bac35846e5ed502aa91adf6a9e7aa210f2dcbe,18,erowan,erowan
0x7865af71cf0b288b4e7f654f4f7851eb46a2b7f8,18,Sentivate,SNTVT
0xc4c2614e694cf534d407ee49f8e44d125e4681c4,18,ChainGames,CHAIN
0x000000000000d0151e748d25b766e77efe2a6c83,18,XDEFIGovernanceToken,XDEX
0x8564653879a18c560e7c0ea0e084c516c62f5653,18,UpBots,UBXT
0x9f9c8ec3534c3ce16f928381372bfbfbfb9f4d24,18,GraphLinq,GLQ
0xc690f7c7fcffa6a82b79fab7508c466fefdfc8c5,18,Lympotokens,LYM
0x3d3d35bb9bec23b06ca00fe472b50e7a4c692c30,18,Vidya,VIDYA
0xf921ae2dac5fa128dc0f6168bf153ea0943d2d43,8,FireProtocol,FIRE
0xfc979087305a826c2b2a0056cfaba50aad3e6439,18,DAFIToken,DAFI
0xcd2828fc4d8e8a0ede91bb38cf64b1a81de65bf6,18,OddzToken,ODDZ
0x72630b1e3b42874bf335020ba0249e3e9e47bafc,18,PaypolitanToken,EPAN
0xe3cb486f3f5c639e98ccbaf57d95369375687f80,8,renDGB,renDGB
0x0ace20a35ee382bda90d1d93cee65f5970fc01c8,18,Chip,CHIP
0xa92e7c82b11d10716ab534051b271d2f6aef7df5,18,AraToken,ARA
0x5dc02ea99285e17656b8350722694c35154db1e8,8,bonded.finance,BONDfinance
0xc96c1609a1a45ccc667b2b7fa6508e29617f7b69,18,2GT_token,2GT
0xc3d088842dcf02c13699f936bb83dfbbc6f721ab,18,VoucherEthereum,vETH
0xca0e7269600d353f70b14ad118a49575455c0f2f,18,AMLT,AMLT
0xceb286c9604c542d3cc08b41aa6c9675b078a832,18,VortexDeFi,VTX
0xba4cfe5741b357fa371b506e5db0774abfecf8fc,18,vVSPpool,vVSP
0xc7283b66eb1eb5fb86327f08e1b5816b0720212b,18,Tribe,TRIBE
0x956f47f50a910163d8bf957cf5846d573e7f87ca,18,Fei USD,FEI
0x70e8de73ce538da2beed35d14187f6959a8eca96,6,XSGD,XSGD
0x626e8036deb333b408be468f951bdb42433cbf18,18,AIOZNetwork,AIOZ
0xa0cf46eb152656c7090e769916eb44a138aaa406,18,Spheroid,SPH
0x1796ae0b0fa4862485106a0de9b654efe301d0b2,18,Polkamon,PMON
0xf293d23bf2cdc05411ca0eddd588eb1977e8dcd4,18,Sylo,SYLO
0xdb0acc14396d108b3c5574483acb817855c9dc8d,8,Emblem,EMB
0x1735db6ab5baa19ea55d0adceed7bcdc008b3136,18,UREEQAToken,URQA
0x99295f1141d58a99e939f7be6bbe734916a875b8,18,LinkPool,LPL
0xe1c7e30c42c24582888c758984f6e382096786bd,8,Curate,XCUR
0xff75ced57419bcaebe5f05254983b013b0646ef5,18,CookToken,COOK
0x26c8afbbfe1ebaca03c2bb082e69d0476bffe099,18,CellframeToken,CELL
0x5b09a0371c1da44a8e24d36bf5deb1141a84d875,18,MADToken,MAD
0xc834fa996fa3bec7aad3693af486ae53d8aa8b50,18,Convergence,CONV
0x106538cc16f938776c7c180186975bca23875287,18,BASv2,BASv2
0x3505f494c3f0fed0b594e01fa41dd3967645ca39,18,SWARM,SWM
0xfbbe9b1142c699512545f47937ee6fae0e4b0aa9,18,EDDA,EDDA
0x29cbd0510eec0327992cd6006e63f9fa8e7f33b7,18,TidalToken,TIDAL
0xb78b3320493a4efaa1028130c5ba26f0b6085ef8,18,DraculaToken,DRC2
0x55296f69f40ea6d20e478533c15a6b08b654e758,18,XYOracle,XYO
0x0f71b8de197a1c84d31de0f1fa7926c365f052b3,18,ArconaDistributionContract,ARCONA
0x9534ad65fb398e27ac8f4251dae1780b989d136e,18,PYRToken,PYR
0xa31b1767e09f842ecfd4bc471fe44f830e3891aa,18,ROOBEE,ROOBEE
0xf3ae5d769e153ef72b4e3591ac004e89f48107a1,18,DeeperNetwork,DPR
0xd9b312d77bc7bed9b9cecb56636300bed4fe5ce9,18,Gains,GAINS
0x5f98805a4e8be255a32880fdec7f6728c6568ba0,18,LUSDStablecoin,LUSD
0x1da87b114f35e1dc91f72bf57fc07a768ad40bb0,18,Equalizer,EQZ
0xe0b9a2c3e9f40cf74b2c7f591b2b0cca055c3112,18,GenShards,GS
0xa8006c4ca56f24d6836727d106349320db7fef82,8,Internxt,INXT
0x383518188c0c6d7730d91b2c03a03c837814a899,9,Olympus,OHM
0x5eaa69b29f99c84fe5de8200340b4e9b4ab38eac,18,RazeNetwork,RAZE
0x88acdd2a6425c3faae4bc9650fd7e27e0bebb7ab,18,Alchemist,MIST
0x3d6f0dea3ac3c607b3998e6ce14b6350721752d9,18,CARD.STARTER,CARDS
0x33840024177a7daca3468912363bed8b425015c5,18,ethboxToken,EBOX
0xec213f83defb583af3a000b1c0ada660b1902a0f,18,Presearch,PRE
0xf418588522d5dd018b425e472991e52ebbeeeeee,18,EthereumPushNotificationService,PUSH
0xd8e3fb3b08eba982f2754988d70d57edc0055ae6,9,Zoracles,ZORA
0xc477d038d5420c6a9e0b031712f61c5120090de9,18,BosonToken,BOSON
0xdd1ad9a21ce722c151a836373babe42c868ce9a4,18,UniversalBasicIncome,UBI
0x53c8395465a84955c95159814461466053dedede,18,DeGateToken,DG2
0x16c52ceece2ed57dad87319d91b5e3637d50afa4,18,TCAPToken,TCAP
0x44564d0bd94343f72e3c8a0d22308b7fa71db0bb,18,BasketDAOGov,BASK
0x5cf04716ba20127f1e2297addcf4b5035000c9eb,18,NKN,NKN
0x4730fb1463a6f1f44aeb45f6c5c422427f37f4d0,18,The4thPillarToken,FOUR
0x08d32b0da63e2c3bcf8019c9c5d849d7a9d791e6,0,Dentacoin,DCN
0x77fba179c79de5b7653f68b5039af940ada60ce0,18,AmpleforthGovernance,FORTH
0xe796d6ca1ceb1b022ece5296226bf784110031cd,18,BlindBoxesToken,BLES
0x48c3399719b582dd63eb5aadf12a40b4c3f52fa2,18,StakeWise,SWISE
0xdefa4e8a7bcba345f687a2f1456f5edd9ce97202,18,KyberNetworkCrystalv2,KNC
0x0ada190c81b814548ddc2f6adc4a689ce7c1fe73,18,yAxisV2,YAXIS
0xff56cc6b1e6ded347aa0b7676c85ab0b3d08b0fa,18,Orbs,ORBS
0xa02120696c7b8fe16c09c749e4598819b2b0e915,18,WirexToken,WXT
0x28a06c02287e657ec3f8e151a13c36a1d43814b0,18,BAG,BAG
0x2f109021afe75b949429fe30523ee7c0d5b27207,18,OCC,OCC
0x767fe9edc9e0df98e07454847909b5e959d7ca0e,18,Illuvium,ILV
0x5218e472cfcfe0b64a064f055b43b4cdc9efd3a6,18,UnFederalReserveToken,eRSDL
0xf16e81dce15b08f326220742020379b855b87df9,18,IceToken,ICE
0xe4cfe9eaa8cdb0942a80b7bc68fd8ab0f6d44903,18,XEND,XEND
0x5a666c7d92e5fa7edcb6390e4efd6d0cdd69cf37,18,UnmarshalToken,MARSH
0x2d80f5f5328fdcb6eceb7cacf5dd8aedaec94e20,4,AGAToken,AGA
0x3d658390460295fb963f54dc0899cfb1c30776df,8,CircuitsOfValue,Coval
0x7ff4169a6b5122b664c51c95727d87750ec07c84,18,10SetToken,10SET
0x1de5e000c41c8d35b9f1f4985c23988f05831057,18,BonFi,BNF
0x818fc6c2ec5986bc6e2cbf00939d90556ab12ce5,18,Kin,KIN
0x7dd9c5cba05e151c895fde1cf355c9a1d5da6429,18,GolemNetworkToken,GLM
0x467bccd9d29f223bce8043b84e8c8b282827790f,2,Telcoin,TEL
0x90de74265a416e1393a450752175aed98fe11517,18,UnlockDiscountToken,UDT
0x808507121b80c02388fad14726482e061b8da827,18,Pendle,PENDLE
0x7a2bc711e19ba6aff6ce8246c546e8c4b4944dfd,8,WAXEconomicToken,WAXE
0x15b543e986b8c34074dfc9901136d9355a537e7e,18,StudentCoin,STC
0x16eccfdbb4ee1a85a33f3a9b21175cd7ae753db4,18,Route,ROUTE
0x7c8155909cd385f120a56ef90728dd50f9ccbe52,15,Nahmii,NII
0x1614f18fc94f47967a3fbe5ffcd46d4e7da3d787,18,PAIDNetwork,PAID
0x182f4c4c97cd1c24e1df8fc4c053e5c47bf53bef,18,keyTangoToken,TANGO
0x7bef710a5759d197ec0bf621c3df802c2d60d848,18,SPLYTSHOPX,SHOPX
0x358aa737e033f34df7c54306960a38d09aabd523,18,AresProtocol,ARES
0xb453f1f2ee776daf2586501361c457db70e1ca0f,8,AGARewards,AGAr
0x0cab5a7dcab521af5404fa604b85113267d38444,6,OpiumxDAIOmniBridgeProtection,OPIUMLP1c60
0xb54539d39529ce58fb63877debc6d6b70e3eca01,6,OpiumMaticBridgeProtection,OPIUMLPaf0d
0x321c2fe4446c7c963dc41dd58879af648838f98d,18,Cryptex,CTX
0x2d94aa3e47d9d5024503ca8491fce9a2fb4da198,18,BanklessToken,BANKBANKLESS
0x0b498ff89709d3838a063f1dfa463091f9801c2b,18,BTC2xFlexibleLeverageIndex,BTC2xFLI
0x841fb148863454a3b3570f515414759be9091465,18,ShihTzu,SHIH
0x7e8539d1e5cb91d63e46b8e188403b3f262a949b,18,SOMIDAX,SMDX
0xa2b4c0af19cc16a6cfacce81f192b024d625817d,9,KishuInu,KISHU
0x389999216860ab8e0175387a0c90e5c52522c945,9,FEGtoken,FEG
0x212dd60d4bf0da8372fe8116474602d429e5735f,18,StoboxToken,STBU
0xee9801669c6138e84bd50deb500827b776777d28,18,O3SwapToken,O3
0xd0660cd418a64a1d44e9214ad8e459324d8157f1,12,Woofy,WOOFY
0xa4cf2afd3b165975afffbf7e487cdd40c894ab6b,0,ShibaKen.Finance,SHIBAKEN
0xb1191f691a355b43542bea9b8847bc73e7abb137,18,Kirobo,KIRO
0x4e3fbd56cd56c3e72c1403e103b45db9da5b9d2b,18,ConvexToken,CVX
0xa3c4dc4a9ce2a6b40b57f25f8b50decc2c64dec2,18,SeedSwapToken,SNFT
0xfeea0bdd3d07eb6fe305938878c0cadbfa169042,18,8PAYNetwork,8PAY
0xde30da39c46104798bb5aa3fe8b9e0e1f348163f,18,Gitcoin,GTC
0x01e0e2e61f554ecaaec0cc933e739ad90f24a86d,18,Graviton,GTON
0xbc6da0fe9ad5f3b0d58160288917aa56653660e9,18,AlchemixUSD,alUSD
0x7697b462a7c4ff5f8b55bdbc2f4076c2af9cf51a,18,Sarcophagus,SARCO
0xa1ed0364d53394209d61ae8bfdb8ff50484d8c91,18,TeraBlockToken,TBC
0x677ddbd918637e5f2c79e164d402454de7da8619,18,VUSD,VUSD
0x3c9d6c1c73b31c837832c72e04d3152f051fc1a9,18,BoringDAO,BOR
0x1341a2257fa7b770420ef70616f888056f90926c,9,ZooToken,ZOOT
0x761d38e5ddf6ccf6cf7c55759d5210750b5d60f3,18,Dogelon,ELON
0x0563dce613d559a47877ffd1593549fb9d3510d6,18,SuperBid,SUPERBID
0x2ab6bb8408ca3199b8fa6c92d5b455f820af03c4,18,TE-FOODTustChain,TONE
0x06a01a4d579479dd5d884ebf61a31727a3d8d442,8,SmartKey,Skey
0xeb58343b36c7528f23caae63a150240241310049,18,Nimbus,NBU
0x27c70cd1946795b66be9d954418546998b546634,18,DOGEKILLER,LEASH
0x0abdace70d3790235af448c88547603b945604ea,18,district0xNetworkToken,DNT
0xdef1fac7bf08f173d286bbbdcbeeade695129840,18,DefiFactoryToken,DEFT
0x9695e0114e12c0d3a3636fab5a18e6b737529023,18,DFYNToken,DFYN
0x72e364f2abdc788b7e918bc238b21f109cd634d7,18,MetaverseIndex,MVI
0x3301ee63fb29f863f2333bd4466acb46cd8323e6,18,AkitaInu,AKITA
0x6dea81c8171d0ba574754ef6f8b412f2ed88c54d,18,LQTY,LQTY
0x87de305311d5788e8da38d19bb427645b09cb4e5,18,Verox,VRX
0xf65b5c5104c4fafd4b709d9d60a185eae063276c,18,Truebit,TRUTruebit
0x9be89d2a4cd102d8fecc6bf9da793be995c22541,8,BinanceWrappedBTC,BBTC
0x944eee930933be5e23b690c8589021ec8619a301,9,MUNCHToken,MUNCH
0xdc349913d53b446485e98b76800b6254f43df695,9,BezogeEarth,BEZOGE
0x1681bcb589b3cfcf0c0616b0ce9b19b240643dc1,9,Island,ISLE
0xa150db9b1fa65b44799d4dd949d922c0a33ee606,0,DigitalReserveCurrency,DRC1
0xe047705117eb07e712c3d684f5b18e74577e83ac,8,BitcashPay,BCP
0x15874d65e649880c2614e7a480cb7c9a55787ff6,18,EthereumMax,eMax
0x6f40d4a6237c257fff2db00fa0510deeecd303eb,18,Instadapp,INST
0x456d8f0d25a4e787ee60c401f8b963a465148f70,9,Cavapoo,CAVA
0x8cb924583681cbfe487a62140a994a49f833c244,18,SwappToken,SWAPP
0x10633216e7e8281e33c86f02bf8e565a635d9770,18,Dvision,DVI
0xcadc0acd4b445166f12d2c07eac6e2544fbe2eef,18,CADCoin,CADC
0x1321f1f1aa541a56c31682c57b80ecfccd9bb288,18,ARCxGovernanceToken,ARCX
0xa9536b9c75a9e0fae3b56a96ac8edf76abc91978,18,AmunDeFiIndex,DFI
0x1660f10b4d610cf482194356ece8efd65b15ba83,18,AmunDeFiMomentumIndex,DMX
0x8798249c2e607446efb7ad49ec89dd1865ff4272,18,SushiBar,xSUSHI
0xd7c49cee7e9188cca6ad8ff264c1da2e69d4cf3b,18,NXM,NXM
0xc76fb75950536d98fa62ea968e1d6b45ffea2a55,18,COL,COL
0xa9b1eb5908cfc3cdf91f9b8b3a74108598009096,18,BounceToken,Auction
0x5b7533812759b45c2b44c19e320ba2cd2681b542,8,SingularityNETToken,AGIX
0x6595b8fd9c920c81500dca94e53cdc712513fb1f,18,Olyseum,OLY
0xc52c326331e9ce41f04484d3b5e5648158028804,18,ZENExchangeToken,ZCX
0x3431f91b3a388115f00c5ba9fdb899851d005fb5,18,GeroWallet,GERO
0x474021845c4643113458ea4414bdb7fb74a01a77,18,UnoRe,UNO
0xcaaa93712bdac37f736c323c93d4d5fdefcc31cc,18,CryptalDash,CRD
0x1f8a626883d7724dbd59ef51cbd4bf1cf2016d13,18,Jigstack,STAK
0x33349b282065b0284d756f0577fb39c158f935e6,18,MapleToken,MPL
0x6243d8cea23066d098a15582d81a598b4e8391f4,18,FlexUngovernanceToken,FLX
0xc53342fd7575f572b0ff4569e31941a5b821ac76,18,ETHVolatilityIndex,ETHV
0x3a707d56d538e85b783e8ce12b346e7fb6511f90,18,InverseETHVolatilityIndex,iETHV
0x51b0bcbeff204b39ce792d1e16767fe6f7631970,18,BTCVolatilityIndex,BTCV
0x2590f1fd14ef8bb0a46c7a889c4cbc146510f9c3,18,InverseBTCVolatilityIndex,iBTCV
0x69fa8e7f6bf1ca1fb0de61e1366f7412b827cc51,9,EnreachDAO,NRCH
0xfb7b4564402e5500db5bb6d63ae671302777c75a,18,DEXTools,DEXT
0x853bb55c1f469902f088a629db8c8803a9be3857,18,Stable1INCH,one1INCH
0x75d12e4f91df721fafcae4c6cd1d5280381370ac,9,Mybu,MYOBU
0xdddddd4301a082e62e84e43f474f044423921918,18,DeversiFiToken,DVF
0x6b4c7a5e3f0b99fcd83e9c089bddd6c7fce5c611,18,Million,MM2
0xc581b735a1688071a1746c968e0798d642ede491,6,EuroTether,EURT
0x72e5390edb7727e3d4e3436451dadaff675dbcc0,12,HanuYokia,HANU
0x011864d37035439e078d64630777ec518138af05,18,ZerogokiToken,REI
0x76417e660df3e5c90c0361674c192da152a806e4,18,ZerogokiUSD,zUSD
0x441761326490cacf7af299725b6292597ee822c2,18,UNFI,UNFI
0x8e6cd950ad6ba651f6dd608dc70e5886b1aa6b24,18,StarLink,STARL
0x70d2b7c19352bb76e4409858ff5746e500f2b67c,18,PawtocolNetworkUPIToken,UPI
0x382f0160c24f5c515a19f155bac14d479433a407,9,KleeKai,KLEE
0x1559fa1b8f28238fd5d76d9f434ad86fd20d1559,18,Eden,EDEN
0x1e6bb68acec8fefbd87d192be09bb274170a0548,9,AaveinterestbearingAMPL,aAMPL
0xe1030b48b2033314979143766d7dc1f40ef8ce11,18,ThePeoplesCoin,PEEPS
0x515d7e9d75e2b76db60f8a051cd890eba23286bc,18,Governor,GDAO
0xb26631c6dda06ad89b93c71400d25692de89c068,18,Minds,MINDS
0x43f11c02439e2736800433b4594994bd43cd066d,9,FLOKI,FLOKI
0x5166e09628b696285e3a151e84fb977736a83575,18,VolatilityProtocolToken,VOL
0x06f3c323f0238c72bf35011071f2b5b7f43a054c,18,MASQ,MASQ
0xc4f6e93aeddc11dc22268488465babcaf09399ac,18,hiDollar,HI
0x106552c11272420aad5d7e94f8acab9095a6c952,9,KeanuInu,KEANU
0xddb3422497e61e13543bea06989c0789117555c5,18,COTIToken,COTI
0x92d6c1e31e14520e676a687f0a93788b716beff5,18,dYdX,DYDX
0xa4ef4b0b23c1fc81d3f9ecf93510e64f58a4a016,18,1MILNFT,1MIL
0xeb494890465f49c2b94457d9b61811392e5b1fea,9,SLINKLABS,SLAB
0x25e1474170c4c0aa64fa98123bdc8db49d7802fa,18,Bidao,BID
0x61107a409fffe1965126aa456af679719695c69c,18,UmiToken,UMI
0x738865301a9b7dd80dc3666dd48cf034ec42bdda,8,AgorasToken,AGRS
0x0af55d5ff28a3269d69b98680fd034f115dd53ac,8,BankSocial,BSL
0x4297394c20800e8a38a619a243e9bbe7681ff24e,18,HotCrossToken,HOTCROSS
0xd85ad783cc94bd04196a13dc042a3054a9b52210,18,TribeOne,HAKA
0x8b3192f5eebd8579568a2ed41e6feb402f93f73f,9,SaitamaInu,SAITAMA
0x62dc4817588d53a056cbbd18231d91ffccd34b2a,18,DeHive.finance,DHV
0x19042021329fddcfbea5f934fb5b2670c91f7d20,6,TakeMyMuffin,TMM
0x15b7c0c907e4c6b9adaaaabc300c08991d6cea05,18,GelatoNetworkToken,GEL
0xa64dfe8d86963151e6496bee513e366f6e42ed79,9,GokuInu,GOKU
0xe94b97b6b43639e238c851a7e693f50033efd75c,18,RainbowToken,RNBW
0x99d8a9c45b2eca8864373a26d1459e3dff1e17f3,18,MagicInternetMoney,MIM
0x090185f2135308bad17527004364ebcc2d37e5f6,18,SpellToken,SPELL
0xde5ed76e7c05ec5e4572cfc88d1acea165109e44,18,DEUS,DEUS
0xf009f5531de69067435e32c4b9d36077f4c4a673,18,Unvest,UNV
0xc221b7e65ffc80de234bbb6667abdd46593d34f0,18,WrappedCentrifuge,wCFG
0x7f39c581f595b53c5cb19bd0b3f8da6c935e2ca0,18,WrappedliquidstakedEther2.0,wstETH
0xf50a07e4ff052a14f3f608da8936d8ae0ed5be50,9,FlokiPup,FLOKIPUP
0x636484a1c41e88e3fc7c99248ca0b3c3a844ab86,9,GogetaInu,GOGETA
0xaecc217a749c2405b5ebc9857a16d58bdc1c367f,9,Pawthereum,PAWTH
0x949d48eca67b17269629c7194f4b727d4ef9e5d6,18,MeritCircle,MC
0xd6014ea05bde904448b743833ddf07c3c7837481,18,SynthiBTC,iBTC
0xa9859874e1743a32409f75bb11549892138bba1e,18,SynthiETH,iETH
0xf48e200eaf9906362bb1442fca31e0835773b8b4,18,SynthsAUD,sAUD
0xfe18be6b3bd88a2d2a7f928d00292e7a9963cfc6,18,SynthsBTC,sBTC
0xe1afe1fd76fd88f78cbf599ea1846231b8ba3b6b,18,SynthsDEFI,sDEFI
0x5e74c9036fb86bd7ecdcb084a0673efc32ea31cb,18,SynthsETH,sETH
0xd71ecff9342a5ced620049e616c5035f1db98620,18,SynthsEUR,sEUR
0x514cdb9cd8a2fb2bdcf7a3b8ddd098caf466e548,9,RedPanda,REDPANDA
0xc50ef449171a51fbeafd7c562b064b6471c36caa,9,ZombieInu,ZINU
0x0c3685559af6f3d20c501b1076a8056a0a14426a,9,miniSAITAMA,miniSAITAMA
0x24e89bdf2f65326b94e36978a7edeac63623dafa,18,TigerKing,TKING
0xc1bfccd4c29813ede019d00d2179eea838a67703,9,LuffyInu,LUFFY
0xaaaaaa20d9e0e2461697782ef11675f668207961,18,Aurora,AURORA
0xed0889f7e1c7c7267407222be277e1f1ef4d4892,18,Melalie,MEL
0xc18360217d8f7ab5e7c516566761ea12ce7f9d72,18,EthereumNameService,ENS
0xf14b9adf84812ba463799357f4dc716b4384010b,8,PensionPlan,PP
0xfb40e79e56cc7d406707b66c4fd175e07eb2ae3c,9,ROTTSCHILD.com,ROTTS
0x28c5805b64d163588a909012a628b5a03c1041f9,9,CHOPPERINU,CHOPPER
0x37fe0f067fa808ffbdd12891c0858532cfe7361d,18,Civilization,CIV
0xf32aa187d5bc16a2c02a6afb7df1459d0d107574,18,HachikoInu,Inu
0xc08512927d12348f6620a698105e1baac6ecd911,6,GMOJPY,GYEN
0xcafe001067cdef266afb7eb5a286dcfd277f3de5,18,ParaSwap,PSP
0xbb1ee07d6c7baeb702949904080eb61f5d5e7732,18,Dogey-Inu,DINU
0x582d872a1b094fc48f5de31d3b73f2d9be47def1,9,WrappedTONCoin,TONCOIN
0x95a1796437bad6502d1c1cce165cd76e522409a9,18,OfficialCryptoCowboyToken,OCCT
0xe87e15b9c7d989474cb6d8c56b3db4efad5b21e8,18,HokkaidoInu,HOKK
0x4104b135dbc9609fc1a9490e61369036497660c8,18,APWineToken,APW
0xae78736cd615f374d3085123a210448e74fc6393,18,RocketPoolETH,rETH
0xafbf03181833ab4e8dec24d708a2a24c2baaa4a4,9,MandoX,MANDOX
0x1fbd3df007eb8a7477a1eab2c63483dcc24effd6,18,ScaleSwapToken,SCA
0xb1a88c33091490218965787919fcc9862c1798ee,9,StudioShibli,SHIBLI
0x8037b1b69d6fa63a9cc053c25f7e168e6e6d857a,18,PartsofFourCoin,P4C
0xcfeaead4947f0705a14ec42ac3d44129e1ef3ed5,8,Notional,NOTE
0x41a3dba3d677e573636ba691a70ff2d606c29666,18,GoBlankToken,BLANK
0xc48b4814faed1ccc885dd6fde62a6474aecbb19a,9,CoinMerge,CMERGE
0x8254e26e453eb5abd29b3c37ac9e8da32e5d3299,18,RBX,RBX"""


ethereum_erc20s = """address,decimals,name,symbol
0x006bea43baa3f7a6f765f14f10a1a1b08334ef45,18,Stox,STX
0x0327112423f3a68efdf1fcf402f6c5cb9f7c33fd,18,PieDAOBTC++,BTC
0x0417912b3a7af768051765040a55bb0925d4ddcf,18,LiquidityDividendsProtocol,LID
0x04fa0d235c4abf4bcf4787af4cf447de572ef828,18,UMAVotingTokenv1,UMA
0x07597255910a51509ca469568b048f2597e72504,18,Uptrennd,1UP
0x08d967bb0134f2d07f7cfb6e246680c53927dd30,18,MATHToken,MATH
0x0a913bead80f321e7ac35285ee10d9d922659cb7,18,DOSNetworkToken,DOS
0x0ae055097c6d159879521c384f1d2123d1f195e6,18,STAKE,STAKE
0x88df592f8eb5d7bd38bfef7deb0fbc02cf3778a0,18,TellorTributes,TRB
0x0bc529c00c6401aef6d220be8c6ea1667f6ad93e,18,yearn.finance,YFI
0x0cf0ee63788a0849fe5297f3407f701e122cc023,18,Streamr,DATA
0x0d438f3b5175bebc262bf23753c1e53d03432bde,18,WrappedNXM,wNXM
0x0d8775f648430679a709e98d2b0cb6250d2887ef,18,BasicAttentionToken,BAT
0x0e8d6b471e332f140e7d9dbb99e5e3822f728da6,18,Abyss,ABYSS
0x0f5d2fb29fb7d3cfee444a200298f468908cc942,18,Mana,MANA
0x0f7f961648ae6db43c75663ac7e5414eb79b5704,18,XIONetwork,XIO
0x12b19d3e2ccc14da04fae33e63652ce469b3f2fd,12,GRID,GRID
0x12f649a9e821f90bb143089a6e56846945892ffb,18,uDOO,uDOO
0x13339fd07934cd674269726edf3b5ccee9dd93de,18,CurToken,CUR
0x1453dbb8a29551ade11d89825ca812e05317eaeb,18,TendiesToken,TEND
0x1776e1f26f98b1a5df9cd347953a26dd3cb46671,18,Numeraire,NMR
0x178c820f862b14f316509ec36b13123da19a6054,18,EnergyWebTokenBridged,EWTB
0x1a5f9352af8af974bfc03399e3767df6370d82e4,18,OWLToken,OWL
0x1beef31946fbbb40b877a72e4ae04a8d1a5cee06,18,Parachute,PAR
0x196f4727526ea7fb1e17b2071b3d8eaa38486988,18,Reserve,RSV
0x1f573d6fb3f13d689ff844b4ce37794d79a7ff1c,18,Bancor,BNT
0x20f7a3ddf244dc9299975b4da1c39f8d5d75f05a,6,SapienNetwork,SPN
0x221657776846890989a759ba2973e427dff5c9bb,18,ReputationV2,REPv2
0x2260fac5e5542a773aa44fbcfedf7c193bc2c599,8,WrappedBTC,WBTC
0x255aa6df07540cb5d3d297f0d0d4d84cb52bc8e6,18,RaidenNetworkToken,RDN
0x26b3038a7fc10b36c426846a9086ef87328da702,18,YieldFarmingToken,YFT
0x27054b13b1b798b345b591a4d22e6562d47ea75a,4,AirSwap,AST
0x28cb7e841ee97947a86b06fa4090c8451f64c0be,18,YFLink,YFL
0x28dee01d53fed0edf5f6e310bf8ef9311513ae40,18,BlitzPredict,XBP
0x2b591e99afe9f32eaa6214f7b7629768c40eeb39,8,HEX,HEX
0x2ba592f78db6436527729929aaf6c908497cb200,18,Cream,CREAM
0x2c4e8f2d746113d0696ce89b35f0d8bf88e0aeca,18,OpenSimpleToken,OST
0x2c537e5624e4af88a7ae4060c022609376c8d0eb,6,BiLira,TRYB
0x2c974b2d0ba1716e644c1fc59982a89ddd2ff724,18,Viberate,VIB
0x301c755ba0fca00b1923768fffb3df7f4e63af31,18,GlobalDigitalContent,GDC
0x309627af60f0926daa6041b8279484312f2bf060,18,USDB,USDB
0x30f271c9e86d2b7d00a6376cd96a1cfbd5f0b9b3,18,Decentr,DEC
0x3166c570935a7d8554c8f4ea792ff965d2efe1f2,18,QDAO,QDAO
0x340d2bde5eb28c1eed91b2f790723e3b160613b7,18,BLOCKv,VEE
0x37e8789bb9996cac9156cd5f5fd32599e6b91289,18,AidCoin,AID
0x3a92bd396aef82af98ebc0aa9030d25a23b11c6b,18,Tokenbox,TBX
0x3c6ff50c9ec362efa359317009428d52115fe643,18,PeerExNetwork,PERX
0x3d1ba9be9f66b8ee101911bc36d3fb562eac2244,18,Rivetz,RVT
0x408e41876cccdc0f92210600ef50372656052a38,18,Republic,REN
0x40fd72257597aa14c7231a7b1aaa29fce868f677,18,SoraToken,XOR
0x419d0d8bdd9af5e606ae2232ed285aff190e711b,8,FunFair,FUN
0x41ab1b6fcbb2fa9dced81acbdec13ea6315f2bf2,18,XinFin,XDCE
0x41e5560054824ea6b0732e656e3ad64e20e94e45,8,Civic,CVC
0x42d6622dece394b54999fbd73d108123806f6a18,18,SPANK,SPANK
0x43044f861ec040db59a7e324c40507addb673142,18,Cap,CAP
0xfef4185594457050cc9c23980d301908fe057bb1,18,VIDTDatalink,VIDT
0x4470bb87d77b963a013db939be332f927f2b992e,4,AdExNetworkV1,ADXv1
0x456ae45c0ce901e2e7c99c0718031cec0a7a59ff,18,VisionNetwork,VSN
0x49184e6dae8c8ecd89d8bdc1b950c597b8167c90,2,LIBERTAS,LIBERTAS
0x4946fcea7c692606e8908002e55a582af44ac121,18,FOAMToken,FOAM
0x4954db6391f4feb5468b6b943d4935353596aec9,18,USDQ,USDQ
0x4a220e6096b25eadb88358cb44068a3248254675,18,Quant,QNT
0x4a57e687b9126435a9b19e4a802113e266adebde,18,Flexacoin,FXC
0x4aac461c86abfa71e9d00d9a2cde8d74e4e1aeea,18,ZINC,ZINC
0x4c327471c44b2dacd6e90525f9d629bd2e4f662c,18,GHOST,GHOST
0x4cc19356f2d37338b9802aa8e8fc58b0373296e7,18,SelfKey,KEY
0x4d953cf077c0c95ba090226e59a18fcf97db44ec,18,MINISWAP,MINI
0x4da9b813057d04baef4e5800e36083717b4a0341,18,AaveInterestbearingTUSD,aTUSDv1
0x4e352cf164e64adcbad318c3a1e222e9eba4ce42,18,MCDEXToken,MCB
0x4f3afec4e5a3f2a6a1a411def7d7dfe50ee057bf,9,DigixGoldToken,DGX
0x4fabb145d64652a948d72533023f6e7a623c7c53,18,BinanceUSD,BUSD
0x5102791ca02fc3595398400bfe0e33d7b6c82267,18,LeadCoin,LDC
0x514910771af9ca656af840dff83e8264ecf986ca,18,ChainLink,LINK
0x543ff227f64aa17ea132bf9886cab5db55dcaddf,18,DAOStack,GEN
0x56d811088235f11c8920698a204a5010a788f4b3,18,bZxProtocolToken,BZRX
0x5732046a883704404f284ce41ffadd5b007fd668,18,Bluzelle,BLZ
0x57700244b20f84799a31c6c96dadff373ca9d6c5,18,TRUSTDAO,TRUST
0x58b6a8a3302369daec383334672404ee733ab239,18,LivepeerToken,LPT
0x595832f8fc6bf59c85c527fec3740a1b7a361269,6,PowerLedger,POWR
0x5adc961d6ac3f7062d2ea45fefb8d8167d44b190,18,Dether,DTH
0x5c872500c00565505f3624ab435c222e558e9ff8,18,CoTrader,COT
0x5caf454ba92e6f2c929df14667ee360ed9fd5b26,18,Dev,DEV
0x5d3a536e4d6dbd6114cc1ead35777bab948e3643,8,CompoundDai,cDAI
0x5d60d8d7ef6d37e16ebabc324de3be57f135e0bc,18,MyBit,MYB
0x607c794cda77efb21f8848b7910ecf27451ae842,18,DeFiPIEToken,PIE
0x607f4c5bb672230e8672085532f7e901544a7375,9,iExecRLC,RLC
0x6226caa1857afbc6dfb6ca66071eb241228031a1,18,Linkart,LAR
0x6251e725cd45fb1af99354035a414a2c0890b929,18,MixTrust,MXT
0x625ae63000f46200499120b906716420bd059240,18,AaveInterestbearingSUSD,aSUSDv1
0x667088b212ce3d06a1b553a7221e1fd19000d9af,18,Wings,WINGS
0x6710c63432a2de02954fc0f851db07146a6c0312,18,SmartMFG,MFG
0x6758b7d441a9739b98552b373703d8d3d14f9e62,18,POA,POA20
0x6810e776880c02933d47db1b9fc05908e5386b96,18,Gnosis,GNO
0x68d57c9a1c35f63e2c83ee8e49a64e9d70528d25,18,SirinLabs,SRN
0x6b785a0322126826d8226d77e173d75dafb84d11,18,BankrollVault,VLT
0x6b9f031d718dded0d681c20cb754f97b3bb81b78,18,Geeq,GEEQ
0x6ba460ab75cd2c56343b3517ffeba60748654d26,8,UpToken,UP
0x6c6ee5e31d828de241282b9606c8e98ea48526e2,18,HoloToken,HOT
0x6f87d756daf0503d08eb8993686c7fc01dc44fb1,18,UniTrade,TRADE
0x6fe56c0bcdd471359019fcbc48863d6c3e9d4f41,18,PropsToken,PROPS
0x71fc860f7d3a592a4a98740e39db31d25db65ae8,6,AaveInterestbearingUSDT,aUSDTv1
0x737f98ac8ca59f2c68ad658e3c3d8c8963e40a4c,18,Amon,AMN
0x744d70fdbe2ba4cf95131626614a1763df805b9e,18,Status,SNT
0x763186eb8d4856d536ed4478302971214febc6a9,18,BetterBetting,BETR
0x780116d91e5592e58a3b3c76a351571b39abcec6,15,Blockparty,BOXX
0x7b0c06043468469967dba22d1af33d77d44056c8,4,MorpheusNetwork,MRPH
0x7b123f53421b1bf8533339bfbdc7c98aa94163db,18,dfohub,buidl1
0x7c5a0ce9267ed19b22f8cae653f198e3e8daf098,18,Santiment,SAN
0x7de91b204c1c737bcee6f000aaa6569cf7061cb7,9,Robonomics,XRT
0x80fb784b7ed66730e8b1dbd9820afd29931aab03,18,EthLend,LEND
0x814e0908b12a99fecf5bc101bb5d0b8b5cdf7d26,18,MeasurableDataToken,MDT1
0x8207c1ffc5b6804f6024322ccf34f29c3541ae26,18,OriginProtocol,OGN
0x821144518dfe9e7b44fcf4d0824e15e8390d4637,18,ATISToken,ATIS
0x83984d6142934bb535793a82adb0a46ef0f66b6d,4,Remme,REM
0x83cee9e086a77e492ee0bb93c2b0437ad6fdeccc,18,Goldmint,MNTP
0x8400d94a5cb0fa0d041a3788e395285d61c9ee5e,8,UniBright,UBT
0x84ca8bc7997272c7cfb4d0cd3d55cd942b3c9419,18,DIAToken,DIA
0x8762db106b2c2a0bccb3a80d1ed41273552616e8,18,ReserveRights,RSR
0x89ab32156e46f46d02ade3fecbe5fc4243b9aaed,18,pNetworkToken,PNT
0x8ab7404063ec4dbcfd4598215992dc3f8ec853d7,18,Akropolis,AKRO
0x7b6f71c8b123b38aa8099e0098bec7fbc35b8a13,8,NerveNetwork,NVT
0x8ce9137d39326ad0cd6491fb5cc0cba0e089b6a9,18,Swipe,SXP
0x8e870d67f660d95d5be530380d0ec0bd388289e1,18,PaxDollar,USDP
0x8eb24319393716668d768dcec29356ae9cffe285,8,SingularityNETToken,AGI
0x8f8221afbb33998d8584a2b05749ba73c37a938a,18,Request,REQ
0x0258f474786ddfd37abce6df6bbb1dd5dfc4434a,8,OrionProtocol,ORN
0x9214ec02cb71cba0ada6896b8da260736a67ab10,18,REAL,REAL
0x93ed3fbe21207ec2e8f2d3c3de6e058cb73bc04d,18,Kleros,PNK
0x95172ccbe8344fecd73d0a30f54123652981bd6f,18,MeridianNetwork,LOCK
0x960b236a07cf122663c4303350609a66a7b288c0,18,AragonNetworkToken,ANTv1
0x967da4048cd07ab37855c090aaf366e4ce1b9f48,18,OceanToken,OCEAN
0x990f341946a3fdb507ae7e52d17851b87168017c,18,Strong,STRONG
0x9992ec3cf6a55b00978cddf2b27bc6882d88d1ec,18,Polymath,POLY
0x9ba00d6856a4edf4665bca2c2309936572473b7e,6,AaveInterestbearingUSDC,aUSDCv1
0x9cb2f26a23b8d89973f08c957c4d7cdf75cd341c,6,DigitalRand,DZAR
0x9f49ed43c90a540d1cf12f6170ace8d0b88a14e6,18,ETHRSI6040YieldII,ETHRSIAPY
0x9f8f72aa9304c8b593d555f12ef6589cc3a579a2,18,Maker,MKR
0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48,6,USDCoin,USDC
0xa15c7ebe1f07caf6bff097d8a589fb8ac49ae5b3,18,PundiX,NPXS
0xa1d0e215a23d7030842fc67ce582a6afa3ccab83,18,YFII.finance,YFII
0xa1d65e8fb6e87b60feccbc582f7f97804b725521,18,DXdao,DXD
0xa3bed4e1c75d00fa6f4e5e6922db7261b5e9acd2,18,Meta,MTA
0xa3d58c4e56fedcae3a7c43a725aee9a71f0ece4e,18,Metronome,MET
0xa462d0e6bb788c7807b1b1c96992ce1f7069e195,18,EQUUSMiningToken,EQMT
0xa4bdb11dc0a2bec88d24a3aa1e6bb17201112ebe,6,StableUSD,USDS
0xa4e8c3ec456107ea67d3075bf9e3df3a75823db0,18,LoomNetwork,LOOM
0xa64bd6c70cb9051f6a9ba1f163fdc07e0dfb5f84,18,AaveInterestbearingLINK,aLINKv1
0xa704fce7b309ec09df16e2f5ab8caf6fe8a4baa9,18,AgriChain,AGRI
0xaaaebe6fe48e54f431b0c390cfaf0b017d09d42d,4,Celsius,CEL
0xaaaf91d9b90df800df4f55c205fd6989c977e73a,8,Monolith,TKN
0xb0280743b44bf7db4b6be482b2ba7b75e5da096c,18,Transcodium,TNS
0xb056c38f6b7dc4064367403e26424cd2c60655e1,18,CEEKVR,CEEK
0xb3319f5d18bc0d84dd1b4825dcde5d5f7266d407,8,Compound0x,cZRX
0xb4272071ecadd69d933adcd19ca99fe80664fc08,18,CryptoFranc,XCHF
0xb4efd85c19999d84251304bda99e90b92300bd93,18,RocketPool,RPL
0xeca82185adce47f39c684352b0439f030f860318,18,Perlin,PERL
0xb4058411967d5046f3510943103805be61f0600e,18,STONK,STONK
0xb683d83a532e2cb7dfa5275eed3698436371cc9f,18,BTUProtocol,BTU
0xb6c4267c4877bb0d6b1685cfd85b0fbe82f105ec,18,Relevant,REL
0xba100000625a3754423978a60c9317c58a424e3d,18,Balancer,BAL
0xba11d00c5f74255f56a5e366f4f77f5a186d7f55,18,BandProtocol,BAND
0xbb1fa4fdeb3459733bf67ebc6f893003fa976a82,18,Bitnation,XPAT
0xbbbbca6a901c926f240b89eacb641d8aec7aeafd,18,Loopring,LRC
0xbc86727e770de68b1060c91f6bb6945c73e10388,18,InkProtocol,XNK
0xbd2949f67dcdc549c6ebe98696449fa79d988a9f,18,MeterGovernancemappedbyMeter.io,eMTRG
0xd9ec3ff1f8be459bb9369b4e79e9ebcf7141c093,18,KardiaChainToken,KAI
0xbe9375c6a420d2eeb258962efb95551a5b722803,18,KyberStormXToken,STMX
0xbf2179859fc6d5bee9bf9158632dc51678a4100e,18,Aelf,ELF
0xc00e94cb662c3520282e6f5717214004a7f26888,18,Compound,COMP
0xc011a73ee8576fb46f5e1c5751ca3b9fe0af2a6f,18,SynthetixNetworkToken,SNX
0x27702a26126e0b3702af63ee09ac4d1a084ef628,18,aleph.imv2,ALEPH
0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2,18,WrappedEther,WETH
0xc11b1268c1a384e55c48c2391d8d480264a3a7f4,8,CompoundWrappedBTC,cWBTC
0xc12d099be31567add4e4e4d0d45691c3f58f5663,18,Auctus,AUC
0xc27a2f05fa577a83ba0fdb4c38443c0718356501,18,LamdenTau,TAU
0xc28e931814725bbeb9e670676fabbcb694fe7df2,18,QuadrantProtocol,EQUAD
0xc3dd23a0a854b4f9ae80670f528094e9eb607ccb,18,Trendering,TRND
0xc5bbae50781be1669306b9e001eff57a2957b09d,5,Gifto,GTO
0xc75f15ada581219c95485c578e124df3985e4ce0,18,zzz.finance,ZZZ
0xcc4304a31d09258b0029ea7fe63d032f52e44efe,18,TrustSwapToken,SWAP
0xcc80c051057b774cd75067dc48f8987c4eb97a5e,18,EthfinexNectarToken,NEC
0xcd62b1c403fa761baadfc74c525ce2b51780b184,18,AragonNetworkJuror,ANJ
0xcee1d3c3a02267e37e6b373060f79d5d7b9e1669,18,yffi.finance,YFFI
0xcf8f9555d55ce45a3a33a81d6ef99a2a2e71dee2,18,CBIIndex7,CBIX7
0xd15ecdcf5ea68e3995b2d0527a0ae0a3258302f8,18,MachiXToken,MCX
0xd26114cd6ee289accf82350c8d8487fedb8a0c07,18,OmiseGO,OMG
0xd341d1680eeee3255b8c4c75bcce7eb57f144dae,18,onG.social,ONG
0xd46ba6d942050d489dbd938a2c909a5d5039a161,9,Ampleforth,AMPL
0xd4c435f5b09f855c3317c8524cb1f586e42795fa,18,Cindicator,CND
0xd559f20296ff4895da39b5bd9add54b442596a61,18,FintruX,FTX
0xd6f0bb2a45110f819e908a915237d652ac7c5aa8,18,DFOHub,BUIDL2
0xd7631787b4dcc87b1254cfd1e5ce48e96823dee8,8,Sociall,SCL
0xd8912c10681d8b21fd3742244f44658dba12264e,18,Pluton,PLU
0xdac17f958d2ee523a2206206994597c13d831ec7,6,TetherUSD,USDT
0xdb25f211ab05b1c97d595516f45794528a807ad8,2,STASISEURS,EURS
0xdd974d5c2e2928dea5f71b9825b8b646686bd200,18,KyberNetworkLegacy,KNCL
0xdf2c7238198ad8b389666574f2d8bc411a4b7428,18,Mainframe,MFT
0xdf574c24545e5ffecb9a659c229253d4111d87e1,8,HUSD,HUSD
0xdfe691f37b6264a90ff507eb359c45d55037951c,4,Karma,KARMA
0xe17f017475a709de58e976081eb916081ff4c9d5,9,RMPL,RMPL
0xe25b0bba01dc5630312b6a21927e578061a13f55,18,ShipChainSHIP,SHIP
0xe3818504c1b32bf1557b16c238b2e01fd3149c17,18,Pillar,PLR
0xe41d2489571d322189246dafa5ebde1f4699f498,18,0xProtocol,ZRX
0xe48972fcd82a274411c01834e2f031d4377fa2c0,18,2key.network,2KEY
0xeb4c2781e4eba804ce9a9803c67d0893436bb27d,8,renBTC,renBTC
0xec67005c4e498ec7f55e092bd1d35cbc47c91892,18,MelonToken,MLN
0xedd7c94fd7b4971b916d15067bc454b9e1bad980,18,Zippie,ZIPT
0xeeee2a622330e6d2036691e983dee87330588603,18,AskobarNetwork,ASKO
0xeeeeeeeee2af8d0e1940679860398308e0ef24d6,18,EthverseToken,ETHVEthverse
0xef9cd7882c067686691b6ff49e650b43afbbcc6b,18,FinNexus,FNX
0xefbd6d7def37ffae990503ecdb1291b2f7e38788,18,Evolution,EVO
0xf04a8ac553fcedb5ba99a64799155826c136b0be,18,Flixxo,FLIXX
0xf0fac7104aac544e4a7ce1a55adf2b5a25c65bd1,18,PampNetwork,PAMP
0xf1290473e210b2108a85237fbcd7b6eb42cc654f,18,HedgeTrade,HEDG
0xf29992d7b589a0a6bd2de7be29a97a6eb73eaf85,18,DMScript,DMST
0xf29e46887ffae92f1ff87dfe39713875da541373,18,UniCrypt,UNC
0xf2f9a7e93f845b3ce154efbeb64fb9346fcce509,18,UniPower,POWER
0xf433089366899d83a9f26a773d59ec7ecf30355e,8,Metal,MTL
0xf629cbd94d3791c9250152bd8dfbdf380e2a3b9c,18,EnjinCoin,ENJ
0xf650c3d88d12db855b8bf7d11be6c55a4e07dcc9,8,CompoundUSDT,cUSDT
0xf8e386eda857484f5a12e4b5daa9984e06e73705,18,Indorse,IND
0xf911a7ec46a2c6fa49193212fe4a2a9b95851c27,9,Antiample,XAMP
0xf970b8e36e23f7fc3fd752eea86f8be8d83375a6,18,Ripio,RCN
0xfc1e690f61efd961294b3e1ce3313fbd8aa4f85d,18,AaveInterestbearingDAI,aDAIv1
0x00006100f7090010005f1bd7ae6122c3c2cf0090,18,TrueAUD,TAUD
0x00000000441378008ea67f4284a57932b1c000a5,18,TrueGBP,TGBP
0x00000100f2a2bd000715001920eb70d229700085,18,TrueCAD,TCAD
0x0000852600ceb001e08e00bc008be620d60031f2,18,TrueHKD,THKD
0x0000000000b3f879cb30fe243b4dfee438691c04,2,Gastoken.io2,GST2
0x06af07097c9eeb7fd685c692751d5c66db49c215,18,ChaiToken,CHAI
0x6c8c6b02e7b2be14d4fa6022dfd6d75921d90e4e,8,CompoundBasicAttentionToken,cBAT
0xf5dce57282a584d2746faf1593d3121fcac444dc,8,CompoundSai,cSAI
0x4ddc2d193948926d02f9b1fe9e1daa0718270ed5,8,CompoundETH,cETH
0x39aa39c021dfbae8fac545936693ac917d5e7563,8,CompoundUSDCoin,cUSDC
0x158079ee67fce2f58472a96584a73c7ab9ac95c1,8,CompoundAugur,cREP
0x2af5d2ad76741191d15dfe7bf6ac92d4bd912ca3,18,BitfinexLEOToken,LEO
0x6f259637dcd74c767781e37bc6133cd6a68aa161,18,HuobiToken,HT
0x7d1afa7b718fb893db30a3abc0cfc608aacfebb0,18,MaticToken,MATIC
0x6b175474e89094c44da98b954eedeac495271d0f,18,DaiStablecoin,DAI
0x89d24a6b4ccb1b6faa2625fe562bdd9a23260359,18,SaiStablecoin,SAI
0x57ab1ec28d129707052df4df418d58a2d46d5f51,18,SynthsUSD,sUSD
0xeb269732ab75a6fd61ea60b06fe994cd32a83549,18,dForce,USDx
0x0000000000085d4780b73119b644ae5ecd22b376,18,TrueUSD,TUSD
0x679131f591b4f369acb8cd8c51e68596806c3916,18,TrustlinesNetworkToken,TLN
0x3a3a65aab0dd2a17e3f1947ba16138cd37d08c04,18,AaveInterestbearingETH,aETHv1
0xe1ba0fb44ccb0d11b80f92f4f8ed94ca3ff51d00,18,AaveInterestbearingBAT,aBATv1
0x9d91be44c06d373a8a226e1f3b146956083803eb,18,AaveInterestbearingKNC,aKNCv1
0x7d2d3688df45ce7c552e19c27e007673da9204b8,18,AaveInterestbearingLEND,aLENDv1
0x6fce4a401b6b80ace52baaefe4421bd188e76f6f,18,AaveInterestbearingMANA,aMANAv1
0x7deb5e830be29f91e298ba5ff1356bb7f8146998,18,AaveInterestbearingMKR,aMKRv1
0x71010a9d003445ac60c4e6a7017c1e89a477b438,18,AaveInterestbearingREP,aREPv1
0x328c4c80bc7aca0834db37e6600a6c49e12da4de,18,AaveInterestbearingSNX,aSNXv1
0xfc4b8ed459e00e5400be803a9bb3954234fd50e3,8,AaveInterestbearingWBTC,aWBTCv1
0x6fb0855c404e09c47c3fbca25f08d4e41f9f062f,18,AaveInterestbearingZRX,aZRXv1
0x66fd97a78d8854fec445cd1c80a07896b0b4851f,18,LunchMoney,LMY
0x16de59092dae5ccf4a1e6439d611fd0653f0bd01,18,iearnDAIv2,yDAIv2
0xc2cb1040220768554cf699b0d863a3cd4324ce32,18,iearnDAIv3,yDAIv3
0x04bc0ab673d88ae9dbc9da2380cb6b79c4bca9ae,18,iearnBUSD,yBUSD
0x04aa51bbcb46541455ccf1b8bef2ebc5d3787ec9,8,iearnWBTC,yBTC
0x73a052500105205d34daf004eab301916da8190f,18,iearnTUSD,yTUSDiearn
0x83f798e925bcd4017eb265844fddabb448f1707d,6,iearnUSDTv2,yUSDTv2
0xe6354ed5bc4b393a5aad09f21c46e101e692d447,6,iearnUSDTv3,yUSDTv3
0xd6ad7a6750a7593e092a9b218d66c0a814a3436e,6,iearnUSDCv2,yUSDCv2
0x26ea744e5b887e5205727f55dfbe8685e3b21951,6,iearnUSDCv3,yUSDCv3
0xf61718057901f84c4eec4339ef8f0d86d2b45600,18,iearnSUSD,ySUSD
0xacfa209fb73bf3dd5bbfb1101b9bc999c49062a5,18,BlockchainCertifiedDataToken,BCDT
0x4de2573e27e648607b50e1cfff921a33e4a34405,18,LendroidSupportToken,LST
0x630d98424efe0ea27fb1b3ab7741907dffeaad78,8,PEAKDEFI,PEAK
0xd56dac73a4d6766464b38ec6d91eb45ce7457c44,18,Panvalapan,PAN
0x056fd409e1d7a124bd7017459dfea2f387b6d5cd,2,Geminidollar,GUSD
0x6ee0f7bb50a54ab5253da0667b0dc2ee526c30a8,18,AaveInterestbearingBinanceUSD,aBUSDv1
0xd7efb00d12c2c13131fd319336fdf952525da2af,4,Proton,XPR
0x85eee30c52b0b379b046fb0f85f4f3dc3009afec,18,KEEPToken,KEEP
0x1c5db575e2ff833e46a2e9864c22f4b22e0b37c2,8,renZEC,renZEC
0x459086f2376525bdceba5bdda135e4e9d3fef5bf,8,renBCH,renBCH
0x8daebade922df735c38c80c7ebd708af50815faa,18,tBTC,tBTC
0x0316eb71485b0ab14103307bf65a021042c6d380,18,HuobiBTC,HBTC
0x3a9fff453d50d4ac52a6890647b823379ba36b9e,18,Shuffle.MonsterV3,SHUF
0xc0f9bd5fa5698b6505f643900ffa515ea5df54a9,18,DONUT,DONUT
0x0000000000004946c0e9f43f4dee607b0ef1fa1c,0,ChiGastokenby1inch,CHI
0xd6a55c63865affd67e2fb9f284f87b7a9e5ff3bd,18,Switch,ESH
0x1fc31488f28ac846588ffa201cde0669168471bd,2,UAX,UAX
0x45804880de22913dafe09f4980848ece6ecbaf78,18,PaxosGold,PAXG
0x035df12e0f3ac6671126525f1015e47d79dfeddf,18,0xMonero,0xMR
0xb64ef51c888972c908cfacf59b47c1afbc0ab8ac,8,Storj,STORJ
0x4156d3342d5c385a87d264f90653733592000581,8,Salt,SALT
0xdf5e0e81dff6faf3a7e52ba697820c5e32d806a8,18,Curve.fiiearnpooltoken,yCurve
0xfca59cd816ab1ead66534d82bc21e7515ce441cf,18,Rarible,RARI
0xea5f88e54d982cbb0c441cde4e79bc305e5b43bc,18,ParetoNetworkToken,PARETO
0x9fbfed658919a896b5dc7b00456ce22d780f9b65,18,PlutusDeFi,PLT
0x5228a22e72ccc52d415ecfd199f99d0665e7733b,18,pTokensBTC,pBTC
0x476c5e26a75bd202a9683ffd34359c0cc15be0ff,6,Serum,SRM
0xc813ea5e3b48bebeedb796ab42a30c5599b01740,4,Autonio,NIOX
0xa7de087329bfcda5639247f96140f9dabe3deed1,18,Statera,STA
0xdc5864ede28bd4405aa04d93e05a0531797d9d59,6,Falcon,FNT
0x0aacfbec6a24756c20d41914f2caba817c0d8521,18,YAM,YAM
0xade00c28244d5ce17d72e40330b1c318cd12b7c3,18,AdExNetwork,ADX
0xd533a949740bb3306d119cc777fa900ba034cd52,18,CurveDAOToken,CRV
0x9469d013805bffb7d3debe5e7839237e535ec483,18,DarwiniaNetworkNativeToken,RING
0x2baecdf43734f22fd5c152db08e3c27233f0c7d2,18,OMToken,OMv1
0x491604c0fdf08347dd1fa4ee062a822a5dd06b5d,18,CartesiToken,CTSI
0x0ff6ffcfda92c53f615a4a75d982f399c989366b,18,Unilayer,LAYER
0xd5525d397898e5502075ea5e830d8914f6f0affe,8,MEME,MEME
0x68a118ef45063051eac49c7e647ce5ace48a68a5,18,BASED,BASED
0xaa7a9ca87d3694b5755f213b5d04094b8d0f0a6f,18,Trace,TRAC
0x4dfd148b532e934a2a26ea65689cf6268753e130,18,GovernancetokenMonolithosDAO,MDT2
0xaba8cac6866b83ae4eec97dd07ed254282f6ad8a,24,YAMv2,YAMv2
0x8a9c67fee641579deba04928c4bc45f66e26343a,18,JarvisRewardToken,JRT
0x45f24baeef268bb6d63aee5129015d69702bcdfa,18,YFValue,YFV
0x674c6ad92fd080e4004b2312b45f796a192d27a0,18,Neutrino,USDN
0x362bc847a3a9637d3af6624eec853618a43ed7d2,18,ParsiqToken,PRQ
0x0e29e5abbb5fd88e28b2d355774e73bd47de3bcd,18,HakkaFinance,HAKKA
0x08ad83d779bdf2bbe1ad9cc0f78aa0d24ab97802,18,RobonomicsWebServicesV1,RWS
0x09e64c2b61a5f1690ee6fbed9baf5d6990f8dfd0,18,Growth,GRO
0x38e4adb44ef08f22f5b5b76a8f0c2d0dcbe7dca1,18,ConcentratedVotingPower,CVP
0x4fe5851c9af07df9e5ad8217afae1ea72737ebda,18,OpenPredictToken,OPT
0x6b3595068778dd592e39a122f4f5a5cf09c90fe2,18,SushiToken,SUSHI
0x3affcca64c2a6f4e3b6bd9c64cd2c969efd1ecbe,18,DSLA,DSLA
0xba21ef4c9f433ede00badefcc2754b8e74bd538a,18,Swapfolio,SWFL
0xfffffffff15abf397da76f1dcc1a1604f45126db,18,FalconSwapToken,FSW
0x94d863173ee77439e4292284ff13fad54b3ba182,18,AkropolisDelphi,ADEL
0xb8baa0e4287890a5f79863ab62b7f175cecbd433,18,SwerveDAOToken,SWRV
0x8a6f3bf52a26a21531514e23016eeae8ba7e7018,8,Multiplier,MXX
0x103c3a209da59d3e7c4a89307e66521e081cfdf0,18,GenesisVision,GVT
0x892b14321a4fcba80669ae30bd0cd99a7ecf6ac0,8,CreamCream,crCream
0xe1237aa7f535b0cc33fd973d66cbf830354d16c7,18,yearnWrappedEther,yWETH
0x5dbcf33d8c2e976c6b560249878e6f1491bca25c,18,yearnCurve.fiyDAIyUSDCyUSDTyTUSD,yUSD
0x50026ad58b338cf3eccc2b422deb8faa725f377f,8,1Step.finance,STEP
0x1aa61c196e76805fcbe394ea00e4ffced24fc469,18,yieldfarming.insure,SAFE
0x556148562d5ddeb72545d7ec4b3ec8edc8f55ba7,18,PredixNetwork,PRDX
0x1494ca1f11d487c2bbe4543e90080aeba4ba3c2b,18,DefiPulseIndex,DPI
0x3f382dbd960e3a9bbceae22651e88158d2791550,18,AavegotchiGHSTToken,GHST
0x1f9840a85d5af5bf1d1762f925bdaddc4201f984,18,Uniswap,UNI
0x2e93fe8d550a7b7e7b2e561cd45cebccbaa79358,5,GXChainCoreAsset,GXC
0x429881672b9ae42b8eba0e26cd9c73711b891ca5,18,PickleToken,PICKLE
0xf8c3527cc04340b208c854e985240c02f7b7793f,18,FrontierToken,FRONT
0xca1207647ff814039530d7d35df0e1dd2e91fa84,18,dHedgeDAOToken,DHT
0xa0246c9032bc3a600820415ae600c6388619a14d,18,FARMRewardToken,FARM
0x488e0369f9bc5c40c002ea7c1fe4fd01a198801c,18,Golff.finance,GOF
0xecbf566944250dde88322581024e611419715f7a,9,xBTC,xBTC
0x2a8e1e676ec238d8a992307b495b45b3feaa5e86,18,OriginDollar,OUSD
0x1d09144f3479bb805cb7c92346987420bcbdc10c,18,CreamYUSD,cyUSD
0x7fc66500c84a76ad7e9c93437bfc5ac33e2ddae9,18,AaveToken,AAVE
0x43dfc4159d86f3a37a5a4b3d4580b888ad7d4ddd,18,DODObird,DODO
0x250a3500f48666561386832f1f1f1019b89a2699,18,SAFE2,SAFE2
0x6006fc2a849fedaba8330ce36f5133de01f96189,18,SHAKEtokenbySpaceSwapv2,SHAKE
0x80c8c3dcfb854f9542567c8dac3f44d709ebc1de,18,MilkyWayTokenbySpaceSwapv2,MILK2
0x62359ed7505efc61ff1d56fef82158ccaffa23d7,18,cVault.finance,CORE
0xbc396689893d065f41bc2c6ecbee5e0085233447,18,Perpetual,PERP
0x49e833337ece7afe375e44f4e3e8481029218e5c,18,ValueLiquidity,VALUE
0x4b7ad3a56810032782afce12d7d27122bdb96eff,8,Sparkle,SPRKL
0x1c48f86ae57291f7686349f12601910bd8d470bb,18,USDK,USDK
0x87edffde3e14c7a66c9b9724747a1c5696b742e6,18,SwagToken,SWAG
0xad6a626ae2b43dcb1b39430ce496d2fa0365ba9c,18,PieDAODEFISmallCap,DEFIS
0xad32a8e6220741182940c5abf610bde99e737b2d,18,PieDAODOUGHv2,DOUGH
0x3f09400313e83d53366147e3ea0e4e2279d80850,18,KUSH.FINANCE,kSEED
0x70e36f6bf80a52b3b46b3af8e106cc0ed743e8e4,8,CompoundCollateral,cCOMP
0x35a18000230da775cac24873d00ff85bccded550,8,CompoundUniswap,cUNI
0x70da48f4b7e83c386ef983d4cef4e58c2c09d8ac,8,QurasToken,XQC
0xe2f2a5c287993345a840db3b0845fbc70f5935a5,18,mStableUSD,mUSD
0x2edf094db69d6dcd487f1b3db9febe2eec0dd4c5,18,ZeroSwapToken,ZEE
0x584bc13c7d411c00c01a62e8019472de68768430,18,Hegic,HEGIC
0x054f76beed60ab6dbeb23502178c52d6c5debe40,18,DeFiner,FIN
0xcbd55d4ffc43467142761a764763652b48b969ff,18,AstroTools.io,ASTRO
0xff20817765cb7f73d4bde2e66e067e58d11095c2,18,Amp,AMP
0x0391d2021f89dc339f60fff84546ea23e337750f,18,BarnBridgeGovernanceToken,BOND
0xa117000000f279d81a1d3cc75430faa017fa5a2e,18,AragonNetworkToken,ANT
0x970b9bb2c0444f5e81e9d0efb84c8ccdcdcaf84d,18,FuseToken,FUSE
0x36f3fd68e7325a35eb768f1aedaae9ea0689d723,18,EmptySetDollar,ESD
0xa9af25a2d43eb04723a6ec0749913519cf347842,6,TealCoin,TEAC
0x1ceb5cb57c4d4e2b2433641b95dd330a33185a44,18,Keep3rV1,KP3R
0x5bc25f649fc4e26069ddf4cf4010f9f706c23831,18,DefiDollar,DUSD
0xb705268213d593b8fd88d3fdeff93aff5cbdcfae,18,IDEXToken,IDEX
0x2e2364966267b5d7d2ce6cd9a9b5bd19d9c7c6a9,18,VoiceToken,VOICE
0xeef9f339514298c6a857efcfc1a762af84438dee,18,HermezNetworkToken,HEZ
0xea319e87cf06203dae107dd8e5672175e3ee976c,18,SURF.Finance,SURF
0x3383c5a8969dc413bfddc9656eb80a1408e4ba20,18,WrappedANATHA,wANATHA
0x18aaa7115705e8be94bffebde57af9bfc265b998,18,Audius,AUDIO
0xa665fed1b0c9da00e91ca582f77df36e325048c5,18,yfarm.finance,YFM
0xdacd69347de42babfaecd09dc88958378780fb62,0,AtariToken,ATRI
0x0954906da0bf32d5479e25f46056d22f08464cab,18,Index,INDEX
0xc57d533c50bc22247d49a368880fb49a1caa39f7,18,PowerTradeFuelToken,PTF
0x20c36f062a31865bed8a5b1e512d9a1a20aa333a,18,DefiDollarDAO,DFD
0x95a4492f028aa1fd432ea71146b433e7b4446611,18,APYGovernanceToken,APY
0xbea98c05eeae2f3bc8c3565db7551eb738c8ccab,18,Geyser,GYSR
0xa89ac6e529acf391cfbbd377f3ac9d93eae9664e,18,Keep4r,KP4R
0xbb0e17ef65f82ab018d8edd776e8dd940327b28b,18,AxieInfinityShard,AXS
0x297e4e5e59ad72b1b0a2fd446929e76117be0e0a,18,ValorToken,VALOR
0x00a8b738e453ffd858a7edf03bccfe20412f0eb0,18,AllianceBlockToken,ALBT
0x5beabaebb3146685dd74176f68a0721f91297d37,18,BounceToken,BOT
0x05d3606d5c81eb9b7b18530995ec9b29da05faba,18,TomoChain,TOMOE
0xb1f66997a5760428d3a87d68b90bfe0ae64121cc,18,LuaToken,LUA
0x39eae99e685906ff1c11a962a743440d0a1a6e09,18,Holyheld,HOLY
0x9d942bd31169ed25a1ca78c776dab92de104e50e,18,DeFISocks2020,DEFISOCKS
0xf4cd3d3fda8d7fd6c5a500203e38640a70bf9577,18,YfDAI.finance,YfDAI
0x83e6f1e41cdd28eaceb20cb649155049fac3d5aa,18,PolkastarterToken,POLS
0xaf9f549774ecedbd0966c52f250acc548d3f36e5,18,RioFuelToken,RFuel
0x0202be363b8a4820f3f4de7faf5224ff05943ab1,18,UniLendFinanceToken,UFT
0xae697f994fc5ebc000f8e22ebffee04612f98a0d,18,LGCYNetwork,LGCY
0x9d47894f8becb68b9cf3428d256311affe8b068b,18,ROPE,ROPE
0x72f020f8f3e8fd9382705723cd26380f8d0c66bb,18,PLOT,PLOT
0x6a7ef4998eb9d0f706238756949f311a59e05745,18,KenysiansNetwork,KEN
0x20945ca1df56d237fd40036d47e866c7dccd2114,18,NsureNetworkToken,Nsure
0x485d17a6f1b8780392d53d64751824253011a260,8,ChronoTechToken,TIME
0x12e51e77daaa58aa0e9247db7510ea4b46f9bead,18,AaveInterestbearingYFI,aYFIv1
0xba3d9687cf50fe253cd2e1cfeede1d6787344ed5,18,AaveInterestbearingAaveToken,aAAVEv1
0xb124541127a0a657f056d9dd06188c4f1b0e5aab,18,AaveInterestbearingUniswap,aUNIv1
0x712db54daa836b53ef1ecbb9c6ba3b9efb073f40,18,AaveInterestbearingENJ,aENJv1
0xb753428af26e81097e7fd17f40c88aaa3e04902c,18,Spice,SFI
0x8888801af4d980682e47f1a9036e589479e835c5,18,88mph.app,MPH
0x5d8d9f5b96f4438195be9b99eee6118ed4304286,18,CoverProtocol,COVERv1
0xc3eb2622190c57429aac3901808994443b64b466,18,OROToken,ORO
0x6468e79a80c0eab0f9a2b574c8d5bc374af59414,18,E-RADIX,eXRD
0x3e780920601d61cedb860fe9c4a90c9ea6a35e78,18,BoostedFinance,BOOST
0x431ad2ff6a9c365805ebad47ee021148d6f7dbe0,18,dForce,DF
0x1695936d6a953df699c38ca21c2140d497c08bd9,18,SynLev,SYN
0x3db6ba6ab6f95efed1a6e794cad492faaabf294d,8,LTONetworkToken,LTO
0xf5238462e7235c7b62811567e63dd17d12c2eaa0,8,CACHEGold,CGT
0x4fe83213d56308330ec302a8bd641f1d0113a4cc,18,NuCypher,NU
0x9e70240d2a8a30a592d3f076441c4f303b26de12,8,WrappedOCT,OCT
0x6e0dade58d2d89ebbe7afc384e3e4f15b70b14d8,18,QuiverX,QRX
0x05079687d35b93538cbd59fe5596380cae9054a9,18,BitSong,BTSG
0x355c665e101b9da58704a8fddb5feef210ef20c0,18,dForceGOLDx,GOLDx
0x69948cc03f478b95283f7dbf1ce764d0fc7ec54c,18,AaveInterestbearingREN,aRENv1
0xe88f8313e61a97cec1871ee37fbbe2a8bf3ed1e4,18,SoraValidatorToken,VAL
0x0b38210ea11411557c13457d4da7dc6ea731b88a,18,API3,API3
0x3449fc1cd036255ba1eb19d65ff4ba2b8903a69a,18,BAC,BAC
0xa7ed29b253d8b4e3109ce07c80fc570f81b63696,18,BAS,BAS
0x26607ac599266b21d13c7acf7942c7701a8b699c,18,PowerIndexPoolToken,PIPT
0x3218a02f8f8b5c3894ce30eb255f10bcba13e654,18,MegaCryptoPolisMEGATokenMEGA,MEGA
0x07150e919b4de5fd6a63de1f9384828396f25fdc,9,BaseProtocol,BASE
0x91dfbee3965baaee32784c2d546b7a0c62f268c9,18,Bondly,BONDLY
0x9d79d5b61de59d882ce90125b18f74af650acb93,6,NeutrinoSystemBaseToken,NSBT
0xb62132e35a6c13ee1ee0f84dc5d40bad8d815206,18,Nexo,NEXO
0xffc97d72e13e01096502cb8eb52dee56f74dad7b,18,AaveinterestbearingAAVE,aAAVE
0x05ec93c0365baaeabf7aeffb0972ea7ecdd39cf1,18,AaveinterestbearingBAT,aBAT
0xa361718326c15715591c299427c62086f69923d9,18,AaveinterestbearingBUSD,aBUSD
0x028171bca77440897b824ca71d1c56cac55b68a3,18,AaveinterestbearingDAI,aDAI
0xac6df26a590f08dcc95d5a4705ae8abbc88509ef,18,AaveinterestbearingENJ,aENJ
0x39c6b3e42d6a679d7d776778fe880bc9487c2eda,18,AaveinterestbearingKNC,aKNC
0xa06bc25b5805d5f8d82847d191cb4af5a3e873e0,18,AaveinterestbearingLINK,aLINK
0xa685a61171bb30d4072b338c80cb7b2c865c873e,18,AaveinterestbearingMANA,aMANA
0xc713e5e149d5d0715dcd1c156a020976e7e56b88,18,AaveinterestbearingMKR,aMKR
0xcc12abe4ff81c9378d670de1b57f8e0dd228d77a,18,AaveInterestbearingREN,aREN
0x35f6b052c598d933d69a4eec4d04c73a191fe6c2,18,AaveinterestbearingSNX,aSNX
0x6c5024cd4f8a59110119c56f8933403a539555eb,18,AaveinterestbearingSUSD,aSUSD
0x101cc05f4a51c0319f570d5e146a8c625198e636,18,AaveinterestbearingTUSD,aTUSD
0xb9d7cb55f463405cdfbe4e90a6d2df01c2b92bf1,18,AaveinterestbearingUNI,aUNI
0xbcca60bb61934080951369a648fb03df4f96263c,6,AaveinterestbearingUSDC,aUSDC
0x3ed3b47dd13ec9a98b44e6204a523e766b225811,6,AaveinterestbearingUSDT,aUSDT
0x9ff58f4ffb29fa2266ab25e75e2a8b3503311656,8,AaveinterestbearingWBTC,aWBTC
0x030ba81f1c18d280636f32af80b9aad02cf0854e,18,AaveinterestbearingWETH,aWETH
0x5165d24277cd063f5ac44efd447b27025e888f37,18,AaveinterestbearingYFI,aYFI
0xdf7ff54aacacbff42dfe29dd6144a69b629f8c9e,18,AaveinterestbearingZRX,aZRX
0xf644d4477cd8db7791cea3013cb053b3fec4beb3,8,CryptoenterLIONtoken,LION
0xa8e7ad77c60ee6f30bac54e2e7c0617bd7b5a03e,18,zLOT,zLOT
0x191557728e4d8caa4ac94f86af842148c0fa8f7e,8,ECOTOKEN,ECO
0xdc9ac3c20d1ed0b540df9b1fedc10039df13f99c,18,UtrustToken,UTK
0x3472a5a71965499acd81997a54bba8d852c6e53d,18,Badger,BADGER
0xc944e90c64b2c07662a292be6244bdf05cda44a7,18,GraphToken,GRT
0x5a98fcbea516cf06857215779fd812ca3bef1b32,18,LidoDAOToken,LDO
0x77777feddddffc19ff86db637967013e6c6a116c,18,TornadoCash,TORN
0xae7ab96520de3a18e5e111b5eaab095312d7fe84,18,stETH,stETH
0xb4d930279552397bba2ee473229f89ec245bc365,18,MahaDAO,MAHA
0x57b946008913b82e4df85f501cbaed910e58d26c,18,MarlinPOND,POND
0x3593d125a4f7849a1b059e64f4517a86dd60c95d,18,MANTRADAO,OMv2
0x3432b6a60d23ca0dfca7761b7ab56459d9c964d0,18,FraxShare,FXS
0x1fdab294eda5112b7d066ed8f2e4e562d5bcc664,18,Spice,SPICE
0x111111111117dc0aa78b770fa6a738034120c302,18,1INCHToken,1INCH
0x054d64b73d3d8a21af3d764efd76bcaa774f3bb2,18,Plasma,PPAY
0x368b3a58b5f49392e5c9e4c998cb0bb966752e51,18,MIC,MIC
0x4b4d2e899658fb59b1d518b68fe836b100ee8958,18,MIS,MIS
0x03066da434e5264ef0b32f787923f974a5726fdc,18,BCS,BCS
0xee573a945b01b788b9287ce062a0cfc15be9fd86,18,Exeedme,XED
0x34950ff2b487d9e5282c5ab342d08a2f712eb79f,18,EFFORCEIEO,WOZX
0xbd2f0cd039e0bfcf88901c98c0bfac5ab27566e3,18,DynamicSetDollar,DSD
0xc770eefad204b5180df6a14ee197d99d808ee52d,18,FOX,FOX
0x4688a8b1f292fdab17e9a90c8bc379dc1dbd8713,18,CoverProtocolGovernanceToken,COVER
0x66a0f676479cee1d7373f3dc2e2952778bff5bd6,18,WiseToken,WISE
0x539f3615c1dbafa0d008d87504667458acbd16fa,18,FERA,FERA
0xffffffff2ba8f66d4e51811c5190992176930278,18,Furucombo,COMBO
0x2b4200a8d373d484993c37d63ee14aee0096cd12,18,USDFreeLiquidity,USDFL
0xaea46a60368a7bd060eec7df8cba43b7ef41ad85,18,Fetch,FET
0x6c5ba91642f10282b576d91922ae6448c9d52f4e,18,Phala,PHA
0xa8b12cc90abf65191532a12bb5394a714a46d358,18,POWBTC-35WT,pBTC35A
0x853d955acef822db058eb8505911ed77f175b99e,18,Frax,FRAX
0xe28b3b32b6c345a34ff64674606124dd5aceca30,18,InjectiveToken,INJ
0x0a50c93c762fdd6e56d86215c24aaad43ab629aa,8,LGOToken,LGO
0xf94b5c5651c888d928439ab6514b93944eee6f48,18,Yield,YLDAPP
0x63b4f3e3fa4e438698ce330e365e831f7ccd1ef4,18,CyberFiToken,CFi
0xd291e7a03283640fdc51b121ac401383a46cc623,18,RariGovernanceToken,RGT
0xfa5047c9c78b8877af97bdcb85db743fd7313d4a,18,ROOK,ROOK
0xae1eaae3f627aaca434127644371b67b18444051,8,YOP,YOP
0x87d73e916d7057945c9bcd8cdd94e42a6f47f776,18,NFTX,NFTX
0xa4eed63db85311e22df4473f87ccfc3dadcfa3e3,18,Rubic,RBC
0x73968b9a57c6e53d41345fd57a6e6ae27d6cdb2f,18,StakeDAOToken,SDT
0x3c4b6e6e1ea3d4863700d7f76b36b7f3d3f13e3d,8,VoyagerToken,VGX
0x9248c485b0b80f76da451f167a8db30f33c70907,18,Debase,DEBASE
0x8290333cef9e6d528dd5618fb97a76f268f3edd4,18,AnkrNetwork,ANKR
0xc719d010b63e5bbf2c0551872cd5316ed26acd83,18,DecentralizedInsuranceProtocol,DIPInsurance
0x3155ba85d5f96b2d030a4966af206230e46849cb,18,THORChainETH.RUNE,RUNE
0x374cb8c27130e2c9e04f44303f3c8351b9de61c1,18,BaoToken,BAO
0xfe3e6a25e6b192a42a44ecddcd13796471735acf,18,Reef.finance,REEF
0x4c19596f5aaff459fa38b0f7ed92f11ae6543784,8,TrustToken,TRU
0x86772b1409b61c639eaac9ba0acfbb6e238e5f83,18,Indexed,NDX
0x67c597624b17b16fb77959217360b7cd18284253,9,Benchmark,MARK
0xf0939011a9bb95c3b791f0cb546377ed2693a574,18,Zero.ExchangeToken,ZERO
0x7240ac91f01233baaf8b064248e80feaa5912ba3,18,Octo.fi,OCTO
0x4c11249814f11b9346808179cf06e71ac328c1b5,18,OraichainToken,ORAI
0xfbeea1c75e4c4465cb2fccc9c6d6afe984558e20,18,DuckDaoDime,DDIM
0x70401dfd142a16dc7031c56e862fc88cb9537ce0,18,Bird.Money,BIRD
0xcae72a7a0fd9046cf6b165ca54c9e3a3872109e0,18,AnRKeyX,ANRX
0x9b02dd390a603add5c07f9fd9175b7dabe8d63b7,18,Shopping.io,SPIShopping
0x86ed939b500e121c0c5f493f399084db596dad20,18,SpaceChainV2,SPC
0x33d0568941c0c64ff7e0fb4fba0b11bd37deed9f,18,RAMPDEFI,RAMP
0xb987d48ed8f2c468d52d6405624eadba5e76d723,18,StabilizeToken,STBZ
0x159751323a9e0415dd3d6d42a1212fe9f4a0848c,18,INFI,INFI
0xa0b73e1ff0b80914ab6fe0444e65848c4c34450b,8,CRO,CRO
0xa8b919680258d369114910511cc87595aec0be6d,18,LUKSOToken,LYXe
0xd084b83c305dafd76ae3e1b4e1f1fe2ecccb3988,18,TerraVirtuaKolect,TVK
0x798d1be841a82a273720ce31c822c61a67a601c3,9,Digg,DIGG
0xffed56a180f23fd32bc6a1d8d3c09c283ab594a8,18,FL,FL
0x5580ab97f226c324c671746a1787524aef42e415,18,JUL,JUL
0xa1faa113cbe53436df28ff0aee54275c13b40975,18,AlphaToken,ALPHA
0x817bbdbc3e8a1204f3691d14bb44992841e3db35,18,CudosToken,CUDOS
0xde4ee8057785a7e8e800db58f9784845a5c2cbd6,18,Dexe,DEXE
0x3845badade8e6dff049820680d1f14bd3903a5d0,18,SAND,SAND
0x3c03b4ec9477809072ff9cc9292c9b25d4a8e6c6,18,PolkaCover,CVR
0x79256db1bdb6259315a1a3d7dd237f69cadfd8fc,18,Typhoon,PHOON
0xe5caef4af8780e59df925470b050fb23c43ca68c,6,FerrumNetworkToken,FRM
0x298d492e8c1d909d3f63bc4a36c66c64acb3d695,18,PolkaBridge,PBR
0xfe9a29ab92522d14fc65880d817214261d8479ae,18,SnowSwap,SNOW
0x220b71671b649c03714da9c621285943f3cbcdc6,18,TosDis,DIS
0x69a95185ee2a045cdc4bcd1b1df10710395e4e23,18,PoolzFinance,POOLZ
0xe4815ae53b124e7263f08dcdbbb757d41ed658c6,18,Zks,ZKS
0x1337def16f9b486faed0293eb623dc8395dfe46a,18,Armor,ARMOR
0x1337def18c680af1f9f45cbcab6309562975b1dd,18,ArmorNXM,arNXM
0x888888888889c00c67689029d7856aac1065ec11,18,OpiumGovernanceToken,OPIUM
0x3fa729b4548becbad4eab6ef18413470e6d5324c,18,Holyheld,HH
0xb4bebd34f6daafd808f73de0d10235a92fbb6c3d,18,YearnEcosystemTokenIndex,YETI
0xfa2562da1bba7b954f26c74725df51fb62646313,18,ASSYIndex,ASSY
0x961c8c0b1aad0c0b10a51fef6a867e3091bcef17,18,DeFiYieldProtocol,DYP
0x7a3eb5727e33a8000030718e82481f33bc20c4e3,8,OpiumcDAIJuniorTranche,OPIUMLPa55e
0x2994529c0652d127b7842094103715ec5299bbed,18,yearnCurve.fiyDAIyUSDCyUSDTyBUSD,ybCRV
0xacd43e627e64355f1861cec6d3a6688b31a6f952,18,yearnDaiStablecoin,yDAI
0x597ad1e0c13bfe8025993d9e79c69e1c0233522e,6,yearnUSDC,yUSDC
0x2f08119c6f07c006695e079aafc638b8789faf18,6,yearnTetherUSD,yUSDT
0x37d19d1c4e1fa9dc47bd1ea12f742a0887eda74a,18,yearnTrueUSD,yTUSDyearn
0x7ff566e1d69deff32a7b244ae7276b9f90e9d0f6,18,yearnCurve.firenBTCwBTCsBTC,ycrvRenWSBTC
0x9aa8f427a17d6b0d91b6262989edc7d45d6aedf8,18,FARM_crvRenWBTC,FcrvRenWBTC
0x898bad2774eb97cf6b94605677f43b41871410b1,18,validator-Eth2,vETH2
0xe95a203b1a91a908f9b9ce46459d101078c2c3cb,18,AnkrEth2RewardBearingBond,aETHc
0xcbc1065255cbc3ab41a6868c22d1f1c573ab89fd,18,CreamETH2,CRETH2
0x4e15361fd6b4bb609fa63c81a2be19d873717870,18,FantomToken,FTM
0x8642a849d0dcb7a15a974794668adcfbe4794b56,18,Prosper,PROS
0xc888a0ab4831a29e6ca432babf52e353d23db3c2,18,FastSwapToken,FAST
0xa1afffe3f4d611d252010e3eaf6f4d77088b0cd7,9,reflect.finance,RFI
0xa47c8bf37f92abed4a126bda807a7b7498661acd,18,WrappedUSTToken,UST
0x3832d2f059e55934220881f831be501d180671a7,8,renDOGE,renDOGE
0x09a3ecafa817268f77be1283176b946c4ff2e608,18,WrappedMIRToken,MIR
0x0d7dea5922535087078dd3d7c554ea9f2655d4cb,18,BerezkaFLEX,FLEX
0xdc76450fd7e6352733fe8550efabff750b2de0e3,18,BerezkaDynamic,DYNA
0x7a8d51b82b36fa5b50fb77001d6d189e920d2f75,18,WrappedOpiumGovernanceToken,wOPIUM
0xefc1c73a3d8728dc4cf2a18ac5705fe93e5914ac,18,Metric.exchange,METRIC
0x1d37986f252d0e349522ea6c3b98cb935495e63e,18,ChartEx,CHART
0x725c263e32c72ddc3a19bea12c5a0479a81ee688,18,BridgeMutual,BMI
0xc666081073e8dff8d3d1c2292a29ae1a2153ec09,18,DigitexFutures,DGTX
0xa283aa7cfbb27ef0cfbcb2493dd9f4330e0fd304,18,MMToken,MM1
0x0000000000095413afc295d19edeb1ad7b71c952,18,Tokenlon,LON
0x1f3f9d3068568f8040775be2e8c03c103c61f3af,18,ArcherDAOGovernanceToken,ARCH
0xbb0a009ba1eb20c5062c790432f080f6597662af,18,BitBotV1,BBP
0x99fe3b1391503a1bc1788051347a1324bff41452,18,SportX,SX
0xa393473d64d2f9f026b60b6df7859a689715d092,8,LatticeToken,LTX
0x6fa0952355607dfb2d399138b7fe10eb90f245e4,18,ClashToken,SCT
0xbbff34e47e559ef680067a6b1c980639eeb64d24,18,LeverjGluon,L2
0xe0ad1806fd3e7edf6ff52fdb822432e847411033,18,OnX.finance,ONX
0x2791bfd60d232150bff86b39b7146c0eaaa2ba81,18,BiFi,BiFi
0xef3a930e1ffffacd2fc13434ac81bd278b0ecc8d,18,StaFi,FIS
0x3e9bc21c9b189c09df3ef1b824798658d5011937,18,LinearToken,LINA
0xb1f871ae9462f1b2c6826e88a7827e76f86751d4,18,GNYerc20,GNYerc20
0x9aeb50f542050172359a0e1a25a9933bc8c01259,8,oinfinance,OIN
0x6b33f15360cedbfb8f60539ec828ef52910aca9b,18,xINCH,xINCHb
0x8f6a193c8b3c949e1046f1547c3a3f0836944e4b,18,xINCH,xINCHa
0xac0104cca91d167873b8601d2e71eb3d4d8c33e0,18,Crowns,CWS
0x95ad61b0a150d79219dcf64e1e6cc01f0b64c4ce,18,SHIBAINU,SHIB
0x6d0f5149c502faf215c89ab306ec3e50b15e2892,18,PortionToken,PRT
0x8a9c4dfe8b9d8962b31e4e16f8321c44d48e246e,18,NameChangeToken,NCT
0xb6ca7399b4f9ca56fc27cbff44f4d2e4eef1fc81,18,Muse,MUSE
0x2b915b505c017abb1547aa5ab355fbe69865cc6d,6,Maps.meToken,MAPS
0x6e36556b3ee5aa28def2a8ec3dae30ec2b208739,18,BUILDFinance,BUILD
0x4f5fa8f2d12e5eb780f6082dd656c565c48e0f24,18,GourmetGalaxy,GUM
0xfa6de2697d59e88ed7fc4dfe5a33dac43565ea41,18,DEFITop5TokensIndex,DEFI5
0x17ac188e09a7890a1844e5e65471fe8b0ccfadf3,18,CryptocurrencyTop10TokensIndex,CC10
0x8eef5a82e6aa222a60f009ac18c24ee12dbf4b41,18,TixlToken,TXL
0x50de6856358cc35f3a9a57eaaa34bd4cb707d2cd,18,RAZOR,RAZOR
0x297d33e17e61c2ddd812389c2105193f8348188a,18,StrudelFinance,TRDL
0xec681f28f4561c2a9534799aa38e0d36a83cf478,18,YVS.Finance,YVS
0x8d3e855f3f55109d473735ab76f753218400fe96,18,Bundles,BUND
0xc28e27870558cf22add83540d2126da2e4b464c2,18,SashimiToken,SASHIMI
0x7968bc6a03017ea2de509aaa816f163db0f35148,6,Hedget,HGET
0xa0afaa285ce85974c3c881256cb7f225e3a1178a,18,WrappedCRES,wCRES
0x15d4c048f83bd7e37d49ea4c83a07267ec4203da,8,Gala,GALA
0x39795344cbcc76cc3fb94b9d1b15c23c2070c66d,9,SeigniorageShares,SHARE
0x9ed8e7c9604790f7ec589f99b94361d8aab64e5e,18,Unistake,UNISTAKE
0x910524678c0b1b23ffb9285a81f99c29c11cbaed,18,DokiDokiAzuki,AZUKI
0xe61fdaf474fac07063f2234fb9e60c1163cfa850,18,CoinUtilityToken,COIN
0x5f64ab1544d28732f0a24f4713c2c8ec0da089f0,18,DEXTFToken,DEXTF
0x018fb5af9d015af25592a014c4266a84143de7a0,18,mp3,MP3
0xb59490ab09a0f526cc7305822ac65f2ab12f9723,18,Litentry,LIT
0xd2877702675e6ceb975b4a1dff9fb7baf4c91ea9,18,WrappedLUNAToken,LUNA
0x00aba6fe5557de1a1d565658cbddddf7c710a1eb,18,EASYV2,EZ
0xb6ff96b8a8d214544ca0dbc9b33f7ad6503efd32,18,SYNC,SYNC
0x8a40c222996f9f3431f63bf80244c36822060f12,18,Finxflo,FXF
0xf9fbe825bfb2bf3e387af0dc18cac8d87f29dea8,18,BotOcean,BOTS
0x66c0dded8433c9ea86c8cf91237b14e10b4d70b7,18,MarsToken,Mars
0x7b3d36eb606f873a75a6ab68f8c999848b04f935,18,NFTLootBox.com,LOOT
0xc0ba369c8db6eb3924965e5c4fd0b4c1b91e305f,18,DLPDuckToken,DUCK
0xaef4f02e31cdbf007f8d98da4ae365188a0e9ecc,8,TheFamousToken,TFT
0xcca0c9c383076649604ee31b20248bc04fdf61ca,18,BitMaxtoken,BTMX
0xebd9d99a3982d547c5bb4db7e3b1f9f14b67eb83,18,EverestID,ID
0x00c83aecc790e8a4453e5dd3b0b4b3680501a7a7,18,SKALE,SKL
0x0f51bb10119727a7e5ea3538074fb341f56b09ad,18,DAOMaker,DAO
0x76c5449f4950f6338a393f53cda8b53b0cd3ca3a,18,BT.Finance,BT
0xb4467e8d621105312a914f1d42f10770c0ffe3c8,18,FlashToken,FLASH
0xf680429328caaacabee69b7a9fdb21a71419c063,18,ButterflyProtocolGovernanceToken,BFLY
0x6fc13eace26590b80cccab1ba5d51890577d83b2,18,Umbrella,UMB
0x903bef1736cddf2a537176cf3c64579c3867a881,9,ichi.farm,ICHI
0x1456688345527be1f37e9e627da0837d6f08c925,18,USDPStablecoin,USDP2
0x23b608675a2b2fb1890d3abbd85c5775c51691d5,18,UnisocksEdition0,SOCKS
0xaa4e3edb11afa93c41db59842b29de64b72e355b,18,MarginSwap,MFI
0x9af15d7b8776fa296019979e70a5be53c714a7ec,18,EvnToken,EVN
0xb97d5cf2864fb0d08b34a484ff48d5492b2324a0,18,KLON,KLON
0xe6c3502997f97f9bde34cb165fbce191065e068f,18,KBTC,KBTC
0xdcb01cc464238396e213a6fdd933e36796eaff9f,18,Yield,YLD
0x69e8b9528cabda89fe846c67675b5d73d463a916,18,BBB,OPEN
0x196c81385bc536467433014042788eb707703934,18,CTASKToken,CTASK
0xd7b7d3c0bda57723fb54ab95fd8f9ea033af37f2,18,PYLON,PYLON
0x89bd2e7e388fab44ae88bef4e1ad12b4f1e0911c,18,NUXPeanut.trade,NUX
0xa0bed124a09ac2bd941b10349d8d224fe3c955eb,18,DePay,DEPAY
0x6bff2fe249601ed0db3a87424a2e923118bb0312,18,FYOOZ,FYZ
0xfdc4a3fc36df16a78edcaf1b837d3acaaedb2cb4,18,ScifiToken,SCIFI
0xb9ef770b6a5e12e45983c5d80545258aa38f3b78,10,0chain,ZCN
0xadb2437e6f65682b85f814fbc12fec0508a7b1d0,18,UniCrypt,UNCX
0xed40834a13129509a89be39a9be9c0e96a0ddd71,18,WarpToken,WARP
0x875773784af8135ea0ef43b5a374aad105c5d39e,18,Idle,IDLE
0x0488401c3f535193fa8df029d9ffe615a06e74e6,18,SparkPoint,SRK
0x038a68ff68c393373ec894015816e33ad41bd564,18,Glitch,GLCH
0x10be9a8dae441d276a5027936c3aaded2d82bc15,18,unimexnetwork,UMX
0x5f0e628b693018f639d10e4a4f59bd4d8b2b6b44,18,WhiteheartToken,WHITE
0x3597bfd533a99c9aa083587b074434e61eb0a258,8,DENT,DENT
0xb1e9157c2fdcc5a856c8da8b2d89b6c32b3c1229,18,ZenfuseTradingPlatformToken,ZEFU
0x260e63d91fccc499606bae3fe945c4ed1cf56a56,18,MoonTools.io,MOONS
0x066798d9ef0833ccc719076dab77199ecbd178b0,18,SakeToken,SAKE
0x945facb997494cc2570096c74b5f66a3507330a1,18,mStableBTC,mBTC
0x1b40183efb4dd766f11bda7a7c3ad8982e998421,18,VesperToken,VSP
0x84810bcf08744d5862b8181f12d17bfd57d3b078,18,SharedStakeGovernanceToken,SGT
0xed0439eacf4c4965ae4613d77a5c2efe10e5f183,18,shroom.finance,SHROOM
0xaac41ec512808d64625576eddd580e7ea40ef8b2,18,gameswap.org,GSWAP
0x2688213fedd489762a281a67ae4f2295d8e17ecc,18,FUD.finance,FUD
0x03ab458634910aad20ef5f1c8ee96f1d6ac54919,18,RaiReflexIndex,RAI
0x95b3497bbcccc46a8f45f5cf54b0878b39f8d96c,18,UniDex,UNIDX
0x9ceb84f92a0561fa3cc4132ab9c0b76a59787544,18,DokiDokiFinance,DOKI
0xfc05987bd2be489accf0f509e44b0145d68240f7,18,ESSENTIA,ESS
0xc3771d47e2ab5a519e2917e61e23078d0c05ed7f,18,Gather,GTH
0xabe580e7ee158da464b51ee1a83ac0289622e6be,18,Offshift,XFT
0xca3fe04c7ee111f0bbb02c328c699226acf9fd33,18,seen.haus,SEEN
0x34612903db071e888a4dadcaa416d3ee263a87b9,18,ethart,arte
0x7ca4408137eb639570f8e647d9bd7b7e8717514a,18,AlpaToken,ALPA
0xa58a4f5c4bb043d2cc1e170613b74e767c94189b,18,UTUCoin,UTU
0x9a0aba393aac4dfbff4333b06c407458002c6183,18,ACoconut,AC
0x7eaf9c89037e4814dc0d9952ac7f888c784548db,18,Royale,ROYA
0x6399c842dd2be3de30bf99bc7d1bbf6fa3650e70,18,Premia,PREMIA
0x4ca0654f4fc1025cf1a17b7459c20ac0479522ad,18,RigelFinance,RIGEL
0x51e00a95748dbd2a3f47bc5c3b3e7b3f0fea666c,18,DVGToken,DVG
0x0cec1a9154ff802e7934fc916ed7ca50bde6844e,18,PoolTogether,POOL
0x72e9d9038ce484ee986fea183f8d8df93f9ada13,18,SMARTCREDITToken,SMARTCREDIT
0xcb5f72d37685c3d5ad0bb5f982443bc8fcdf570e,18,RootKit,ROOT
0x557b933a7c2c45672b610f8954a3deb39a51a8ca,18,REVV,REVV
0x38a2fdc11f526ddd5a607c1f251c065f40fbf2f7,18,PhoenixDAO,PHNX
0x7866e48c74cbfb8183cd1a929cd9b95a7a5cb4f4,18,DexKit,KIT
0x4691937a7508860f876c9c0a2a617e7d9e945d4b,18,WootradeNetwork,WOO
0xea1ea0972fa092dd463f2968f9bb51cc4c981d71,18,Modefi,MOD
0x946112efab61c3636cbd52de2e1392d7a75a6f01,18,Hydro,HYDRO
0x0fe629d1e84e171f8ff0c1ded2cc2221caa48a3f,18,Mask,MASK
0xf99d58e463a2e07e5692127302c20a191861b4d6,18,Anyswap,ANY
0xf1f955016ecbcd7321c7266bccfb96c68ea5e49b,18,Rally,RLY
0x16980b3b4a3f9d89e33311b5aa8f80303e5ca4f8,6,KIRANetwork,KEX
0xd13c7342e1ef687c5ad21b27c2b65d772cab5c8c,4,UltraToken,UOS
0x147faf8de9d8d8daae129b187f0d02d819126750,18,GeoDBCoin,GEO
0x8a854288a5976036a725879164ca3e91d30c6a1b,18,GET,GET
0x79c75e2e8720b39e258f41c37cc4f309e0b0ff80,8,PhantasmaStake,SOUL
0x9e78b8274e1d6a76a0dbbf90418894df27cbceb5,18,UniFi,UniFi
0x3a880652f47bfaa771908c07dd8673a787daed3a,18,DerivaDAO,DDX
0x6c28aef8977c9b773996d0e8376d2ee379446f2f,18,Quickswap,QUICK
0xa1d6df714f91debf4e0802a542e13067f31b8262,18,RFOX,RFOX
0x275f5ad03be0fa221b4c6649b8aee09a42d9412a,18,Monavale,MONA
0x9b53e429b0badd98ef7f01f03702986c516a5715,18,hybrixhydra,HY
0xe53ec727dbdeb9e2d5456c3be40cff031ab40a55,18,SuperFarm,SUPER
0x123151402076fc819b7564510989e475c9cd93ca,8,wrapped-DGLD,wDGLD
0xada0a1202462085999652dc5310a7a9e2bf3ed42,18,CoinSharesGoldandCryptoassetsIndexLite,CGI
0xea6412fb370e8d1605e6aeeaa21ad07c3c7e9f24,18,MUSH,MUSH
0x6149c26cd2f7b5ccdb32029af817123f6e37df5b,18,Launchpooltoken,LPOOL
0x69af81e73a73b40adf4f3d4223cd9b1ece623074,18,MaskNetwork,MASKNTWRK
0x7f3edcdd180dbe4819bd98fee8929b5cedb3adeb,18,xToken,XTK
0x7777777777697cfeecf846a76326da79cc606517,18,xSigma,SIG
0xf5581dfefd8fb0e4aec526be659cfab1f8c781da,18,HOPRToken,HOPR
0x6c972b70c533e2e045f333ee28b9ffb8d717be69,18,FoundryLogisticsToken,FRY
0x63f88a2298a5c4aee3c216aa6d926b184a4b2437,18,GAMECredits,GAME
0xc8d2ab2a6fdebc25432e54941cb85b55b9f152db,18,GRAP,GRAP
0x6de037ef9ad2725eb40118bb1702ebb27e4aeb24,18,RenderToken,RNDR
0x21bfbda47a0b4b5b1248c767ee49f7caa9b23697,18,OVR,OVR
0x2e1e15c44ffe4df6a0cb7371cd00d5028e571d14,18,Mettalex,MTLX
0xd478161c952357f05f0292b56012cd8457f1cfbf,18,Polkamarkets,POLK
0x48fb253446873234f2febbf9bdeaa72d9d387f94,18,BancorGovernanceToken,vBNT
0x011a014d5e8eb4771e575bb1000318d509230afa,8,CreamUniswapV2WBTC-ETH,crUNIV2WBTCETH
0x01da76dea59703578040012357b81ffe62015c2d,8,CreamyETH,crYETH
0x054b7ed3f45714d3091e82aad64a1588dc4096ed,8,CreamHuobiBTC,crHBTC
0x10a3da2bb0fae4d591476fd97d6636fd172923a8,8,CreamFTXToken,crHEGIC
0x10fdbd1e48ee2fd9336a482d746138ae19e649db,8,CreamFTXToken,crFTT
0x17107f40d70f4470d20cb3f138a052cae8ebd4be,8,CreamrenBTC,crRENBTC
0x197070723ce0d3810a0e47f06e935c30a480d4fc,8,CreamWrappedBTC,crWBTC
0x19d1666f543d42ef17f66e376944a22aea1a8e46,8,CreamCompound,crCOMP
0x1d0986fb43985c88ffa9ad959cc24e6a087c7e35,8,CreamAlphaToken,crALPHA
0x1ff8cdb51219a8838b52e9cac09b71e591bc998e,8,CreamBinanceUSD,crBUSD
0x21011bc93d9e515b9511a817a1ed1d6d468f49fc,8,CreamCoverProtocolGovernanceToken,crCOVER
0x228619cca194fbe3ebeb2f835ec1ea5080dafbb2,8,CreamSushiBar,crXSUSHI
0x22b243b96495c547598d9042b6f94b01c22b2e9e,8,CreamSwagToken,crSWAG
0x25555933a8246ab67cbf907ce3d1949884e82b55,8,CreamSynthsUSD,crSUSD
0x2a537fa9ffaea8c1a41d3c2b68a9cb791529366d,8,CreamDefiPulseIndex,crDPI
0x2db6c82ce72c8d7d770ba1b5f5ed0b6e075066d6,8,CreamAmp,crAMP
0x3225e3c669b39c7c8b3e204a8614bb218c5e31bc,8,CreamAAVEToken,crAAVE
0x338286c0bc081891a4bda39c7667ae150bf5d206,8,CreamSushiToken,crSUSHI
0x3623387773010d9214b10c551d6e7fc375d31f58,8,CreamMeta,crMTA
0x38f27c03d6609a86ff7716ad03038881320be4ad,8,CreamSushiSwapLPTokenDAI-ETH,crSLPDAIETH
0x3c6c553a95910f9fc81c98784736bd628636d296,8,CreamEmptySetDollar,crESD
0x44fbebd2f576670a6c33f6fc0b00aa8c5753b322,8,CreamUSDCoin,crUSDC
0x460ea730d204c822ce709f00a8e5959921715adc,8,CreamBasisCash,crBAC
0x4ee15f44c6f0d8d1136c83efd2e8e4ac768954c6,8,CreamyyCRV,crYYCRV
0x4fe11bc316b6d7a345493127fbe298b95adaad85,8,CreamUniswapV2USDC-ETH,crUNIV2USDCETH
0x59089279987dd76fc65bf94cb40e186b96e03cb3,8,CreamOriginToken,crOGN
0x5c291bc83d15f71fb37805878161718ea4b6aee9,8,CreamSushiSwapLPTokenETH-USDT,crSLPETHUSDT
0x5ecad8a75216cea7dff978525b2d523a251eea92,8,CreamSushiSwapLPTokenUSDC-ETH,crSLPUSDCETH
0x65883978ada0e707c3b2be2a6825b1c4bdf76a90,8,CreamAkropolis,crAKRO
0x697256caa3ccafd62bb6d3aa1c7c5671786a5fd9,8,CreamChainLinkToken,crLINK
0x6ba0c66c48641e220cf78177c144323b3838d375,8,CreamSushiSwapLPTokenSUSHI-ETH,crSLPSUSHIETH
0x73f6cba38922960b7092175c0add22ab8d0e81fc,8,CreamSushiSwapLPTokenWBTC-ETH,crSLPWBTCETH
0x797aab1ce7c01eb727ab980762ba88e7133d2157,8,CreamUSDT,crUSDT
0x7aaa323d7e398be4128c7042d197a2545f0f1fea,8,CreamOMGToken,crOMG
0x7ea9c63e216d5565c3940a2b3d150e59c2907db3,8,CreamBinanceWrappedBTC,crBBTC
0x85759961b116f1d36fd697855c57a6ae40793d9b,8,Cream1INCHToken,cr1INCH
0x8b3ff1ed4f36c2c2be675afb13cc3aa5d73685a5,8,CreamCelsius,crCEL
0x8b86e0598616a8d4f1fdae8b59e55fb5bc33d0d6,8,CreamEthLendToken,crLEND
0x8b950f43fcac4931d408f1fcda55c6cb6cbf3096,8,CreamBadgerSettBadger,crBBADGER
0x903560b1cce601794c584f58898da8a8b789fc5d,8,CreamKeep3rV1,crKP3R
0x92b767185fb3b04f881e3ac8e5b0662a027a1d9f,8,CreamDaiStablecoin,crDAI
0x9baf8a5236d44ac410c0186fe39178d5aad0bb87,8,CreamyCRV,crYCRV
0xb092b4601850e23903a42eacbc9d8a0eec26a4d5,8,CreamFrax,crFRAX
0xc25eae724f189ba9030b2556a1533e7c8a732e14,8,CreamSynthetixNetworkToken,crSNX
0xc68251421edda00a10815e273fa4b1191fac651b,8,CreamPickleToken,crPICKLE
0xc7fd8dcee4697ceef5a2fd4608a7bd6a94c77480,8,CreamCRV,crCRV
0xcbae0a83f4f9926997c8339545fb8ee32edc6b76,8,CreamYFI,crYFI
0xcd22c4110c12ac41acefa0091c432ef44efaafa0,8,CreamUniswapV2DAI-ETH,crUNIV2DAIETH
0xce4fe9b4b8ff61949dcfeb7e03bc9faca59d2eb3,8,CreamBalancer,crBAL
0xd06527d5e56a3495252a528c4987003b712860ee,8,CreamEther,crETH
0xd5103afcd0b3fa865997ef2984c66742c51b2a8b,8,CreamHuobiFIL,crHFIL
0xd532944df6dfd5dd629e8772f03d4fc861873abf,8,CreamSushiSwapLPTokenYFI-ETH,crSLPYFIETH
0xd692ac3245bb82319a31068d6b8412796ee85d2c,8,CreamHUSD,crHUSD
0xe6c3120f38f56deb38b69b65cc7dcaf916373963,8,CreamUniswapV2ETH-USDT,crUNIV2ETHUSDT
0xe89a6d0509faf730bd707bf868d9a2a744a363c7,8,CreamUniswap,crUNI
0xef58b2d5a1b8d3cde67b8ab054dc5c831e9bc025,8,CreamSerum,crSRM
0xeff039c3c1d668f408d09dd7b63008622a77532c,8,CreamWrappedNXM,crWNXM
0xf047d4be569fb770db143a6a90ef203fc1295922,8,CreamtBTC,crTBTC
0xf55bbe0255f7f4e70f63837ff72a577fbddbe924,8,CreamBarnBridgeGovernanceToken,crBOND
0xfd609a03b393f1a1cfcacedabf068cad09a924e2,8,CreamCreamETH2,crCRETH2
0x6e9730ecffbed43fd876a264c982e254ef05a0de,18,NordToken,NORD
0x661ab0ed68000491d98c796146bcf28c20d7c559,18,ShadowsNetwork,DOWS
0x0cdf9acd87e940837ff21bb40c9fd55f68bba059,18,PublicMint,MINT
0x126c121f99e1e211df2e5f8de2d96fa36647c855,18,DEGENIndex,DEGEN
0x0c7d5ae016f806603cb1782bea29ac69471cab9c,18,Bifrost,BFC
0xd23ac27148af6a2f339bd82d0e3cff380b5093de,18,SIREN,SI
0x4c25bdf026ea05f32713f00f73ca55857fbf6342,18,Font,FONT
0x68a3637ba6e75c0f66b61a42639c4e9fcd3d4824,18,MoonToken,MOON
0x4b1e80cac91e2216eeb63e29b957eb91ae9c2be8,18,Jupiter,JUP
0xa44e5137293e855b1b7bc7e2c6f8cd796ffcb037,8,SENTinel,SENT
0x88ef27e69108b2633f8e1c184cc37940a075cc02,18,dego.finance,DEGO
0xee06a81a695750e71a662b51066f2c74cf4478a0,18,decentral.games,DG
0x0dde6f6e345bfd23f3f419f0dfe04e93143b44fb,18,SOTA,SOTA
0x739763a258640919981f9ba610ae65492455be53,18,NodeRunners,NDR
0xd82bb924a1707950903e2c0a619824024e254cd1,18,DAOfi,DAOfi
0x31c8eacbffdd875c74b94b077895bd78cf1e64a3,18,Radicle,RAD
0xdbdb4d16eda451d0503b854cf79d55697f90c8df,18,Alchemix,ALCX
0x24a6a37576377f63f194caa5f518a60f45b42921,18,FloatBank,BANK
0x9b99cca871be05119b2012fd4474731dd653febe,18,Antimatter.FinanceGovernanceToken,MATTER
0xf4d861575ecc9493420a3f5a14f85b13f0b50eb3,18,FractalProtocolToken,FCL
0xf411903cbc70a74d22900a5de66a2dda66507255,18,VERA,VRA
0xe5feeac09d36b18b3fa757e5cf3f8da6b8e27f4c,18,NFTINDEX,NFTI
0x5affe41805a9e57fed3657d0e64d96aea0b77885,6,OpiumUSDTProtectionv2USDC,OPIUMLP34b7
0xf3dcbc6d72a4e1892f7917b7c43b74131df8480e,18,BDPToken,BDP
0xbbc2ae13b23d715c30720f079fcd9b4a74093505,18,EthernityChainERNToken,ERN
0x67b6d479c7bb412c54e03dca8e1bc6740ce6b99c,18,KylinNetwork,KYL
0xfb5453340c03db5ade474b27e68b6a9c6b2823eb,18,MetaFactory,ROBOT
0x59e9261255644c411afdd00bd89162d09d862e38,18,ETHA,ETHA
0x4c6ec08cf3fc987c6c4beb03184d335a2dfc4042,18,Paint,PAINT
0x2aeccb42482cc64e087b6d2e5da39f5a7a7001f8,18,RulerProtocol,RULER
0x892a6f9df0147e5f079b0993f486f9aca3c87881,9,unification.comxfund,xFUND
0x7a5ce6abd131ea6b148a022cb76fc180ae3315a6,18,bAlpha,bALPHA
0x80ab141f324c3d6f2b18b030f1c4e95d4d658778,18,DEA,DEA
0x3506424f91fd33084466f402d5d97f05f8e3b4af,18,chiliZ,CHZ
0x41d5d79431a913c4ae7d69a668ecdfe5ff9dfb68,18,InverseDAO,INV
0xeeaa40b28a2d1b0b08f6f97bb1dd4b75316c6107,18,GOVI,GOVI
0xcf3c8be2e2c42331da80ef210e9b1b307c03d36a,18,BetProtocolToken,BEPRO
0x1cbb83ebcd552d5ebf8131ef8c9cd9d9bab342bc,18,Non-FungibleYearn,NFY
0x2216e873ea4282ebef7a02ac5aea220be6391a7c,18,smol,SMOL
0x92e187a03b6cd19cb6af293ba17f2745fd2357d5,18,UnitProtocol,DUCKUNIT
0x888888435fde8e7d4c54cab67f206e4199454c60,18,DFXToken,DFX
0xc4de189abf94c57f396bd4c52ab13b954febefd8,18,B.20,B20
0x7f1f2d3dfa99678675ece1c243d3f7bc3746db5d,18,Tapmydata,TAP
0x00d1793d7c3aae506257ba985b34c76aaf642557,18,Tacos,TACO
0x19a2cf2a1b2f76e52e2b0c572bd80a95b4fa8643,18,FyoozNFT,FYZNFT
0x92ef4ffbfe0df030837b65d7fccfe1abd6549579,18,Swirge,SWG
0xed30dd7e50edf3581ad970efc5d9379ce2614adb,18,ARCGovernanceTokenOld,ARCXOLD
0xcd91538b91b4ba7797d39a2f66e63810b50a33d0,18,ARCSTABLEx,STABLEx
0x940a2db1b7008b6c776d4faaca729d6d4a4aa551,18,DuskNetwork,DUSK
0xaa6e8127831c9de45ae56bb1b0d4d4da6e5665bd,18,ETH2xFlexibleLeverageIndex,ETH2xFLI
0x50d1c9771902476076ecfc8b2a83ad6b9355a4c9,18,FTT,FTXToken
0x037a54aab062628c9bbae1fdb1583c195585fe41,18,LCX,LCX
0x544c42fbb96b39b21df61cf322b5edc285ee7429,18,InsurAce,INSUR
0x29239242a83479a4074cb1c9e2a3e6705a4a4455,18,TOZEX,TOZ
0xf938424f7210f31df2aee3011291b658f872e91e,18,VISOR,VISR
0x8185bc4757572da2a610f887561c32298f1a5748,18,Aluna,ALN
0x1fe24f25b1cf609b9c4e7e12d802e3640dfa5e43,18,ChainGuardiansGovernanceToken,CGG
0x32a7c02e79c4ea1008dd6564b35f131428673c41,18,CRUST,CRU
0xac51066d7bec65dc4589368da368b212745d63e8,6,ALICE,ALICE
0x1c9922314ed1415c95b9fd453c3818fd41867d0b,18,TOWER,TOWER
0xa8b61cff52564758a204f841e636265bebc8db9b,18,YieldProtocol,YIELD
0x850aab69f0e0171a9a49db8be3e71351c8247df4,18,Konomi,KONO
0xbae5f2d8a1299e5c4963eaff3312399253f27ccb,9,SOAR.FI,SOAR
0xac3211a5025414af2866ff09c23fc18bc97e79b1,18,DOVU,DOV
0x3f4e02741b155f5ce8d6190d294d4f916125b896,18,CircleEx,CX
0x73374ea518de7addd4c2b624c0e8b113955ee041,18,JuggernautDeFi,JGN
0xfad45e47083e4607302aa43c65fb3106f1cd7607,9,hoge.finance,HOGE
0xe9a95d175a5f4c9369f3b74222402eb1b837693b,8,ChangeNOW,NOW
0xeabb8996ea1662cad2f7fb715127852cd3262ae9,18,ConnectFinancial,CNFI
0xdea67845a51e24461d5fed8084e69b426af3d5db,18,HodlTree,HTRE
0x8b39b70e39aa811b69365398e0aace9bee238aeb,18,PolkaFoundry,PKF
0xb20043f149817bff5322f1b928e89abfc65a9925,8,EXRT,EXRT
0xa487bf43cf3b10dffc97a9a744cbb7036965d3b9,18,Deri,DERI
0x14da7b27b2e0fedefe0a664118b0c9bc68e2e9af,18,BlockchainCutiesUniverseGovernanceToken,BCUG
0x8b0e42f366ba502d787bb134478adfae966c8798,18,LABSGroup,LABS
0xcbfef8fdd706cde6f208460f2bf39aa9c785f05d,18,KineGovernanceToken,KINE
0xd9c2d319cd7e6177336b0a9c93c21cb48d84fb54,18,HAPI,HAPI
0xc5bddf9843308380375a611c18b50fb9341f502a,18,veCRV-DAOyVault,yveCRVDAO
0xb9d99c33ea2d86ec5ec6b8a4dd816ebba64404af,18,k21.kanon.art,K21
0x226f7b842e0f0120b7e194d05432b3fd14773a9d,18,UNIONProtocolGovernanceToken,UNN
0x0ace32f6e87ac1457a5385f8eb0208f37263b415,10,HabitatToken,HBT
0xa42f266684ac2ad6ecb00df95b1c76efbb6f136c,18,CashTech,CATE
0xbc4171f45ef0ef66e76f979df021a34b46dcc81d,18,Dorayaki,DORA
0x07bac35846e5ed502aa91adf6a9e7aa210f2dcbe,18,erowan,erowan
0x7865af71cf0b288b4e7f654f4f7851eb46a2b7f8,18,Sentivate,SNTVT
0xc4c2614e694cf534d407ee49f8e44d125e4681c4,18,ChainGames,CHAIN
0x000000000000d0151e748d25b766e77efe2a6c83,18,XDEFIGovernanceToken,XDEX
0x8564653879a18c560e7c0ea0e084c516c62f5653,18,UpBots,UBXT
0x9f9c8ec3534c3ce16f928381372bfbfbfb9f4d24,18,GraphLinq,GLQ
0xc690f7c7fcffa6a82b79fab7508c466fefdfc8c5,18,Lympotokens,LYM
0x3d3d35bb9bec23b06ca00fe472b50e7a4c692c30,18,Vidya,VIDYA
0xf921ae2dac5fa128dc0f6168bf153ea0943d2d43,8,FireProtocol,FIRE
0xfc979087305a826c2b2a0056cfaba50aad3e6439,18,DAFIToken,DAFI
0xcd2828fc4d8e8a0ede91bb38cf64b1a81de65bf6,18,OddzToken,ODDZ
0x72630b1e3b42874bf335020ba0249e3e9e47bafc,18,PaypolitanToken,EPAN
0xe3cb486f3f5c639e98ccbaf57d95369375687f80,8,renDGB,renDGB
0x0ace20a35ee382bda90d1d93cee65f5970fc01c8,18,Chip,CHIP
0xa92e7c82b11d10716ab534051b271d2f6aef7df5,18,AraToken,ARA
0x5dc02ea99285e17656b8350722694c35154db1e8,8,bonded.finance,BONDfinance
0xc96c1609a1a45ccc667b2b7fa6508e29617f7b69,18,2GT_token,2GT
0xc3d088842dcf02c13699f936bb83dfbbc6f721ab,18,VoucherEthereum,vETH
0xca0e7269600d353f70b14ad118a49575455c0f2f,18,AMLT,AMLT
0xceb286c9604c542d3cc08b41aa6c9675b078a832,18,VortexDeFi,VTX
0xba4cfe5741b357fa371b506e5db0774abfecf8fc,18,vVSPpool,vVSP
0xc7283b66eb1eb5fb86327f08e1b5816b0720212b,18,Tribe,TRIBE
0x956f47f50a910163d8bf957cf5846d573e7f87ca,18,FeiUSD,FEI
0x70e8de73ce538da2beed35d14187f6959a8eca96,6,XSGD,XSGD
0x626e8036deb333b408be468f951bdb42433cbf18,18,AIOZNetwork,AIOZ
0xa0cf46eb152656c7090e769916eb44a138aaa406,18,Spheroid,SPH
0x1796ae0b0fa4862485106a0de9b654efe301d0b2,18,Polkamon,PMON
0xf293d23bf2cdc05411ca0eddd588eb1977e8dcd4,18,Sylo,SYLO
0xdb0acc14396d108b3c5574483acb817855c9dc8d,8,Emblem,EMB
0x1735db6ab5baa19ea55d0adceed7bcdc008b3136,18,UREEQAToken,URQA
0x99295f1141d58a99e939f7be6bbe734916a875b8,18,LinkPool,LPL
0xe1c7e30c42c24582888c758984f6e382096786bd,8,Curate,XCUR
0xff75ced57419bcaebe5f05254983b013b0646ef5,18,CookToken,COOK
0x26c8afbbfe1ebaca03c2bb082e69d0476bffe099,18,CellframeToken,CELL
0x5b09a0371c1da44a8e24d36bf5deb1141a84d875,18,MADToken,MAD
0xc834fa996fa3bec7aad3693af486ae53d8aa8b50,18,Convergence,CONV
0x106538cc16f938776c7c180186975bca23875287,18,BASv2,BASv2
0x3505f494c3f0fed0b594e01fa41dd3967645ca39,18,SWARM,SWM
0xfbbe9b1142c699512545f47937ee6fae0e4b0aa9,18,EDDA,EDDA
0x29cbd0510eec0327992cd6006e63f9fa8e7f33b7,18,TidalToken,TIDAL
0xb78b3320493a4efaa1028130c5ba26f0b6085ef8,18,DraculaToken,DRC2
0x55296f69f40ea6d20e478533c15a6b08b654e758,18,XYOracle,XYO
0x0f71b8de197a1c84d31de0f1fa7926c365f052b3,18,ArconaDistributionContract,ARCONA
0x9534ad65fb398e27ac8f4251dae1780b989d136e,18,PYRToken,PYR
0xa31b1767e09f842ecfd4bc471fe44f830e3891aa,18,ROOBEE,ROOBEE
0xf3ae5d769e153ef72b4e3591ac004e89f48107a1,18,DeeperNetwork,DPR
0xd9b312d77bc7bed9b9cecb56636300bed4fe5ce9,18,Gains,GAINS
0x5f98805a4e8be255a32880fdec7f6728c6568ba0,18,LUSDStablecoin,LUSD
0x1da87b114f35e1dc91f72bf57fc07a768ad40bb0,18,Equalizer,EQZ
0xe0b9a2c3e9f40cf74b2c7f591b2b0cca055c3112,18,GenShards,GS
0xa8006c4ca56f24d6836727d106349320db7fef82,8,Internxt,INXT
0x383518188c0c6d7730d91b2c03a03c837814a899,9,Olympus,OHM
0x5eaa69b29f99c84fe5de8200340b4e9b4ab38eac,18,RazeNetwork,RAZE
0x88acdd2a6425c3faae4bc9650fd7e27e0bebb7ab,18,Alchemist,MIST
0x3d6f0dea3ac3c607b3998e6ce14b6350721752d9,18,CARD.STARTER,CARDS
0x33840024177a7daca3468912363bed8b425015c5,18,ethboxToken,EBOX
0xec213f83defb583af3a000b1c0ada660b1902a0f,18,Presearch,PRE
0xf418588522d5dd018b425e472991e52ebbeeeeee,18,EthereumPushNotificationService,PUSH
0xd8e3fb3b08eba982f2754988d70d57edc0055ae6,9,Zoracles,ZORA
0xc477d038d5420c6a9e0b031712f61c5120090de9,18,BosonToken,BOSON
0xdd1ad9a21ce722c151a836373babe42c868ce9a4,18,UniversalBasicIncome,UBI
0x53c8395465a84955c95159814461466053dedede,18,DeGateToken,DG2
0x16c52ceece2ed57dad87319d91b5e3637d50afa4,18,TCAPToken,TCAP
0x44564d0bd94343f72e3c8a0d22308b7fa71db0bb,18,BasketDAOGov,BASK
0x5cf04716ba20127f1e2297addcf4b5035000c9eb,18,NKN,NKN
0x4730fb1463a6f1f44aeb45f6c5c422427f37f4d0,18,The4thPillarToken,FOUR
0x08d32b0da63e2c3bcf8019c9c5d849d7a9d791e6,0,Dentacoin,DCN
0x77fba179c79de5b7653f68b5039af940ada60ce0,18,AmpleforthGovernance,FORTH
0xe796d6ca1ceb1b022ece5296226bf784110031cd,18,BlindBoxesToken,BLES
0x48c3399719b582dd63eb5aadf12a40b4c3f52fa2,18,StakeWise,SWISE
0xdefa4e8a7bcba345f687a2f1456f5edd9ce97202,18,KyberNetworkCrystalv2,KNC
0x0ada190c81b814548ddc2f6adc4a689ce7c1fe73,18,yAxisV2,YAXIS
0xff56cc6b1e6ded347aa0b7676c85ab0b3d08b0fa,18,Orbs,ORBS
0xa02120696c7b8fe16c09c749e4598819b2b0e915,18,WirexToken,WXT
0x28a06c02287e657ec3f8e151a13c36a1d43814b0,18,BAG,BAG
0x2f109021afe75b949429fe30523ee7c0d5b27207,18,OCC,OCC
0x767fe9edc9e0df98e07454847909b5e959d7ca0e,18,Illuvium,ILV
0x5218e472cfcfe0b64a064f055b43b4cdc9efd3a6,18,UnFederalReserveToken,eRSDL
0xf16e81dce15b08f326220742020379b855b87df9,18,IceToken,ICE
0xe4cfe9eaa8cdb0942a80b7bc68fd8ab0f6d44903,18,XEND,XEND
0x5a666c7d92e5fa7edcb6390e4efd6d0cdd69cf37,18,UnmarshalToken,MARSH
0x2d80f5f5328fdcb6eceb7cacf5dd8aedaec94e20,4,AGAToken,AGA
0x3d658390460295fb963f54dc0899cfb1c30776df,8,CircuitsOfValue,Coval
0x7ff4169a6b5122b664c51c95727d87750ec07c84,18,10SetToken,10SET
0x1de5e000c41c8d35b9f1f4985c23988f05831057,18,BonFi,BNF
0x818fc6c2ec5986bc6e2cbf00939d90556ab12ce5,18,Kin,KIN
0x7dd9c5cba05e151c895fde1cf355c9a1d5da6429,18,GolemNetworkToken,GLM
0x467bccd9d29f223bce8043b84e8c8b282827790f,2,Telcoin,TEL
0x90de74265a416e1393a450752175aed98fe11517,18,UnlockDiscountToken,UDT
0x808507121b80c02388fad14726482e061b8da827,18,Pendle,PENDLE
0x7a2bc711e19ba6aff6ce8246c546e8c4b4944dfd,8,WAXEconomicToken,WAXE
0x15b543e986b8c34074dfc9901136d9355a537e7e,18,StudentCoin,STC
0x16eccfdbb4ee1a85a33f3a9b21175cd7ae753db4,18,Route,ROUTE
0x7c8155909cd385f120a56ef90728dd50f9ccbe52,15,Nahmii,NII
0x1614f18fc94f47967a3fbe5ffcd46d4e7da3d787,18,PAIDNetwork,PAID
0x182f4c4c97cd1c24e1df8fc4c053e5c47bf53bef,18,keyTangoToken,TANGO
0x7bef710a5759d197ec0bf621c3df802c2d60d848,18,SPLYTSHOPX,SHOPX
0x358aa737e033f34df7c54306960a38d09aabd523,18,AresProtocol,ARES
0xb453f1f2ee776daf2586501361c457db70e1ca0f,8,AGARewards,AGAr
0x0cab5a7dcab521af5404fa604b85113267d38444,6,OpiumxDAIOmniBridgeProtection,OPIUMLP1c60
0xb54539d39529ce58fb63877debc6d6b70e3eca01,6,OpiumMaticBridgeProtection,OPIUMLPaf0d
0x321c2fe4446c7c963dc41dd58879af648838f98d,18,Cryptex,CTX
0x2d94aa3e47d9d5024503ca8491fce9a2fb4da198,18,BanklessToken,BANKBANKLESS
0x0b498ff89709d3838a063f1dfa463091f9801c2b,18,BTC2xFlexibleLeverageIndex,BTC2xFLI
0x841fb148863454a3b3570f515414759be9091465,18,ShihTzu,SHIH
0x7e8539d1e5cb91d63e46b8e188403b3f262a949b,18,SOMIDAX,SMDX
0xa2b4c0af19cc16a6cfacce81f192b024d625817d,9,KishuInu,KISHU
0x389999216860ab8e0175387a0c90e5c52522c945,9,FEGtoken,FEG
0x212dd60d4bf0da8372fe8116474602d429e5735f,18,StoboxToken,STBU
0xee9801669c6138e84bd50deb500827b776777d28,18,O3SwapToken,O3
0xd0660cd418a64a1d44e9214ad8e459324d8157f1,12,Woofy,WOOFY
0xa4cf2afd3b165975afffbf7e487cdd40c894ab6b,0,ShibaKen.Finance,SHIBAKEN
0xb1191f691a355b43542bea9b8847bc73e7abb137,18,Kirobo,KIRO
0x4e3fbd56cd56c3e72c1403e103b45db9da5b9d2b,18,ConvexToken,CVX
0xa3c4dc4a9ce2a6b40b57f25f8b50decc2c64dec2,18,SeedSwapToken,SNFT
0xfeea0bdd3d07eb6fe305938878c0cadbfa169042,18,8PAYNetwork,8PAY
0xde30da39c46104798bb5aa3fe8b9e0e1f348163f,18,Gitcoin,GTC
0x01e0e2e61f554ecaaec0cc933e739ad90f24a86d,18,Graviton,GTON
0xbc6da0fe9ad5f3b0d58160288917aa56653660e9,18,AlchemixUSD,alUSD
0x7697b462a7c4ff5f8b55bdbc2f4076c2af9cf51a,18,Sarcophagus,SARCO
0xa1ed0364d53394209d61ae8bfdb8ff50484d8c91,18,TeraBlockToken,TBC
0x677ddbd918637e5f2c79e164d402454de7da8619,18,VUSD,VUSD
0x3c9d6c1c73b31c837832c72e04d3152f051fc1a9,18,BoringDAO,BOR
0x1341a2257fa7b770420ef70616f888056f90926c,9,ZooToken,ZOOT
0x761d38e5ddf6ccf6cf7c55759d5210750b5d60f3,18,Dogelon,ELON
0x0563dce613d559a47877ffd1593549fb9d3510d6,18,SuperBid,SUPERBID
0x2ab6bb8408ca3199b8fa6c92d5b455f820af03c4,18,TE-FOODTustChain,TONE
0x06a01a4d579479dd5d884ebf61a31727a3d8d442,8,SmartKey,Skey
0xeb58343b36c7528f23caae63a150240241310049,18,Nimbus,NBU
0x27c70cd1946795b66be9d954418546998b546634,18,DOGEKILLER,LEASH
0x0abdace70d3790235af448c88547603b945604ea,18,district0xNetworkToken,DNT
0xdef1fac7bf08f173d286bbbdcbeeade695129840,18,DefiFactoryToken,DEFT
0x9695e0114e12c0d3a3636fab5a18e6b737529023,18,DFYNToken,DFYN
0x72e364f2abdc788b7e918bc238b21f109cd634d7,18,MetaverseIndex,MVI
0x3301ee63fb29f863f2333bd4466acb46cd8323e6,18,AkitaInu,AKITA
0x6dea81c8171d0ba574754ef6f8b412f2ed88c54d,18,LQTY,LQTY
0x87de305311d5788e8da38d19bb427645b09cb4e5,18,Verox,VRX
0xf65b5c5104c4fafd4b709d9d60a185eae063276c,18,Truebit,TRUTruebit
0x9be89d2a4cd102d8fecc6bf9da793be995c22541,8,BinanceWrappedBTC,BBTC
0x944eee930933be5e23b690c8589021ec8619a301,9,MUNCHToken,MUNCH
0xdc349913d53b446485e98b76800b6254f43df695,9,BezogeEarth,BEZOGE
0x1681bcb589b3cfcf0c0616b0ce9b19b240643dc1,9,Island,ISLE
0xa150db9b1fa65b44799d4dd949d922c0a33ee606,0,DigitalReserveCurrency,DRC1
0xe047705117eb07e712c3d684f5b18e74577e83ac,8,BitcashPay,BCP
0x15874d65e649880c2614e7a480cb7c9a55787ff6,18,EthereumMax,eMax
0x6f40d4a6237c257fff2db00fa0510deeecd303eb,18,Instadapp,INST
0x456d8f0d25a4e787ee60c401f8b963a465148f70,9,Cavapoo,CAVA
0x8cb924583681cbfe487a62140a994a49f833c244,18,SwappToken,SWAPP
0x10633216e7e8281e33c86f02bf8e565a635d9770,18,Dvision,DVI
0xcadc0acd4b445166f12d2c07eac6e2544fbe2eef,18,CADCoin,CADC
0x1321f1f1aa541a56c31682c57b80ecfccd9bb288,18,ARCxGovernanceToken,ARCX
0xa9536b9c75a9e0fae3b56a96ac8edf76abc91978,18,AmunDeFiIndex,DFI
0x1660f10b4d610cf482194356ece8efd65b15ba83,18,AmunDeFiMomentumIndex,DMX
0x8798249c2e607446efb7ad49ec89dd1865ff4272,18,SushiBar,xSUSHI
0xd7c49cee7e9188cca6ad8ff264c1da2e69d4cf3b,18,NXM,NXM
0xc76fb75950536d98fa62ea968e1d6b45ffea2a55,18,COL,COL
0xa9b1eb5908cfc3cdf91f9b8b3a74108598009096,18,BounceToken,Auction
0x5b7533812759b45c2b44c19e320ba2cd2681b542,8,SingularityNETToken,AGIX
0x6595b8fd9c920c81500dca94e53cdc712513fb1f,18,Olyseum,OLY
0xc52c326331e9ce41f04484d3b5e5648158028804,18,ZENExchangeToken,ZCX
0x3431f91b3a388115f00c5ba9fdb899851d005fb5,18,GeroWallet,GERO
0x474021845c4643113458ea4414bdb7fb74a01a77,18,UnoRe,UNO
0xcaaa93712bdac37f736c323c93d4d5fdefcc31cc,18,CryptalDash,CRD
0x1f8a626883d7724dbd59ef51cbd4bf1cf2016d13,18,Jigstack,STAK
0x33349b282065b0284d756f0577fb39c158f935e6,18,MapleToken,MPL
0x6243d8cea23066d098a15582d81a598b4e8391f4,18,FlexUngovernanceToken,FLX
0xc53342fd7575f572b0ff4569e31941a5b821ac76,18,ETHVolatilityIndex,ETHV
0x3a707d56d538e85b783e8ce12b346e7fb6511f90,18,InverseETHVolatilityIndex,iETHV
0x51b0bcbeff204b39ce792d1e16767fe6f7631970,18,BTCVolatilityIndex,BTCV
0x2590f1fd14ef8bb0a46c7a889c4cbc146510f9c3,18,InverseBTCVolatilityIndex,iBTCV
0x69fa8e7f6bf1ca1fb0de61e1366f7412b827cc51,9,EnreachDAO,NRCH
0xfb7b4564402e5500db5bb6d63ae671302777c75a,18,DEXTools,DEXT
0x853bb55c1f469902f088a629db8c8803a9be3857,18,Stable1INCH,one1INCH
0x75d12e4f91df721fafcae4c6cd1d5280381370ac,9,Mybu,MYOBU
0xdddddd4301a082e62e84e43f474f044423921918,18,DeversiFiToken,DVF
0x6b4c7a5e3f0b99fcd83e9c089bddd6c7fce5c611,18,Million,MM2
0xc581b735a1688071a1746c968e0798d642ede491,6,EuroTether,EURT
0x72e5390edb7727e3d4e3436451dadaff675dbcc0,12,HanuYokia,HANU
0x011864d37035439e078d64630777ec518138af05,18,ZerogokiToken,REI
0x76417e660df3e5c90c0361674c192da152a806e4,18,ZerogokiUSD,zUSD
0x441761326490cacf7af299725b6292597ee822c2,18,UNFI,UNFI
0x8e6cd950ad6ba651f6dd608dc70e5886b1aa6b24,18,StarLink,STARL
0x70d2b7c19352bb76e4409858ff5746e500f2b67c,18,PawtocolNetworkUPIToken,UPI
0x382f0160c24f5c515a19f155bac14d479433a407,9,KleeKai,KLEE
0x1559fa1b8f28238fd5d76d9f434ad86fd20d1559,18,Eden,EDEN
0x1e6bb68acec8fefbd87d192be09bb274170a0548,9,AaveinterestbearingAMPL,aAMPL
0xe1030b48b2033314979143766d7dc1f40ef8ce11,18,ThePeoplesCoin,PEEPS
0x515d7e9d75e2b76db60f8a051cd890eba23286bc,18,Governor,GDAO
0xb26631c6dda06ad89b93c71400d25692de89c068,18,Minds,MINDS
0x43f11c02439e2736800433b4594994bd43cd066d,9,FLOKI,FLOKI
0x5166e09628b696285e3a151e84fb977736a83575,18,VolatilityProtocolToken,VOL
0x06f3c323f0238c72bf35011071f2b5b7f43a054c,18,MASQ,MASQ
0xc4f6e93aeddc11dc22268488465babcaf09399ac,18,hiDollar,HI
0x106552c11272420aad5d7e94f8acab9095a6c952,9,KeanuInu,KEANU
0xddb3422497e61e13543bea06989c0789117555c5,18,COTIToken,COTI
0x92d6c1e31e14520e676a687f0a93788b716beff5,18,dYdX,DYDX
0xa4ef4b0b23c1fc81d3f9ecf93510e64f58a4a016,18,1MILNFT,1MIL
0xeb494890465f49c2b94457d9b61811392e5b1fea,9,SLINKLABS,SLAB
0x25e1474170c4c0aa64fa98123bdc8db49d7802fa,18,Bidao,BID
0x61107a409fffe1965126aa456af679719695c69c,18,UmiToken,UMI
0x738865301a9b7dd80dc3666dd48cf034ec42bdda,8,AgorasToken,AGRS
0x0af55d5ff28a3269d69b98680fd034f115dd53ac,8,BankSocial,BSL
0x4297394c20800e8a38a619a243e9bbe7681ff24e,18,HotCrossToken,HOTCROSS
0xd85ad783cc94bd04196a13dc042a3054a9b52210,18,TribeOne,HAKA
0x8b3192f5eebd8579568a2ed41e6feb402f93f73f,9,SaitamaInu,SAITAMA
0x62dc4817588d53a056cbbd18231d91ffccd34b2a,18,DeHive.finance,DHV
0x19042021329fddcfbea5f934fb5b2670c91f7d20,6,TakeMyMuffin,TMM
0x15b7c0c907e4c6b9adaaaabc300c08991d6cea05,18,GelatoNetworkToken,GEL
0xa64dfe8d86963151e6496bee513e366f6e42ed79,9,GokuInu,GOKU
0xe94b97b6b43639e238c851a7e693f50033efd75c,18,RainbowToken,RNBW
0x99d8a9c45b2eca8864373a26d1459e3dff1e17f3,18,MagicInternetMoney,MIM
0x090185f2135308bad17527004364ebcc2d37e5f6,18,SpellToken,SPELL
0xde5ed76e7c05ec5e4572cfc88d1acea165109e44,18,DEUS,DEUS
0xf009f5531de69067435e32c4b9d36077f4c4a673,18,Unvest,UNV
0xc221b7e65ffc80de234bbb6667abdd46593d34f0,18,WrappedCentrifuge,wCFG
0x7f39c581f595b53c5cb19bd0b3f8da6c935e2ca0,18,WrappedliquidstakedEther2.0,wstETH
0xf50a07e4ff052a14f3f608da8936d8ae0ed5be50,9,FlokiPup,FLOKIPUP
0x636484a1c41e88e3fc7c99248ca0b3c3a844ab86,9,GogetaInu,GOGETA
0xaecc217a749c2405b5ebc9857a16d58bdc1c367f,9,Pawthereum,PAWTH
0x949d48eca67b17269629c7194f4b727d4ef9e5d6,18,MeritCircle,MC
0xd6014ea05bde904448b743833ddf07c3c7837481,18,SynthiBTC,iBTC
0xa9859874e1743a32409f75bb11549892138bba1e,18,SynthiETH,iETH
0xf48e200eaf9906362bb1442fca31e0835773b8b4,18,SynthsAUD,sAUD
0xfe18be6b3bd88a2d2a7f928d00292e7a9963cfc6,18,SynthsBTC,sBTC
0xe1afe1fd76fd88f78cbf599ea1846231b8ba3b6b,18,SynthsDEFI,sDEFI
0x5e74c9036fb86bd7ecdcb084a0673efc32ea31cb,18,SynthsETH,sETH
0xd71ecff9342a5ced620049e616c5035f1db98620,18,SynthsEUR,sEUR
0x514cdb9cd8a2fb2bdcf7a3b8ddd098caf466e548,9,RedPanda,REDPANDA
0xc50ef449171a51fbeafd7c562b064b6471c36caa,9,ZombieInu,ZINU
0x0c3685559af6f3d20c501b1076a8056a0a14426a,9,miniSAITAMA,miniSAITAMA
0x24e89bdf2f65326b94e36978a7edeac63623dafa,18,TigerKing,TKING
0xc1bfccd4c29813ede019d00d2179eea838a67703,9,LuffyInu,LUFFY
0xaaaaaa20d9e0e2461697782ef11675f668207961,18,Aurora,AURORA
0xed0889f7e1c7c7267407222be277e1f1ef4d4892,18,Melalie,MEL
0xc18360217d8f7ab5e7c516566761ea12ce7f9d72,18,EthereumNameService,ENS
0xf14b9adf84812ba463799357f4dc716b4384010b,8,PensionPlan,PP
0xfb40e79e56cc7d406707b66c4fd175e07eb2ae3c,9,ROTTSCHILD.com,ROTTS
0x28c5805b64d163588a909012a628b5a03c1041f9,9,CHOPPERINU,CHOPPER
0x37fe0f067fa808ffbdd12891c0858532cfe7361d,18,Civilization,CIV
0xf32aa187d5bc16a2c02a6afb7df1459d0d107574,18,HachikoInu,Inu
0xc08512927d12348f6620a698105e1baac6ecd911,6,GMOJPY,GYEN
0xcafe001067cdef266afb7eb5a286dcfd277f3de5,18,ParaSwap,PSP
0xbb1ee07d6c7baeb702949904080eb61f5d5e7732,18,Dogey-Inu,DINU
0x582d872a1b094fc48f5de31d3b73f2d9be47def1,9,WrappedTONCoin,TONCOIN
0x95a1796437bad6502d1c1cce165cd76e522409a9,18,OfficialCryptoCowboyToken,OCCT
0xe87e15b9c7d989474cb6d8c56b3db4efad5b21e8,18,HokkaidoInu,HOKK
0x4104b135dbc9609fc1a9490e61369036497660c8,18,APWineToken,APW
0xae78736cd615f374d3085123a210448e74fc6393,18,RocketPoolETH,rETH
0xafbf03181833ab4e8dec24d708a2a24c2baaa4a4,9,MandoX,MANDOX
0x1fbd3df007eb8a7477a1eab2c63483dcc24effd6,18,ScaleSwapToken,SCA
0xb1a88c33091490218965787919fcc9862c1798ee,9,StudioShibli,SHIBLI
0x8037b1b69d6fa63a9cc053c25f7e168e6e6d857a,18,PartsofFourCoin,P4C
0xcfeaead4947f0705a14ec42ac3d44129e1ef3ed5,8,Notional,NOTE
0x41a3dba3d677e573636ba691a70ff2d606c29666,18,GoBlankToken,BLANK
0xc48b4814faed1ccc885dd6fde62a6474aecbb19a,9,CoinMerge,CMERGE
0x8254e26e453eb5abd29b3c37ac9e8da32e5d3299,18,RBX,RBX"""
