"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Key = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
const perms = require("./private/perms");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @param statement The policy statement to add.
     * @param allowNoOp If this is set to `false` and there is no policy defined (i.e. external key), the operation will fail. Otherwise, it will no-op.
     * @stability stable
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return { statementAdded: false };
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (this.trustAccountIdentities && !crossEnvironment) {
            return iam.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return iam.Grant.addToPrincipalAndResource({
                ...grantOptions,
                // if the key is used in a cross-environment matter,
                // we can't access the Key ARN (they don't have physical names),
                // so fall back to using '*'. ToDo we need to make this better... somehow
                resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
                resourcePolicyPrincipal: principal,
            });
        }
    }
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, ...perms.DECRYPT_ACTIONS);
    }
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, ...perms.ENCRYPT_ACTIONS);
    }
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, ...[...perms.DECRYPT_ACTIONS, ...perms.ENCRYPT_ACTIONS]);
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        const grantPrincipal = grantee.grantPrincipal;
        if (!(grantPrincipal instanceof constructs_1.Construct)) {
            return undefined;
        }
        // this logic should only apply to newly created
        // (= not imported) resources
        if (!this.principalIsANewlyCreatedResource(grantPrincipal)) {
            return undefined;
        }
        // return undefined;
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantPrincipal);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    principalIsANewlyCreatedResource(principal) {
        // yes, this sucks
        // this is just a temporary stopgap to stem the bleeding while we work on a proper fix
        return principal instanceof iam.Role ||
            principal instanceof iam.User ||
            principal instanceof iam.Group;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * Defines a KMS key.
 *
 * @stability stable
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        super(scope, id);
        const defaultKeyPoliciesFeatureEnabled = core_1.FeatureFlags.of(this).isEnabled(cxapi.KMS_DEFAULT_KEY_POLICIES);
        this.policy = (_a = props.policy) !== null && _a !== void 0 ? _a : new iam.PolicyDocument();
        if (defaultKeyPoliciesFeatureEnabled) {
            if (props.trustAccountIdentities === false) {
                throw new Error('`trustAccountIdentities` cannot be false if the @aws-cdk/aws-kms:defaultKeyPolicies feature flag is set');
            }
            this.trustAccountIdentities = true;
            // Set the default key policy if one hasn't been provided by the user.
            if (!props.policy) {
                this.addDefaultAdminPolicy();
            }
        }
        else {
            this.trustAccountIdentities = (_b = props.trustAccountIdentities) !== null && _b !== void 0 ? _b : false;
            if (this.trustAccountIdentities) {
                this.addDefaultAdminPolicy();
            }
            else {
                this.addLegacyAdminPolicy();
            }
        }
        let pendingWindowInDays;
        if (props.pendingWindow) {
            pendingWindowInDays = props.pendingWindow.toDays();
            if (pendingWindowInDays < 7 || pendingWindowInDays > 30) {
                throw new Error(`'pendingWindow' value must between 7 and 30 days. Received: ${pendingWindowInDays}`);
            }
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
            pendingWindowInDays: pendingWindowInDays,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        ((_c = props.admins) !== null && _c !== void 0 ? _c : []).forEach((p) => this.grantAdmin(p));
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope the construct that will "own" the imported key.
     * @param id the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     * @stability stable
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).parseArn(keyArn).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    /**
     * Grant admins permissions using this key to the given principal.
     *
     * Key administrators have permissions to manage the key (e.g., change permissions, revoke), but do not have permissions
     * to use the key in cryptographic operations (e.g., encrypt, decrypt).
     *
     * @stability stable
     */
    grantAdmin(grantee) {
        return this.grant(grantee, ...perms.ADMIN_ACTIONS);
    }
    /**
     * Adds the default key policy to the key. This policy gives the AWS account (root user) full access to the CMK,
     * which reduces the risk of the CMK becoming unmanageable and enables IAM policies to allow access to the CMK.
     * This is the same policy that is default when creating a Key via the KMS API or Console.
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     */
    addDefaultAdminPolicy() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Grants the account admin privileges -- not full account access -- plus the GenerateDataKey action.
     * The GenerateDataKey action was added for interop with S3 in https://github.com/aws/aws-cdk/issues/3458.
     *
     * This policy is discouraged and deprecated by the '@aws-cdk/aws-kms:defaultKeyPolicies' feature flag.
     *
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @deprecated
     */
    addLegacyAdminPolicy() {
        // This is equivalent to `[...perms.ADMIN_ACTIONS, 'kms:GenerateDataKey']`,
        // but keeping this explicit ordering for backwards-compatibility (changing the ordering causes resource updates)
        const actions = [
            'kms:Create*',
            'kms:Describe*',
            'kms:Enable*',
            'kms:List*',
            'kms:Put*',
            'kms:Update*',
            'kms:Revoke*',
            'kms:Disable*',
            'kms:Get*',
            'kms:Delete*',
            'kms:ScheduleKeyDeletion',
            'kms:CancelKeyDeletion',
            'kms:GenerateDataKey',
            'kms:TagResource',
            'kms:UntagResource',
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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