""" Bid for Game
    Acol OverCallersBid module
"""

import inspect
from .bidding import Pass, Double, Call
from .hand import Hand

inspection = inspect.currentframe


class OverCallersThirdBid(Hand):
    """BfG OverCallersRebid class."""
    def __init__(self, hand_cards, board):
        super(OverCallersThirdBid, self).__init__(hand_cards, board)
        self.trace = 0

    def suggested_bid(self):
        """Direct control to relevant method and return a Bid object."""
        my_first_bid = Call(self.bid_history[-8])
        my_second_bid = Call(self.bid_history[-4])
        partners_first_bid = Call(self.bid_history[-6])
        partners_last_bid = Call(self.bid_history[-2])
        if (my_first_bid.is_pass and my_second_bid._is_pass and partners_last_bid.is_pass):
            bid = Pass('2835')
        elif self._advancer_supports_major_after_stayman():
            if self.hcp >= 16:
                bid = self.next_level_bid(self.bid_two.denomination, '2801')
            else:
                bid = Pass('2802')
        elif self._can_show_second_major_after_stayman():
            bid = self.next_level_bid(self.spade_suit, '2803')
        elif self._has_thirteen_points_advancer_supports_second_suit():
            bid = self.next_level_bid(self.bid_two.denomination, '2804')
        elif self._has_fifteen_points_advancer_bids_two_suits():
            bid = self._fifteen_points_advancer_bids_two_suits()
        elif self._is_very_strong_and_five_four():
            if self.second_suit not in self.opponents_suits:
                suit = self.second_suit
            else:
                suit = self.longest_suit
            bid = self.next_level_bid(suit, '2805')
        elif self._is_strong_and_advancer_bids():
            bid = self._strong_hand()
        elif self._three_suits_have_been_bid():
            bid = self._three_suits_bid()
        elif self._has_two_card_support_for_advancers_major():
            bid = self.next_level_bid(self.partner_bid_one.denomination, '2806')
        elif self._has_six_card_suit_advancer_repeats_suit():
            bid = self.next_level_bid(self.longest_suit, '2807')
        elif self._is_strong_advancer_bids_nt_after_stayman():
            bid = self.nt_bid(3, '2808')
        elif partners_last_bid.is_game:
            bid = Pass('2826')
        elif partners_last_bid.denomination.name == self.longest_suit.name:
            bid = Pass('2828')
        elif (my_first_bid.name == '1NT' and partners_last_bid.name == '2NT' and
                self.hcp <= 15):
            bid = Pass('2829')
        elif (my_first_bid.name == '1NT' and partners_last_bid.name == '2NT' and
                self.hcp <= 15):
            bid = Pass('2829')
        elif (not partners_last_bid.is_pass and (partners_last_bid.denomination == my_first_bid.denomination or
                    partners_last_bid.denomination == my_second_bid.denomination)):
            bid = Pass('2830')
        elif (not partners_last_bid.is_pass and partners_first_bid.denomination == partners_last_bid.denomination and
                self.suit_length(partners_first_bid.denomination) <= 2):
            bid = Pass('2832')
        elif (not partners_last_bid.is_pass and partners_first_bid.denomination == partners_last_bid.denomination and
                self.hcp < 16):
            bid = Pass('2833')
        elif partners_last_bid.level == 2:
            bid = Pass('2834')
        elif partners_last_bid.is_pass:
            bid = Pass('2831')
        elif my_first_bid.is_double and my_second_bid.is_double and partners_first_bid.is_pass:
            bid = Pass('2836')
        else:
            bid = Pass('2809')
        self.tracer(__name__, inspection(), bid, self.trace)
        return bid

    def _strong_hand(self):
        """Return bid with strong hand."""
        if self.bid_one.is_double and self.bid_two.is_double:
            bid = Pass('2810')
        elif self._support_for_advancer():
            bid = self.next_level_bid(self.advancer_bid_one.denomination, '2811')
        elif self.is_balanced and self.nt_level <= 3:
            bid = self.next_nt_bid('2812')
        elif self.five_five and self.second_suit not in self.opponents_suits:
            bid = self.next_level_bid(self.second_suit, '2813')
        elif self.five_four and self.second_suit not in self.opponents_suits:
            bid = self.next_level_bid(self.second_suit, '2814')
        elif (self.partner_bid_one.denomination != self.partner_last_bid.denomination and
                self.suit_length(self.second_suit) >= 5 and
                self.second_suit not in self.opponents_suits):
            bid = self.next_level_bid(self.second_suit, '2827')
        elif (self.partner_bid_one.denomination != self.partner_last_bid.denomination and
                self.longest_suit not in self.opponents_suits):
            bid = self.next_level_bid(self.longest_suit, '2815')
        elif (not self.partner_bid_one.is_pass and
                self.suit_length(self.partner_bid_one.denomination) >= 3):
            bid = self.next_level_bid(self.partner_bid_one.denomination, '2816')
        else:
            bid = Pass('2817')
        self.tracer(__name__, inspection(), bid, self.trace)
        return bid

    def _three_suits_bid(self):
        """Return bid after 3 suits have been bid."""
        suit_to_bid = self._select_suit_after_three_suits_bid()
        if self._suit_is_minor(suit_to_bid):
            bid = self.next_nt_bid('2818')
        elif suit_to_bid not in self.opponents_suits:
            bid = self.next_level_bid(suit_to_bid, '2821')
        else:
            bid = Pass('2820')
        self.tracer(__name__, inspection(), bid, self.trace)
        return bid

    def _fifteen_points_advancer_bids_two_suits(self):
        """Return bid if strong and advancer bids two suits."""
        if self.bidding_above_game and self.can_double:
            bid = Double('2825')
        elif self._advancer_has_passed_after_double():
            bid = Pass('2822')
        elif (self.suit_length(self.opener_bid_one.denomination) >= 3 and
                self.suit_points(self.opener_bid_one.denomination) >= 5):
            bid = Pass('2823')
        else:
            suit = self._suit_preference()
            bid = self.next_level_bid(suit, '2824')
        self.tracer(__name__, inspection(), bid, self.trace)
        return bid

    # Various utility functions

    def _suit_preference(self):
        """Return preference (if any) for partner's suits."""
        suit = None
        if (self.partner_bid_one.denomination.is_major or
                self.partner_bid_one.denomination.is_minor):
            suit_one = self.partner_bid_one.denomination
            if self.suit_length(suit_one) >= 3:
                suit = suit_one
        if (self.partner_last_bid.denomination.is_major or
                self.partner_last_bid.denomination.is_minor):
            suit_two = self.partner_last_bid.denomination
            if (self.suit_length(suit_two) >= 4 or
                    not suit):
                suit = suit_two
        if not suit:
            suit = self.no_trumps
        return suit

    def _select_suit_after_three_suits_bid(self):
        if self.suit_length(self.partner_last_bid.denomination) >= 4:
            suit_to_bid = self.partner_last_bid.denomination
        elif self.five_five:
            suit_to_bid = self.second_suit
            if suit_to_bid in self.opponents_suits:
                suit_to_bid = self.longest_suit
        else:
            suit_to_bid = self.longest_suit
        return suit_to_bid

    # Various boolean functions

    def _advancer_supports_major_after_stayman(self):
        """Return True if advancer supports major after Stayman."""
        result = (self.bid_one.name == '1NT' and
                  self.partner_bid_one.name == '2C' and
                  self.partner_last_bid.denomination == self.bid_two.denomination and
                  self.partner_last_bid.level == 3)
        return result

    def _can_show_second_major_after_stayman(self):
        """Return True if got 4 spades and 4 hearts."""
        result = (self.bid_one.name == '1NT' and
                  self.partner_bid_one.name == '2C' and
                  self.bid_two.name == '2H' and
                  self.partner_last_bid.name == '2NT' and
                  self.spades >= 4 and
                  self.spade_suit not in self.opponents_suits)
        return result

    def _has_thirteen_points_advancer_supports_second_suit(self):
        """Return True if 13 points and advancer support second suit."""
        result = (self.partner_last_bid.denomination == self.bid_two.denomination and
                  self.partner_last_bid.is_suit_call and
                  not self.partner_last_bid.is_game and
                  self.hcp >= 13 and
                  not self.singleton_honour)
        return result

    def _has_fifteen_points_advancer_bids_two_suits(self):
        """Return True if fifteen points and advancer bids two suits."""
        result = (self.partner_bid_one.denomination != self.partner_last_bid.denomination and
                  not self.bid_one.is_nt and
                  not self.partner_last_bid.is_nt and
                  not self.partner_last_bid.is_game and
                  self._suit_preference().is_suit and
                  self.hcp >= 15)
        return result

    def _is_very_strong_and_five_four(self):
        """Return True if strong and 5/4."""
        result = (self.hcp >= 19 and
                  self.five_four_or_better and
                  not self.bid_two.is_nt and
                  not self.bidding_above_game and
                  self.longest_suit not in self.opponents_suits)
        return result

    def _is_strong_and_advancer_bids(self):
        """Return True if strong and advancer bids."""
        result = (self.hcp >= 16 and
                  not self.partner_last_bid.is_pass and
                  not self.partner_last_bid.is_game)
        return result

    def _three_suits_have_been_bid(self):
        """Return True if 3 suits bid."""
        my_suits = [self.bid_one.denomination, self.bid_two.denomination]
        result = (self.partner_bid_one.denomination != self.partner_last_bid.denomination and
                  self.partner_bid_one.is_suit_call and
                  self.partner_last_bid.level == 3 and
                  self.bid_one.denomination != self.partner_last_bid.denomination and
                  not self.partner_last_bid.is_game and
                  self.partner_last_bid.denomination not in my_suits)
        return result

    def _advancer_has_passed_after_double(self):
        """Return True if advancer passes after double."""
        result = (self.bid_one.is_double and
                  (self.partner_bid_one.is_pass or
                   self.partner_last_bid.is_pass))
        return result

    def _has_two_card_support_for_advancers_major(self):
        """Return True if 2 card support for advancers major."""
        result = (self.partner_bid_one.denomination == self.partner_last_bid.denomination and
                  self.partner_bid_one.is_value_call and
                  self.partner_bid_one.denomination.is_major and
                  self.suit_length(self.partner_bid_one.denomination) >= 2 and
                  self.hcp >= 15 and
                  not self.partner_last_bid.is_game)
        return result

    def _has_six_card_suit_advancer_repeats_suit(self):
        """Return True if 6 card suit and advancer repeats their suit."""
        result = (self.partner_bid_one.denomination == self.partner_last_bid.denomination and
                  self.partner_bid_one.is_value_call and
                  self.shape[0] >= 6 and
                  not self.partner_last_bid.is_game and
                  self.longest_suit not in self.opponents_suits and
                  self.hcp >= 12)
        return result

    def _is_strong_advancer_bids_nt_after_stayman(self):
        """Return True if strong and advancer bids NT after stayman."""
        result = (self.bid_one.name == '1NT' and
                  self.partner_bid_one.name == '2C' and
                  self.partner_last_bid.name == '2NT' and
                  self.hcp >= 17 and
                  self.nt_level <= 3)
        return result

    def _suit_is_minor(self, suit_to_bid):
        """Return True if suit is minor."""
        result = (self.nt_level <= 3 and
                  (suit_to_bid.is_minor or
                   suit_to_bid in self.opponents_suits))
        return result

    def _support_for_advancer(self):
        """Return True if xxx."""
        if not self.advancer_bid_two:
            result = False
        else:
            result = (self.advancer_bid_one.denomination == self.advancer_bid_two.denomination and
                      self.suit_length(self.advancer_bid_one.denomination) >= 3)
        return result
