"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3CodeCommitBackup = void 0;
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
class S3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, computeType, repository } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'BackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [`echo "[===== Clone repository: ${repositoryName} =====]"`, `git clone "${repositoryCloneUrlHttp}"`],
                    },
                    build: {
                        commands: [
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            `zipfile="${repositoryName}_backup_\${dt}_UTC.tar.gz"`,
                            `echo "Compressing repository: ${repositoryName} into file: \${zipfile} and uploading to S3 bucket: ${backupBucket.bucketName}/${repositoryName}"`,
                            `tar -zcvf "\${zipfile}" "${repositoryName}/"`,
                            `aws s3 cp "\${zipfile}" "s3://${backupBucket.bucketName}/${repositoryName}/\${zipfile}"`,
                        ],
                    },
                },
            }),
        });
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.S3CodeCommitBackup = S3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,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