/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as lambda from '@aws-cdk/aws-lambda';
import * as s3 from '@aws-cdk/aws-s3';
import * as ec2 from '@aws-cdk/aws-ec2';
import { Construct } from '@aws-cdk/core';
/**
 * @summary The properties for the LambdaToS3 class.
 */
export interface LambdaToS3Props {
    /**
     * Existing instance of Lambda Function object, providing both this and `lambdaFunctionProps` will cause an error.
     *
     * @default - None
     */
    readonly existingLambdaObj?: lambda.Function;
    /**
     * Optional user provided props to override the default props for the Lambda function.
     *
     * @default - Default properties are used.
     */
    readonly lambdaFunctionProps?: lambda.FunctionProps;
    /**
     * Existing instance of S3 Bucket object, providing both this and `bucketProps` will cause an error.
     *
     * @default - None
     */
    readonly existingBucketObj?: s3.IBucket;
    /**
     * Optional user provided props to override the default props for the S3 Bucket.
     *
     * @default - Default props are used
     */
    readonly bucketProps?: s3.BucketProps;
    /**
     * Optional bucket permissions to grant to the Lambda function.
     * One or more of the following may be specified: "Delete", "Put", "Read", "ReadWrite", "Write".
     *
     * @default - Read/write access is given to the Lambda function if no value is specified.
     */
    readonly bucketPermissions?: string[];
    /**
     * An existing VPC for the construct to use (construct will NOT create a new VPC in this case)
     */
    readonly existingVpc?: ec2.IVpc;
    /**
     * Properties to override default properties if deployVpc is true
     */
    readonly vpcProps?: ec2.VpcProps;
    /**
     * Whether to deploy a new VPC
     *
     * @default - false
     */
    readonly deployVpc?: boolean;
    /**
     * Optional Name for the Lambda function environment variable set to the name of the bucket.
     *
     * @default - S3_BUCKET_NAME
     */
    readonly bucketEnvironmentVariableName?: string;
    /**
     * Optional user provided props to override the default props for the S3 Logging Bucket.
     *
     * @default - Default props are used
     */
    readonly loggingBucketProps?: s3.BucketProps;
    /**
     * Whether to turn on Access Logs for the S3 bucket with the associated storage costs.
     * Enabling Access Logging is a best practice.
     *
     * @default - true
     */
    readonly logS3AccessLogs?: boolean;
}
/**
 * @summary The LambdaToS3 class.
 */
export declare class LambdaToS3 extends Construct {
    readonly lambdaFunction: lambda.Function;
    readonly s3Bucket?: s3.Bucket;
    readonly s3LoggingBucket?: s3.Bucket;
    readonly vpc?: ec2.IVpc;
    readonly s3BucketInterface: s3.IBucket;
    /**
     * @summary Constructs a new instance of the LambdaToS3 class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToS3Props} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: LambdaToS3Props);
}
//# sourceMappingURL=index.d.ts.map