import unittest

from leeger.calculator.year_calculator.PointsScoredYearCalculator import (
    PointsScoredYearCalculator,
)
from leeger.enum.MatchupType import MatchupType
from leeger.model.league.Matchup import Matchup
from leeger.model.league.Week import Week
from leeger.model.league.Year import Year
from leeger.util.Deci import Deci
from test.helper.prototypes import getNDefaultOwnersAndTeams


class TestPointsScoredYearCalculator(unittest.TestCase):
    def test_getPointsScored_happyPath(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScored(year)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("3.6"), response[teams[0].id])
        self.assertEqual(Deci("7.5"), response[teams[1].id])

    def test_getPointsScored_noneIfNoGamesPlayed(self):
        owners, teams = getNDefaultOwnersAndTeams(3)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[1].id, teamBId=teams[2].id, teamAScore=1.2, teamBScore=2.5
        )
        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2])

        response = PointsScoredYearCalculator.getPointsScored(year, weekNumberEnd=1)

        self.assertIsInstance(response, dict)
        self.assertEqual(3, len(response.keys()))
        self.assertEqual(Deci("1.1"), response[teams[0].id])
        self.assertEqual(Deci("2.4"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])

    def test_getPointsScored_onlyPostSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScored(year, onlyPostSeason=True)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.5"), response[teams[0].id])
        self.assertEqual(Deci("5.1"), response[teams[1].id])

    def test_getPointsScored_onlyRegularSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScored(
            year, onlyRegularSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.3"), response[teams[0].id])
        self.assertEqual(Deci("4.9"), response[teams[1].id])

    def test_getPointsScored_onlyChampionshipIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2, week3])

        response = PointsScoredYearCalculator.getPointsScored(
            year, onlyChampionship=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.3"), response[teams[0].id])
        self.assertEqual(Deci("2.6"), response[teams[1].id])

    def test_getPointsScored_weekNumberStartGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScored(year, weekNumberStart=2)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.5"), response[teams[0].id])
        self.assertEqual(Deci("5.1"), response[teams[1].id])

    def test_getPointsScored_weekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScored(year, weekNumberEnd=2)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.3"), response[teams[0].id])
        self.assertEqual(Deci("4.9"), response[teams[1].id])

    def test_getPointsScored_weekNumberStartGivenAndWeekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup4 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.4,
            teamBScore=2.7,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])
        week4 = Week(weekNumber=4, matchups=[matchup4])

        year = Year(
            yearNumber=2000,
            teams=[teams[0], teams[1]],
            weeks=[week1, week2, week3, week4],
        )

        response = PointsScoredYearCalculator.getPointsScored(
            year, weekNumberStart=2, weekNumberEnd=3
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.5"), response[teams[0].id])
        self.assertEqual(Deci("5.1"), response[teams[1].id])

    def test_getPointsScoredPerGame_happyPath(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(year)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.2"), response[teams[0].id])
        self.assertEqual(Deci("2.5"), response[teams[1].id])

    def test_getPointsScoredPerGame_noneIfNoGamesPlayed(self):
        owners, teams = getNDefaultOwnersAndTeams(3)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[1].id, teamBId=teams[2].id, teamAScore=1.2, teamBScore=2.5
        )
        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2])

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, weekNumberEnd=1
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(3, len(response.keys()))
        self.assertEqual(Deci("1.1"), response[teams[0].id])
        self.assertEqual(Deci("2.4"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])

    def test_getPointsScoredPerGame_onlyPostSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, onlyPostSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.25"), response[teams[0].id])
        self.assertEqual(Deci("2.55"), response[teams[1].id])

    def test_getPointsScoredPerGame_onlyRegularSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, onlyRegularSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.15"), response[teams[0].id])
        self.assertEqual(Deci("2.45"), response[teams[1].id])

    def test_getPointsScoredPerGame_onlyChampionshipIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(4)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )
        matchup4 = Matchup(
            teamAId=teams[2].id,
            teamBId=teams[3].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3, matchup4])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2, week3])

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, onlyChampionship=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(4, len(response.keys()))
        self.assertEqual(Deci("1.3"), response[teams[0].id])
        self.assertEqual(Deci("2.6"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])
        self.assertIsNone(response[teams[3].id])

    def test_getPointsScoredPerGame_weekNumberStartGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, weekNumberStart=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.25"), response[teams[0].id])
        self.assertEqual(Deci("2.55"), response[teams[1].id])

    def test_getPointsScoredPerGame_weekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, weekNumberEnd=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.15"), response[teams[0].id])
        self.assertEqual(Deci("2.45"), response[teams[1].id])

    def test_getPointsScoredPerGame_weekNumberStartGivenAndWeekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup4 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.4,
            teamBScore=2.7,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])
        week4 = Week(weekNumber=4, matchups=[matchup4])

        year = Year(
            yearNumber=2000,
            teams=[teams[0], teams[1]],
            weeks=[week1, week2, week3, week4],
        )

        response = PointsScoredYearCalculator.getPointsScoredPerGame(
            year, weekNumberStart=2, weekNumberEnd=3
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("1.25"), response[teams[0].id])
        self.assertEqual(Deci("2.55"), response[teams[1].id])

    def test_getOpponentPointsScored_happyPath(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(year)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("7.5"), response[teams[0].id])
        self.assertEqual(Deci("3.6"), response[teams[1].id])

    def test_getOpponentPointsScored_noneIfNoGamesPlayed(self):
        owners, teams = getNDefaultOwnersAndTeams(3)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[1].id, teamBId=teams[2].id, teamAScore=1.2, teamBScore=2.5
        )
        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2])

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, weekNumberEnd=1
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(3, len(response.keys()))
        self.assertEqual(Deci("2.4"), response[teams[0].id])
        self.assertEqual(Deci("1.1"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])

    def test_getOpponentPointsScored_onlyPostSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, onlyPostSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("5.1"), response[teams[0].id])
        self.assertEqual(Deci("2.5"), response[teams[1].id])

    def test_getOpponentPointsScored_onlyRegularSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, onlyRegularSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("4.9"), response[teams[0].id])
        self.assertEqual(Deci("2.3"), response[teams[1].id])

    def test_getOpponentPointsScored_onlyChampionshipIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, onlyChampionship=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.6"), response[teams[0].id])
        self.assertEqual(Deci("1.3"), response[teams[1].id])

    def test_getOpponentPointsScored_weekNumberStartGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, weekNumberStart=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("5.1"), response[teams[0].id])
        self.assertEqual(Deci("2.5"), response[teams[1].id])

    def test_getOpponentPointsScored_weekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, weekNumberEnd=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("4.9"), response[teams[0].id])
        self.assertEqual(Deci("2.3"), response[teams[1].id])

    def test_getOpponentPointsScored_weekNumberStartGivenAndWeekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup4 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.4,
            teamBScore=2.7,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])
        week4 = Week(weekNumber=4, matchups=[matchup4])

        year = Year(
            yearNumber=2000,
            teams=[teams[0], teams[1]],
            weeks=[week1, week2, week3, week4],
        )

        response = PointsScoredYearCalculator.getOpponentPointsScored(
            year, weekNumberStart=2, weekNumberEnd=3
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("5.1"), response[teams[0].id])
        self.assertEqual(Deci("2.5"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_happyPath(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(year)

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.5"), response[teams[0].id])
        self.assertEqual(Deci("1.2"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_noneIfNoGamesPlayed(self):
        owners, teams = getNDefaultOwnersAndTeams(3)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[1].id, teamBId=teams[2].id, teamAScore=1.2, teamBScore=2.5
        )
        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2])

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, weekNumberEnd=1
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(3, len(response.keys()))
        self.assertEqual(Deci("2.4"), response[teams[0].id])
        self.assertEqual(Deci("1.1"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])

    def test_getOpponentPointsScoredPerGame_onlyPostSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, onlyPostSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.55"), response[teams[0].id])
        self.assertEqual(Deci("1.25"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_onlyRegularSeasonIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, onlyRegularSeason=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.45"), response[teams[0].id])
        self.assertEqual(Deci("1.15"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_onlyChampionshipIsTrue(self):
        owners, teams = getNDefaultOwnersAndTeams(4)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.2,
            teamBScore=2.5,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.CHAMPIONSHIP,
        )
        matchup4 = Matchup(
            teamAId=teams[2].id,
            teamBId=teams[3].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3, matchup4])

        year = Year(yearNumber=2000, teams=teams, weeks=[week1, week2, week3])

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, onlyChampionship=True
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(4, len(response.keys()))
        self.assertEqual(Deci("2.6"), response[teams[0].id])
        self.assertEqual(Deci("1.3"), response[teams[1].id])
        self.assertIsNone(response[teams[2].id])
        self.assertIsNone(response[teams[3].id])

    def test_getOpponentPointsScoredPerGame_weekNumberStartGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, weekNumberStart=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.55"), response[teams[0].id])
        self.assertEqual(Deci("1.25"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_weekNumberEndGiven(self):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])

        year = Year(
            yearNumber=2000, teams=[teams[0], teams[1]], weeks=[week1, week2, week3]
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, weekNumberEnd=2
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.45"), response[teams[0].id])
        self.assertEqual(Deci("1.15"), response[teams[1].id])

    def test_getOpponentPointsScoredPerGame_weekNumberStartGivenAndWeekNumberEndGiven(
        self,
    ):
        owners, teams = getNDefaultOwnersAndTeams(2)

        matchup1 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.1, teamBScore=2.4
        )
        matchup2 = Matchup(
            teamAId=teams[0].id, teamBId=teams[1].id, teamAScore=1.2, teamBScore=2.5
        )
        matchup3 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.3,
            teamBScore=2.6,
            matchupType=MatchupType.PLAYOFF,
        )
        matchup4 = Matchup(
            teamAId=teams[0].id,
            teamBId=teams[1].id,
            teamAScore=1.4,
            teamBScore=2.7,
            matchupType=MatchupType.PLAYOFF,
        )

        week1 = Week(weekNumber=1, matchups=[matchup1])
        week2 = Week(weekNumber=2, matchups=[matchup2])
        week3 = Week(weekNumber=3, matchups=[matchup3])
        week4 = Week(weekNumber=4, matchups=[matchup4])

        year = Year(
            yearNumber=2000,
            teams=[teams[0], teams[1]],
            weeks=[week1, week2, week3, week4],
        )

        response = PointsScoredYearCalculator.getOpponentPointsScoredPerGame(
            year, weekNumberStart=2, weekNumberEnd=3
        )

        self.assertIsInstance(response, dict)
        self.assertEqual(2, len(response.keys()))
        self.assertEqual(Deci("2.55"), response[teams[0].id])
        self.assertEqual(Deci("1.25"), response[teams[1].id])
