"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
// import { Construct } from 'constructs';
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const statement = require("cdk-iam-floyd");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket.
 *
 * This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 *
 * @stability stable
 */
class ProwlerAudit extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(parent, id, props) {
        var _b, _c, _d, _e, _f;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.5.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: core_1.RemovalPolicy.DESTROY });
        const preBuildCommands = [];
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            const prowlerFilename = 'prowler-exemptions.txt';
            preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            if (props.allowlist.isZipArchive) {
                // this isn't as simple as just unzipping.. we need to know the filename(s?) that are in there
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            preBuildCommands.push('ls prowler'); // remove this after testing
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: core_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    ADDITIONAL_S3_ARGS: { value: (_d = props === null || props === void 0 ? void 0 : props.additionalS3CopyArgs) !== null && _d !== void 0 ? _d : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/toniblyx/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_f = prowlerBuild.role) === null || _f === void 0 ? void 0 : _f.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: core_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new core_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: core_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "1.120.1" };
//# sourceMappingURL=data:application/json;base64,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