"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
// import { expect as expectCDK, haveResource } from '@aws-cdk/assert';
// import { LambdaToElasticachememcached, LambdaToElasticachememcachedProps } from "../lib";
// import * as lambda from '@aws-cdk/aws-lambda';
// import * as cdk from "@aws-cdk/core";
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const lib_1 = require("../lib");
const testPort = 12321;
const testFunctionName = "something-unique";
const testClusterName = "something-else";
test("When provided a VPC, does not create a second VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
});
test("When provided an existingCache, does not create a second cache", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc, testPort);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingCache,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
    });
    expect(stack).toCountResources("AWS::ElastiCache::CacheCluster", 1);
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        Port: testPort,
    });
});
test("When provided an existingFunction, does not create a second function", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: ".handler",
        vpc: existingVpc,
        functionName: testFunctionName,
    });
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingLambdaObj: existingFunction,
    });
    expect(stack).toCountResources("AWS::Lambda::Function", 1);
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test custom environment variable name", () => {
    const stack = new cdk.Stack();
    const testEnvironmentVariableName = "CUSTOM_CLUSTER_NAME";
    new lib_1.LambdaToElasticachememcached(stack, "test-construct", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
        cacheEndpointEnvironmentVariableName: testEnvironmentVariableName,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CUSTOM_CLUSTER_NAME: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
    });
});
test("Test setting custom function properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
            functionName: testFunctionName,
        },
    });
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test setting custom cache properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
        cacheProps: {
            clusterName: testClusterName,
        },
    });
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        ClusterName: testClusterName,
    });
});
test("Test setting custom VPC properties", () => {
    const stack = new cdk.Stack();
    const testCidrBlock = "192.168.0.0/16";
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
        vpcProps: {
            cidr: testCidrBlock,
        },
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: testCidrBlock,
    });
});
test("Test all default values", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
    });
    expect(stack).toCountResources("AWS::Lambda::Function", 1);
    expect(stack).toCountResources("AWS::ElastiCache::CacheCluster", 1);
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CACHE_ENDPOINT: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
        Handler: ".handler",
        Runtime: "nodejs14.x",
    });
    // All values taken from elasticache-defaults.ts
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        CacheNodeType: "cache.t3.medium",
        Engine: "memcached",
        NumCacheNodes: 2,
        Port: 11222,
        AZMode: "cross-az",
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Test for the proper self referencing security group', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: ".handler",
        },
        cacheProps: {
            port: 22223
        }
    });
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: 22223,
        ToPort: 22223,
        GroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
    });
});
// test('', () => {});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existing function and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: ".handler",
        vpc: existingVpc,
    });
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingLambdaObj: existingFunction,
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existingCache and cacheProps", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            existingVpc,
            cacheProps: {
                numCacheNodes: 4,
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("Cannot specify existingCache and cacheProps");
});
test("Test error from trying to launch Redis", () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            cacheProps: {
                numCacheNodes: 4,
                engine: "redis",
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("This construct can only launch memcached clusters");
});
//# sourceMappingURL=data:application/json;base64,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