import json
from typing import Dict, Any, List, Optional
from web3 import Web3
from web3.contract import Contract

from ..exceptions import ContractError
from ..utils.web3_utils import Web3Utils

class ContractManager:
    """Manager for smart contract interactions"""
    
    def __init__(self, web3_manager, contract_address: Optional[str] = None, abi: Optional[List[Dict]] = None):
        self.web3_manager = web3_manager
        self.contract_address = contract_address
        self.abi = abi
        self.contract = None
        
        if contract_address and abi:
            self.load_contract(contract_address, abi)
    
    def load_contract(self, contract_address: str, abi: List[Dict]) -> None:
        """Load a smart contract"""
        self.contract_address = contract_address
        self.abi = abi
        self.contract = self.web3_manager.web3.eth.contract(
            address=Web3.to_checksum_address(contract_address),
            abi=abi
        )
    
    def load_contract_from_json(self, contract_address: str, abi_path: str) -> None:
        """Load contract from JSON ABI file"""
        with open(abi_path, 'r') as f:
            abi = json.load(f)
        self.load_contract(contract_address, abi)
    
    def call_function(self, function_name: str, *args, **kwargs) -> Any:
        """Call a contract view function"""
        if not self.contract:
            raise ContractError("No contract loaded")
        
        function = getattr(self.contract.functions, function_name)
        return function(*args, **kwargs).call(**kwargs)
    
    def send_transaction(self, function_name: str, *args, 
                        gas_limit: Optional[int] = None, 
                        gas_price: Optional[int] = None,
                        value: int = 0) -> str:
        """Send a transaction to a contract function"""
        if not self.contract:
            raise ContractError("No contract loaded")
        
        if self.web3_manager.account is None:
            raise ContractError("No account set for transaction")
        
        function = getattr(self.contract.functions, function_name)
        built_tx = function(*args).build_transaction({
            'from': self.web3_manager.account.address,
            'chainId': self.web3_manager.chain_id,
            'gas': gas_limit or 200000,
            'gasPrice': gas_price or self.web3_manager.web3.eth.gas_price,
            'nonce': self.web3_manager.web3.eth.get_transaction_count(
                self.web3_manager.account.address
            ),
            'value': value
        })
        
        signed_txn = self.web3_manager.web3.eth.account.sign_transaction(
            built_tx, self.web3_manager.account.key
        )
        tx_hash = self.web3_manager.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return tx_hash.hex()
    
    def get_events(self, event_name: str, from_block: int = 0, to_block: str = 'latest') -> List[Dict]:
        """Get contract events"""
        if not self.contract:
            raise ContractError("No contract loaded")
        
        event = getattr(self.contract.events, event_name)
        return event.get_logs(fromBlock=from_block, toBlock=to_block)