import os
import json
import time
from typing import Optional, Dict, Any, List, Tuple, Union
from web3 import Web3
from web3.middleware import geth_poa_middleware
from web3.gas_strategies.time_based import medium_gas_price_strategy
from eth_account import Account
from eth_account.signers.local import LocalAccount
import logging

from ..exceptions import RubyWeb3Error, TransactionError
from ..models.transactions import Transaction, TokenTransfer, GasEstimate
from ..utils.converter import RubyConverter
from ..utils.formatter import ResponseFormatter
from ..utils.security import validate_private_key
from ..utils.validators import validate_eth_amount, validate_gas_params

class RubyWeb3Manager:
    """
    Comprehensive Web3 manager for Ruby-style operations with POA blockchain
    """
    
    def __init__(self, provider_url: str = "https://bridge-rpc.rubyscan.io", 
                 private_key: Optional[str] = None, 
                 chain_id: int = 18359,
                 timeout: int = 30):
        """
        Initialize the Web3 manager for Ruby blockchain
        
        Args:
            provider_url: Web3 provider URL
            private_key: Private key for transaction signing
            chain_id: Chain ID for the network
            timeout: Request timeout in seconds
        """
        self.web3 = Web3(Web3.HTTPProvider(provider_url, request_kwargs={'timeout': timeout}))
        
        # Add POA middleware for Ruby blockchain
        self.web3.middleware_onion.inject(geth_poa_middleware, layer=0)
        
        # Set gas strategy
        self.web3.eth.set_gas_price_strategy(medium_gas_price_strategy)
        
        self.chain_id = chain_id
        self.account = None
        self.logger = logging.getLogger(__name__)
        
        if private_key:
            self.set_account(private_key)
    
    # ===== WALLET & ACCOUNT MANAGEMENT =====
    
    def set_account(self, private_key: str) -> None:
        """Set the account using private key"""
        normalized_key = RubyConverter.normalize_private_key(private_key)
        if validate_private_key(normalized_key):
            self.account = Account.from_key(normalized_key)
            self.logger.info(f"Account set: {self.get_ruby_address()}")
        else:
            raise RubyWeb3Error("Invalid private key")
    
    def create_wallet(self) -> Dict[str, str]:
        """Create new wallet and return details in Ruby format"""
        account = Account.create()
        return {
            'address': RubyConverter.to_ruby_format(account.address),
            'private_key': account.key.hex(),
            'public_key': account.key.public_key.to_hex()
        }
    
    def create_wallet_from_mnemonic(self, mnemonic: str, path: str = "m/44'/60'/0'/0/0") -> Dict[str, str]:
        """Create wallet from mnemonic phrase"""
        try:
            # Note: This requires additional dependencies
            # pip install eth-account[hdwallet]
            account = Account.from_mnemonic(mnemonic, account_path=path)
            return {
                'address': RubyConverter.to_ruby_format(account.address),
                'private_key': account.key.hex(),
                'public_key': account.key.public_key.to_hex()
            }
        except ImportError:
            raise RubyWeb3Error("HD wallet support requires eth-account[hdwallet]")
    
    def recover_account_from_private_key(self, private_key: str) -> Dict[str, str]:
        """Recover account details from private key"""
        normalized_key = RubyConverter.normalize_private_key(private_key)
        account = Account.from_key(normalized_key)
        return {
            'address': RubyConverter.to_ruby_format(account.address),
            'public_key': account.key.public_key.to_hex()
        }
    
    def get_ruby_address(self) -> str:
        """Get current account address in Ruby format"""
        if self.account is None:
            raise RubyWeb3Error("No account set")
        return RubyConverter.to_ruby_format(self.account.address)
    
    # ===== BLOCKCHAIN INFO & NETWORK =====
    
    @property
    def is_connected(self) -> bool:
        """Check if connected to blockchain"""
        return self.web3.is_connected()
    
    def get_network_info(self) -> Dict[str, Any]:
        """Get comprehensive network information"""
        return {
            'chain_id': self.chain_id,
            'block_number': self.get_block_number(),
            'gas_price': self.get_gas_price(),
            'is_listening': self.web3.net.listening,
            'peer_count': self.web3.net.peer_count,
            'client_version': self.web3.client_version
        }
    
    def get_block_number(self) -> int:
        """Get current block number"""
        return self.web3.eth.block_number
    
    def get_block_details(self, block_identifier: Any = 'latest', full_transactions: bool = False) -> Dict[str, Any]:
        """Get block details"""
        block = self.web3.eth.get_block(block_identifier, full_transactions=full_transactions)
        return ResponseFormatter.format_block(dict(block))
    
    def get_block_range(self, start_block: int, end_block: int) -> List[Dict[str, Any]]:
        """Get multiple blocks in range"""
        blocks = []
        for block_num in range(start_block, end_block + 1):
            try:
                block = self.get_block_details(block_num)
                blocks.append(block)
            except Exception as e:
                self.logger.warning(f"Error fetching block {block_num}: {e}")
        return blocks
    
    # ===== BALANCE & TOKEN OPERATIONS =====
    
    def get_balance(self, address: Optional[str] = None) -> int:
        """Get balance in wei for address or current account"""
        if address is None:
            if self.account is None:
                raise RubyWeb3Error("No account set and no address provided")
            address = self.account.address
        else:
            address = RubyConverter.convert_address(address)
        
        return self.web3.eth.get_balance(address)
    
    def get_balance_ruby(self, address: Optional[str] = None) -> float:
        """Get balance in RUBY (native token)"""
        balance_wei = self.get_balance(address)
        return self.web3.from_wei(balance_wei, 'ether')
    
    def get_multiple_balances(self, addresses: List[str]) -> Dict[str, float]:
        """Get balances for multiple addresses"""
        balances = {}
        for address in addresses:
            try:
                balance = self.get_balance_ruby(address)
                ruby_address = RubyConverter.format_response_address(address)
                balances[ruby_address] = balance
            except Exception as e:
                self.logger.warning(f"Error getting balance for {address}: {e}")
                balances[RubyConverter.format_response_address(address)] = 0.0
        return balances
    
    # ===== TRANSACTION OPERATIONS =====
    
    def send_coin(self, to_address: str, amount: float, 
                 gas_limit: Optional[int] = None, 
                 gas_price: Optional[int] = None,
                 nonce: Optional[int] = None,
                 data: str = "0x") -> str:
        """
        Send native coins (RUBY) to address
        """
        if self.account is None:
            raise TransactionError("No account set for transaction")
        
        validate_eth_amount(amount)
        
        # Convert addresses
        to_address_hex = RubyConverter.convert_address(to_address)
        from_address_hex = self.account.address
        
        # Prepare transaction
        transaction = {
            'to': to_address_hex,
            'value': self.web3.to_wei(amount, 'ether'),
            'gas': gas_limit or self.estimate_gas(to_address, amount, data),
            'gasPrice': gas_price or self.get_gas_price(),
            'nonce': nonce or self.web3.eth.get_transaction_count(from_address_hex),
            'chainId': self.chain_id,
            'data': data
        }
        
        # Sign and send transaction
        signed_txn = self.web3.eth.account.sign_transaction(transaction, self.account.key)
        tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return RubyConverter.format_response_hash(tx_hash.hex())
    
    def send_batch_transactions(self, transactions: List[Dict]) -> List[str]:
        """Send multiple transactions in batch"""
        tx_hashes = []
        for tx in transactions:
            try:
                tx_hash = self.send_coin(
                    to_address=tx['to'],
                    amount=tx['amount'],
                    gas_limit=tx.get('gas_limit'),
                    gas_price=tx.get('gas_price')
                )
                tx_hashes.append(tx_hash)
            except Exception as e:
                self.logger.error(f"Failed to send transaction to {tx['to']}: {e}")
                tx_hashes.append(None)
        return tx_hashes
    
    def cancel_transaction(self, nonce: int, gas_price_increase: float = 1.1) -> str:
        """Cancel a pending transaction by sending zero-value transaction with same nonce"""
        if self.account is None:
            raise TransactionError("No account set for transaction")
        
        current_gas_price = self.get_gas_price()
        new_gas_price = int(current_gas_price * gas_price_increase)
        
        cancel_tx = {
            'to': self.account.address,  # Send to self
            'value': 0,
            'gas': 21000,
            'gasPrice': new_gas_price,
            'nonce': nonce,
            'chainId': self.chain_id
        }
        
        signed_txn = self.web3.eth.account.sign_transaction(cancel_tx, self.account.key)
        tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return RubyConverter.format_response_hash(tx_hash.hex())
    
    def speed_up_transaction(self, nonce: int, gas_price_increase: float = 1.5) -> str:
        """Speed up a pending transaction by resending with higher gas price"""
        return self.cancel_transaction(nonce, gas_price_increase)
    
    # ===== TRANSACTION QUERYING =====
    
    def get_transaction_details(self, tx_hash: str) -> Dict[str, Any]:
        """Get transaction details by hash"""
        tx_hash_hex = RubyConverter.convert_address(tx_hash)
        
        tx_data = self.web3.eth.get_transaction(tx_hash_hex)
        tx_receipt = self.web3.eth.get_transaction_receipt(tx_hash_hex)
        
        # Combine data
        result = dict(tx_data)
        if tx_receipt:
            result.update(dict(tx_receipt))
        
        return ResponseFormatter.format_transaction(result)
    
    def get_transaction_receipt(self, tx_hash: str) -> Dict[str, Any]:
        """Get transaction receipt"""
        tx_hash_hex = RubyConverter.convert_address(tx_hash)
        receipt = self.web3.eth.get_transaction_receipt(tx_hash_hex)
        return ResponseFormatter.format_receipt(dict(receipt))
    
    def get_transaction_status(self, tx_hash: str) -> Dict[str, Any]:
        """Get transaction status with confirmation details"""
        receipt = self.get_transaction_receipt(tx_hash)
        current_block = self.get_block_number()
        
        status = {
            'transaction_hash': receipt['transactionHash'],
            'status': 'pending',
            'block_number': receipt.get('blockNumber'),
            'confirmations': 0
        }
        
        if receipt.get('blockNumber'):
            status['confirmations'] = current_block - receipt['blockNumber']
            status['status'] = 'confirmed' if receipt.get('status') == 1 else 'failed'
        
        return status
    
    def get_transaction_history(self, address: str, from_block: int = 0, to_block: str = 'latest') -> List[Dict[str, Any]]:
        """Get transaction history for address"""
        address_hex = RubyConverter.convert_address(address)
        transactions = []
        
        # This is a simplified implementation
        # In production, you might want to use indexer services
        for block_num in range(from_block, self.get_block_number() + 1):
            try:
                block = self.get_block_details(block_num, full_transactions=True)
                for tx in block.get('transactions', []):
                    if tx.get('from') == address_hex or tx.get('to') == address_hex:
                        transactions.append(ResponseFormatter.format_transaction(tx))
            except Exception as e:
                self.logger.warning(f"Error processing block {block_num}: {e}")
        
        return transactions
    
    # ===== GAS & FEE OPERATIONS =====
    
    def get_gas_price(self) -> int:
        """Get current gas price"""
        return self.web3.eth.gas_price
    
    def estimate_gas(self, to_address: str, value: float = 0, data: str = "0x") -> int:
        """Estimate gas for transaction"""
        to_address_hex = RubyConverter.convert_address(to_address)
        from_address = self.account.address if self.account else None
        
        transaction = {
            'to': to_address_hex,
            'value': self.web3.to_wei(value, 'ether'),
            'data': data
        }
        
        if from_address:
            transaction['from'] = from_address
        
        return self.web3.eth.estimate_gas(transaction)
    
    def get_gas_estimate(self, to_address: str, value: float = 0, data: str = "0x") -> GasEstimate:
        """Get comprehensive gas estimate"""
        gas_limit = self.estimate_gas(to_address, value, data)
        gas_price = self.get_gas_price()
        total_cost_wei = gas_limit * gas_price
        total_cost_eth = self.web3.from_wei(total_cost_wei, 'ether')
        
        return GasEstimate(
            gas_limit=gas_limit,
            gas_price=gas_price,
            total_cost_wei=total_cost_wei,
            total_cost_eth=total_cost_eth
        )
    
    def get_fee_history(self, block_count: int = 10, newest_block: str = 'latest') -> Dict[str, Any]:
        """Get fee history"""
        return self.web3.eth.fee_history(block_count, newest_block)
    
    # ===== CONTRACT INTERACTIONS =====
    
    def deploy_contract(self, abi: List[Dict], bytecode: str, 
                       args: Optional[List] = None,
                       gas_limit: Optional[int] = None,
                       gas_price: Optional[int] = None) -> str:
        """Deploy a new contract"""
        if self.account is None:
            raise TransactionError("No account set for deployment")
        
        contract = self.web3.eth.contract(abi=abi, bytecode=bytecode)
        
        constructor_args = args or []
        transaction = contract.constructor(*constructor_args).build_transaction({
            'from': self.account.address,
            'gas': gas_limit or 2000000,
            'gasPrice': gas_price or self.get_gas_price(),
            'nonce': self.web3.eth.get_transaction_count(self.account.address),
            'chainId': self.chain_id
        })
        
        signed_txn = self.web3.eth.account.sign_transaction(transaction, self.account.key)
        tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return RubyConverter.format_response_hash(tx_hash.hex())
    
    def get_contract_address_from_tx(self, tx_hash: str) -> str:
        """Get contract address from deployment transaction"""
        receipt = self.get_transaction_receipt(tx_hash)
        contract_address = receipt.get('contractAddress')
        if contract_address:
            return RubyConverter.format_response_address(contract_address)
        raise RubyWeb3Error("No contract address found in transaction receipt")
    
    # ===== EVENT LISTENING =====
    
    def create_block_filter(self) -> str:
        """Create filter for new blocks"""
        return self.web3.eth.filter('latest')
    
    def create_pending_tx_filter(self) -> str:
        """Create filter for pending transactions"""
        return self.web3.eth.filter('pending')
    
    def get_filter_changes(self, filter_id: str) -> List[str]:
        """Get filter changes"""
        changes = self.web3.eth.get_filter_changes(filter_id)
        return [RubyConverter.format_response_hash(change.hex()) if isinstance(change, bytes) else change for change in changes]
    
    def listen_for_events(self, contract_address: str, abi: List[Dict], 
                         event_name: str, callback: callable, 
                         poll_interval: int = 5):
        """Listen for contract events"""
        from .contracts import ContractManager
        
        contract_manager = ContractManager(self, contract_address, abi)
        latest_block = self.get_block_number()
        
        while True:
            try:
                events = contract_manager.get_events(event_name, from_block=latest_block)
                for event in events:
                    callback(event)
                latest_block = self.get_block_number() + 1
                time.sleep(poll_interval)
            except Exception as e:
                self.logger.error(f"Error in event listener: {e}")
                time.sleep(poll_interval)
    
    # ===== UTILITY FUNCTIONS =====
    
    def get_transaction_count(self, address: Optional[str] = None) -> int:
        """Get transaction count for address"""
        if address is None:
            if self.account is None:
                raise RubyWeb3Error("No account set and no address provided")
            address = self.account.address
        else:
            address = RubyConverter.convert_address(address)
        
        return self.web3.eth.get_transaction_count(address)
    
    def get_code(self, address: str) -> str:
        """Get contract code at address"""
        address_hex = RubyConverter.convert_address(address)
        return self.web3.eth.get_code(address_hex).hex()
    
    def is_contract_address(self, address: str) -> bool:
        """Check if address is a contract"""
        code = self.get_code(address)
        return code != '0x' and code != '0x0'
    
    def get_storage_at(self, address: str, position: int) -> str:
        """Get storage value at position"""
        address_hex = RubyConverter.convert_address(address)
        return self.web3.eth.get_storage_at(address_hex, position).hex()
    
    def get_proof(self, address: str, positions: List[int], block_identifier: Any = 'latest') -> Dict[str, Any]:
        """Get storage proof for address"""
        address_hex = RubyConverter.convert_address(address)
        return self.web3.eth.get_proof(address_hex, positions, block_identifier)
    
    def wait_for_transaction(self, tx_hash: str, timeout: int = 120, poll_interval: int = 2) -> Dict[str, Any]:
        """Wait for transaction confirmation with polling"""
        tx_hash_hex = RubyConverter.convert_address(tx_hash)
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            try:
                receipt = self.web3.eth.get_transaction_receipt(tx_hash_hex)
                if receipt is not None:
                    return ResponseFormatter.format_receipt(dict(receipt))
            except Exception as e:
                self.logger.debug(f"Error checking receipt: {e}")
            
            time.sleep(poll_interval)
        
        raise TransactionError(f"Transaction not confirmed within {timeout} seconds")
    
    def get_chain_id(self) -> int:
        """Get chain ID from node"""
        return self.web3.eth.chain_id
    
    def get_protocol_version(self) -> str:
        """Get Ethereum protocol version"""
        return self.web3.eth.protocol_version
    
    def get_syncing_status(self) -> Union[bool, Dict[str, Any]]:
        """Get node syncing status"""
        return self.web3.eth.syncing