from typing import Dict, Any, List, Optional
from web3 import Web3
from web3.contract import Contract

from ..exceptions import RubyWeb3Error
from ..utils.converter import RubyConverter
from ..utils.formatter import ResponseFormatter
from ..models.tokens import TokenInfo, TokenBalance

class TokenManager:
    """Manager for ERC20 token operations"""
    
    # Common ERC20 ABI
    ERC20_ABI = [
        {
            "constant": True,
            "inputs": [],
            "name": "name",
            "outputs": [{"name": "", "type": "string"}],
            "type": "function"
        },
        {
            "constant": True,
            "inputs": [],
            "name": "symbol",
            "outputs": [{"name": "", "type": "string"}],
            "type": "function"
        },
        {
            "constant": True,
            "inputs": [],
            "name": "decimals",
            "outputs": [{"name": "", "type": "uint8"}],
            "type": "function"
        },
        {
            "constant": True,
            "inputs": [{"name": "_owner", "type": "address"}],
            "name": "balanceOf",
            "outputs": [{"name": "balance", "type": "uint256"}],
            "type": "function"
        },
        {
            "constant": True,
            "inputs": [
                {"name": "_owner", "type": "address"},
                {"name": "_spender", "type": "address"}
            ],
            "name": "allowance",
            "outputs": [{"name": "", "type": "uint256"}],
            "type": "function"
        },
        {
            "constant": False,
            "inputs": [
                {"name": "_to", "type": "address"},
                {"name": "_value", "type": "uint256"}
            ],
            "name": "transfer",
            "outputs": [{"name": "", "type": "bool"}],
            "type": "function"
        },
        {
            "constant": False,
            "inputs": [
                {"name": "_from", "type": "address"},
                {"name": "_to", "type": "address"},
                {"name": "_value", "type": "uint256"}
            ],
            "name": "transferFrom",
            "outputs": [{"name": "", "type": "bool"}],
            "type": "function"
        },
        {
            "constant": False,
            "inputs": [
                {"name": "_spender", "type": "address"},
                {"name": "_value", "type": "uint256"}
            ],
            "name": "approve",
            "outputs": [{"name": "", "type": "bool"}],
            "type": "function"
        }
    ]
    
    def __init__(self, web3_manager):
        self.web3_manager = web3_manager
        self.web3 = web3_manager.web3
    
    def get_token_info(self, token_address: str) -> TokenInfo:
        """Get token information"""
        token_address_hex = RubyConverter.convert_address(token_address)
        contract = self.web3.eth.contract(address=token_address_hex, abi=self.ERC20_ABI)
        
        try:
            name = contract.functions.name().call()
            symbol = contract.functions.symbol().call()
            decimals = contract.functions.decimals().call()
            total_supply = contract.functions.totalSupply().call()
        except Exception as e:
            raise RubyWeb3Error(f"Error getting token info: {e}")
        
        return TokenInfo(
            address=RubyConverter.format_response_address(token_address),
            name=name,
            symbol=symbol,
            decimals=decimals,
            total_supply=total_supply
        )
    
    def get_token_balance(self, token_address: str, wallet_address: str) -> TokenBalance:
        """Get token balance for wallet"""
        token_address_hex = RubyConverter.convert_address(token_address)
        wallet_address_hex = RubyConverter.convert_address(wallet_address)
        
        contract = self.web3.eth.contract(address=token_address_hex, abi=self.ERC20_ABI)
        
        try:
            balance = contract.functions.balanceOf(wallet_address_hex).call()
            decimals = contract.functions.decimals().call()
            
            # Calculate readable balance
            readable_balance = balance / (10 ** decimals)
            
            return TokenBalance(
                token_address=RubyConverter.format_response_address(token_address),
                wallet_address=RubyConverter.format_response_address(wallet_address),
                balance=balance,
                readable_balance=readable_balance,
                decimals=decimals
            )
        except Exception as e:
            raise RubyWeb3Error(f"Error getting token balance: {e}")
    
    def get_multiple_token_balances(self, token_addresses: List[str], wallet_address: str) -> Dict[str, TokenBalance]:
        """Get balances for multiple tokens"""
        balances = {}
        for token_address in token_addresses:
            try:
                balance = self.get_token_balance(token_address, wallet_address)
                balances[balance.token_address] = balance
            except Exception as e:
                self.web3_manager.logger.warning(f"Error getting balance for {token_address}: {e}")
        return balances
    
    def transfer_token(self, token_address: str, to_address: str, amount: float,
                      gas_limit: int = 100000, gas_price: Optional[int] = None) -> str:
        """Transfer tokens"""
        if self.web3_manager.account is None:
            raise RubyWeb3Error("No account set for transaction")
        
        token_address_hex = RubyConverter.convert_address(token_address)
        to_address_hex = RubyConverter.convert_address(to_address)
        from_address_hex = self.web3_manager.account.address
        
        contract = self.web3.eth.contract(address=token_address_hex, abi=self.ERC20_ABI)
        
        # Get token decimals
        decimals = contract.functions.decimals().call()
        token_amount = int(amount * (10 ** decimals))
        
        # Build transaction
        transaction = contract.functions.transfer(
            to_address_hex, token_amount
        ).build_transaction({
            'from': from_address_hex,
            'gas': gas_limit,
            'gasPrice': gas_price or self.web3.eth.gas_price,
            'nonce': self.web3.eth.get_transaction_count(from_address_hex),
            'chainId': self.web3_manager.chain_id
        })
        
        # Sign and send
        signed_txn = self.web3.eth.account.sign_transaction(transaction, self.web3_manager.account.key)
        tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return RubyConverter.format_response_hash(tx_hash.hex())
    
    def approve_token(self, token_address: str, spender_address: str, amount: float,
                     gas_limit: int = 50000, gas_price: Optional[int] = None) -> str:
        """Approve token spending"""
        if self.web3_manager.account is None:
            raise RubyWeb3Error("No account set for transaction")
        
        token_address_hex = RubyConverter.convert_address(token_address)
        spender_address_hex = RubyConverter.convert_address(spender_address)
        from_address_hex = self.web3_manager.account.address
        
        contract = self.web3.eth.contract(address=token_address_hex, abi=self.ERC20_ABI)
        
        # Get token decimals
        decimals = contract.functions.decimals().call()
        token_amount = int(amount * (10 ** decimals))
        
        # Build transaction
        transaction = contract.functions.approve(
            spender_address_hex, token_amount
        ).build_transaction({
            'from': from_address_hex,
            'gas': gas_limit,
            'gasPrice': gas_price or self.web3.eth.gas_price,
            'nonce': self.web3.eth.get_transaction_count(from_address_hex),
            'chainId': self.web3_manager.chain_id
        })
        
        # Sign and send
        signed_txn = self.web3.eth.account.sign_transaction(transaction, self.web3_manager.account.key)
        tx_hash = self.web3.eth.send_raw_transaction(signed_txn.rawTransaction)
        
        return RubyConverter.format_response_hash(tx_hash.hex())
    
    def get_allowance(self, token_address: str, owner_address: str, spender_address: str) -> int:
        """Get token allowance"""
        token_address_hex = RubyConverter.convert_address(token_address)
        owner_address_hex = RubyConverter.convert_address(owner_address)
        spender_address_hex = RubyConverter.convert_address(spender_address)
        
        contract = self.web3.eth.contract(address=token_address_hex, abi=self.ERC20_ABI)
        
        return contract.functions.allowance(owner_address_hex, spender_address_hex).call()