"""
Event service for listening to blockchain events
"""

import time
import threading
from typing import Dict, List, Callable, Any, Optional
from web3 import Web3
from web3.types import LogReceipt

from ..exceptions import RubyWeb3Error
from ..utils.converter import RubyConverter
from ..utils.formatter import ResponseFormatter

class EventService:
    """
    Service for handling blockchain events and subscriptions
    """
    
    def __init__(self, web3_manager):
        self.web3_manager = web3_manager
        self.web3 = web3_manager.web3
        self.active_listeners = {}
        self.logger = web3_manager.logger
    
    def create_event_filter(self, contract_address: str, abi: list, event_name: str, 
                          from_block: int = 0, to_block: str = 'latest') -> str:
        """
        Create a filter for specific contract events
        """
        contract_address_hex = RubyConverter.convert_address(contract_address)
        contract = self.web3.eth.contract(address=contract_address_hex, abi=abi)
        
        event = getattr(contract.events, event_name)
        event_filter = event.create_filter(fromBlock=from_block, toBlock=to_block)
        
        filter_id = str(len(self.active_listeners))
        self.active_listeners[filter_id] = event_filter
        
        return filter_id
    
    def get_filter_events(self, filter_id: str) -> List[Dict[str, Any]]:
        """
        Get events from a filter
        """
        if filter_id not in self.active_listeners:
            raise RubyWeb3Error(f"Filter {filter_id} not found")
        
        event_filter = self.active_listeners[filter_id]
        events = event_filter.get_all_entries()
        
        formatted_events = []
        for event in events:
            formatted_event = dict(event)
            formatted_event['args'] = self._format_event_args(event['args'])
            formatted_events.append(ResponseFormatter.format_transaction(formatted_event))
        
        return formatted_events
    
    def listen_for_events(self, contract_address: str, abi: list, event_name: str,
                         callback: Callable, poll_interval: int = 5,
                         start_block: Optional[int] = None) -> threading.Thread:
        """
        Start listening for events in a separate thread
        """
        if start_block is None:
            start_block = self.web3_manager.get_block_number()
        
        def event_listener():
            last_block = start_block
            
            while True:
                try:
                    current_block = self.web3_manager.get_block_number()
                    
                    if current_block > last_block:
                        # Get events since last block
                        filter_id = self.create_event_filter(
                            contract_address, abi, event_name, 
                            from_block=last_block + 1, to_block=current_block
                        )
                        
                        events = self.get_filter_events(filter_id)
                        for event in events:
                            callback(event)
                        
                        last_block = current_block
                        self.remove_filter(filter_id)
                    
                    time.sleep(poll_interval)
                    
                except Exception as e:
                    self.logger.error(f"Error in event listener: {e}")
                    time.sleep(poll_interval)
        
        listener_thread = threading.Thread(target=event_listener, daemon=True)
        listener_thread.start()
        
        return listener_thread
    
    def listen_for_transfers(self, token_address: str, callback: Callable, 
                           poll_interval: int = 5) -> threading.Thread:
        """
        Listen for ERC20 transfer events
        """
        transfer_abi = [
            {
                "anonymous": False,
                "inputs": [
                    {"indexed": True, "name": "from", "type": "address"},
                    {"indexed": True, "name": "to", "type": "address"},
                    {"indexed": False, "name": "value", "type": "uint256"}
                ],
                "name": "Transfer",
                "type": "event"
            }
        ]
        
        return self.listen_for_events(
            token_address, transfer_abi, "Transfer", callback, poll_interval
        )
    
    def listen_for_approvals(self, token_address: str, callback: Callable,
                           poll_interval: int = 5) -> threading.Thread:
        """
        Listen for ERC20 approval events
        """
        approval_abi = [
            {
                "anonymous": False,
                "inputs": [
                    {"indexed": True, "name": "owner", "type": "address"},
                    {"indexed": True, "name": "spender", "type": "address"},
                    {"indexed": False, "name": "value", "type": "uint256"}
                ],
                "name": "Approval",
                "type": "event"
            }
        ]
        
        return self.listen_for_events(
            token_address, approval_abi, "Approval", callback, poll_interval
        )
    
    def remove_filter(self, filter_id: str) -> bool:
        """
        Remove an event filter
        """
        if filter_id in self.active_listeners:
            del self.active_listeners[filter_id]
            return True
        return False
    
    def cleanup_filters(self):
        """
        Clean up all active filters
        """
        self.active_listeners.clear()
    
    def _format_event_args(self, args: Any) -> Any:
        """
        Format event arguments to use Ruby-style addresses
        """
        if isinstance(args, dict):
            formatted_args = {}
            for key, value in args.items():
                if isinstance(value, str) and Web3.is_address(value):
                    formatted_args[key] = RubyConverter.format_response_address(value)
                else:
                    formatted_args[key] = value
            return formatted_args
        return args
    
    def get_pending_transactions(self) -> List[str]:
        """
        Get pending transactions from mempool
        """
        try:
            pending_filter = self.web3.eth.filter('pending')
            pending_txs = pending_filter.get_new_entries()
            return [RubyConverter.format_response_hash(tx.hex()) for tx in pending_txs]
        except Exception as e:
            self.logger.error(f"Error getting pending transactions: {e}")
            return []