from typing import List, Dict, Any, Tuple
from web3 import Web3
from web3.contract import Contract

from ..utils.converter import RubyConverter

class MulticallService:
    """Service for batch contract calls"""
    
    def __init__(self, web3_manager, multicall_contract_address: Optional[str] = None):
        self.web3_manager = web3_manager
        self.web3 = web3_manager.web3
        
        # Default multicall contract addresses by chain
        self.multicall_addresses = {
            1: "0xeefBa1e63905eF1D7ACbA5a8513c70307C1cE441",  # Mainnet
            18359: multicall_contract_address  # Ruby chain
        }
        
        self.multicall_abi = [
            {
                "constant": False,
                "inputs": [
                    {
                        "components": [
                            {"name": "target", "type": "address"},
                            {"name": "callData", "type": "bytes"}
                        ],
                        "name": "calls",
                        "type": "tuple[]"
                    }
                ],
                "name": "aggregate",
                "outputs": [
                    {"name": "blockNumber", "type": "uint256"},
                    {"name": "returnData", "type": "bytes[]"}
                ],
                "type": "function"
            }
        ]
    
    def multicall(self, calls: List[Tuple[str, str, List]]) -> List[Any]:
        """Execute multiple contract calls in one transaction"""
        multicall_address = self.multicall_addresses.get(
            self.web3_manager.chain_id,
            self.multicall_addresses[1]  # Default to mainnet
        )
        
        if not multicall_address:
            raise ValueError("Multicall contract not available for this chain")
        
        multicall_contract = self.web3.eth.contract(
            address=Web3.to_checksum_address(multicall_address),
            abi=self.multicall_abi
        )
        
        # Prepare calls
        multicall_calls = []
        for contract_address, function_signature, args in calls:
            contract_address_hex = RubyConverter.convert_address(contract_address)
            contract = self.web3.eth.contract(
                address=Web3.to_checksum_address(contract_address_hex),
                abi=[{"name": function_signature.split('(')[0], "type": "function", "inputs": [], "outputs": []}]
            )
            call_data = contract.encodeABI(function_signature, args)
            multicall_calls.append((contract_address_hex, call_data))
        
        # Execute multicall
        result = multicall_contract.functions.aggregate(multicall_calls).call()
        
        return result