"""
Price feed service for cryptocurrency prices
"""

import requests
import time
from typing import Dict, Optional, List
from decimal import Decimal

from ..exceptions import RubyWeb3Error

class PriceFeedService:
    """
    Service for getting cryptocurrency price feeds
    """
    
    def __init__(self):
        self.cache = {}
        self.cache_timeout = 300  # 5 minutes
        self.coingecko_base_url = "https://api.coingecko.com/api/v3"
    
    def get_token_price(self, token_id: str, vs_currency: str = 'usd') -> Optional[float]:
        """
        Get token price from CoinGecko
        """
        cache_key = f"{token_id}_{vs_currency}"
        
        # Check cache
        if cache_key in self.cache:
            cached_data = self.cache[cache_key]
            if time.time() - cached_data['timestamp'] < self.cache_timeout:
                return cached_data['price']
        
        try:
            url = f"{self.coingecko_base_url}/simple/price"
            params = {
                'ids': token_id,
                'vs_currencies': vs_currency
            }
            
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            price = data.get(token_id, {}).get(vs_currency)
            
            if price is not None:
                # Update cache
                self.cache[cache_key] = {
                    'price': price,
                    'timestamp': time.time()
                }
            
            return price
            
        except requests.exceptions.RequestException as e:
            raise RubyWeb3Error(f"Error fetching price from CoinGecko: {e}")
    
    def get_multiple_prices(self, token_ids: List[str], vs_currency: str = 'usd') -> Dict[str, Optional[float]]:
        """
        Get multiple token prices at once
        """
        prices = {}
        
        for token_id in token_ids:
            try:
                price = self.get_token_price(token_id, vs_currency)
                prices[token_id] = price
            except Exception as e:
                prices[token_id] = None
        
        return prices
    
    def get_historical_price(self, token_id: str, vs_currency: str = 'usd', 
                           date: str = None, timestamp: int = None) -> Optional[float]:
        """
        Get historical token price
        """
        try:
            url = f"{self.coingecko_base_url}/coins/{token_id}/history"
            params = {
                'vs_currency': vs_currency,
                'date': date or self._timestamp_to_date(timestamp)
            }
            
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            return data.get('market_data', {}).get('current_price', {}).get(vs_currency)
            
        except requests.exceptions.RequestException as e:
            raise RubyWeb3Error(f"Error fetching historical price: {e}")
    
    def get_price_change(self, token_id: str, vs_currency: str = 'usd', 
                        days: int = 1) -> Dict[str, float]:
        """
        Get price change percentage over time
        """
        try:
            url = f"{self.coingecko_base_url}/coins/{token_id}/market_chart"
            params = {
                'vs_currency': vs_currency,
                'days': days
            }
            
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            prices = data.get('prices', [])
            
            if len(prices) >= 2:
                old_price = prices[0][1]
                new_price = prices[-1][1]
                change_percent = ((new_price - old_price) / old_price) * 100
                
                return {
                    'old_price': old_price,
                    'new_price': new_price,
                    'change_percent': change_percent,
                    'change_absolute': new_price - old_price
                }
            
            return {}
            
        except requests.exceptions.RequestException as e:
            raise RubyWeb3Error(f"Error fetching price change: {e}")
    
    def get_token_info(self, token_id: str) -> Dict[str, any]:
        """
        Get comprehensive token information
        """
        try:
            url = f"{self.coingecko_base_url}/coins/{token_id}"
            params = {
                'localization': 'false',
                'tickers': 'false',
                'market_data': 'true',
                'community_data': 'false',
                'developer_data': 'false',
                'sparkline': 'false'
            }
            
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
            
            return response.json()
            
        except requests.exceptions.RequestException as e:
            raise RubyWeb3Error(f"Error fetching token info: {e}")
    
    def convert_value(self, from_token: str, to_token: str, amount: float) -> Optional[float]:
        """
        Convert value between different tokens
        """
        try:
            from_price = self.get_token_price(from_token, 'usd')
            to_price = self.get_token_price(to_token, 'usd')
            
            if from_price and to_price:
                usd_value = amount * from_price
                return usd_value / to_price
            
            return None
            
        except Exception as e:
            raise RubyWeb3Error(f"Error converting value: {e}")
    
    def _timestamp_to_date(self, timestamp: int) -> str:
        """
        Convert timestamp to date string for CoinGecko API
        """
        import datetime
        dt = datetime.datetime.fromtimestamp(timestamp)
        return dt.strftime('%d-%m-%Y')
    
    def clear_cache(self):
        """
        Clear price cache
        """
        self.cache.clear()