from typing import Any, Optional, List, Dict
from pydantic import BaseModel, Field
from langchain_core.embeddings import Embeddings
from langchain.chains.query_constructor.base import AttributeInfo
import os
import pickle
import uuid
from .base import SharedMemoryBase
from datetime import datetime, timezone
import networkx as nx
import inspect
import hashlib
import json
from time import sleep
import asyncio

PICKLE_PROTOCOL_VERSION = 4  # Fix the pickle protocol version

# Auxiliar functions
def build_graph(objects):
    G = nx.DiGraph()
    id_to_obj = {obj['object_id']: obj for obj in objects}
    
    for obj in objects:
        G.add_node(obj['object_id'], **obj)
        # Add edges based on function_generator if it exists
        if obj.get('function_generator'):
            G.add_edge(obj['function_generator'], obj['object_id'])
        # Add edges based on objects_args if they exist
        for arg_id in obj.get('objects_args', []):
            if arg_id in id_to_obj:
                G.add_edge(arg_id, obj['object_id'])
    
    return G, id_to_obj

def generate_summary_with_processing_history(objects, language: str = "pt_br"):
    G, id_to_obj = build_graph(objects)

    if not nx.is_directed_acyclic_graph(G):
        raise ValueError("The graph contains cycles and cannot be topologically sorted.")

    ordered_ids_topological = list(nx.topological_sort(G))
    objects_sorted_by_created_at = sorted(
        [obj_id for obj_id in ordered_ids_topological if id_to_obj[obj_id].get("object_type") != "python-callable"],
        key=lambda oid: id_to_obj[oid].get("SharedObjectMetadata", {}).get("created_at", 0)
    )

    # Language-specific labels
    labels = {
        "pt_br": {
            "object_list": "Lista de objetos:",
            "processing_steps": "Etapas de processamento:",
            "functions_section": "Funções utilizadas:",
            "type": "tipo",
            "description": "descrição",
            "function_description": "descrição da função:",
            "no_name": "<sem nome>",
            "no_description": "<sem descrição>",
            "unknown_type": "<tipo desconhecido>",
            "function_without_name": "<função sem nome>",
            "no_code_available": "<sem código disponível>",
            "processed_by_function_singular": "foi processado pela função",
            "processed_by_function_plural": "foram processados pela função",
            "and_generated_object_singular": "e gerou o objeto",
            "and_generated_object_plural": "e geraram os objetos",
            "function_code": "Código da função:",
            "created_at": "criado em",
            "date_format": "%d/%m/%Y %H:%M:%S",
            "with_parameters": "com os parâmetros",
            "generated_by": "gerado por",
            "kwargs_used": "parâmetros utilizados",
        },
        "en": {
            "object_list": "Object list:",
            "processing_steps": "Processing steps:",
            "functions_section": "Used functions:",
            "type": "type",
            "description": "description",
            "function_description": "function description:",
            "no_name": "<no name>",
            "no_description": "<no description>",
            "unknown_type": "<unknown type>",
            "function_without_name": "<unnamed function>",
            "no_code_available": "<no code available>",
            "processed_by_function_singular": "was processed by the function",
            "processed_by_function_plural": "were processed by the function",
            "and_generated_object_singular": "and generated the object",
            "and_generated_object_plural": "and generated the objects",
            "function_code": "Function code:",
            "created_at": "created at",
            "date_format": "%Y-%m-%d %H:%M:%S",
            "with_parameters": "with parameters",
            "generated_by": "generated by",
            "kwargs_used": "parameters used",
        }
    }

    lang = labels.get(language, labels["pt_br"])

    # First part: Object list
    object_lines = [lang["object_list"]]
    for object_id in objects_sorted_by_created_at:
        obj = id_to_obj[object_id]
        metadata = obj.get("SharedObjectMetadata", {})
        name = metadata.get("name", lang["no_name"])
        description = metadata.get("description", lang["no_description"])
        obj_type = obj.get("object_type", lang["unknown_type"])
        created_at = metadata.get("created_at", 0)
        created_at_str = datetime.fromtimestamp(created_at / 1000).strftime(lang["date_format"])
        
        object_summary = [
            f"- {name} (id: {object_id})",
            f"    {lang['type']}: {obj_type}",
            f"    {lang['description']}: {description}",
            f"    {lang['created_at']}: {created_at_str}"
        ]
        
        # Add generator function information if it exists
        function_generator_id = metadata.get("function_generator")
        if function_generator_id:
            func_obj = id_to_obj.get(function_generator_id)
            if func_obj:
                func_metadata = func_obj.get("SharedObjectMetadata", {})
                func_name = func_metadata.get("name", lang["function_without_name"])
                object_summary.append(f"    {lang['generated_by']}: {func_name} (id: {function_generator_id})")
                
                # Add kwargs information if available
                kwargs = metadata.get("function_generator_kwargs", {})
                if kwargs:
                    kwargs_str = ", ".join(f"{k}={v}" for k, v in kwargs.items())
                    object_summary.append(f"    {lang['kwargs_used']}: {kwargs_str}")
        
        object_lines.append("\n".join(object_summary))

    # Second part: Processing history (without function code)
    process_lines = []
    step_counter = 1
    functions_used = set()

    # Rastrear execuções únicas de funções
    execution_to_objects = {}
    for object_id in objects_sorted_by_created_at:
        obj = id_to_obj[object_id]
        metadata = obj.get("SharedObjectMetadata", {})
        function_generator_id = metadata.get("function_generator")
        if function_generator_id:
            functions_used.add(function_generator_id)
        if not function_generator_id:
            continue

        # Criar uma chave única para a execução
        inputs = tuple(sorted(metadata.get("objects_args", [])))
        if metadata.get("function_generator_kwargs") is not None:
            kwargs = metadata.get("function_generator_kwargs").items()
        else:
            kwargs = {}
        combination = (function_generator_id, inputs, kwargs)
        execution_key =  hashlib.sha256(str(combination).encode('utf-8')).hexdigest()

        # Agrupar outputs por execução única
        if execution_key not in execution_to_objects:
            execution_to_objects[execution_key] = {"inputs": inputs, "outputs": set()}
        execution_to_objects[execution_key]["outputs"].add(object_id)

    for execution_key, data in execution_to_objects.items():
        function_id, inputs, kwargs = combination
        func_obj = id_to_obj.get(function_id)
        if not func_obj:
            continue

        func_metadata = func_obj.get("SharedObjectMetadata", {})
        func_name = func_metadata.get("name", lang["function_without_name"])
        inputs = [
            f"{id_to_obj[arg_id].get('SharedObjectMetadata', {}).get('name', f'id: {arg_id}')} (id: {arg_id})"
            for arg_id in data["inputs"]
        ]
        outputs = [
            f"{id_to_obj[obj_id].get('SharedObjectMetadata', {}).get('name', f'id: {obj_id}')} (id: {obj_id})"
            for obj_id in data["outputs"]
        ]
        kwargs_text = (
            f" {lang['with_parameters']} {{{', '.join(f'{k}={repr(v)}' for k, v in kwargs)}}}"
            if kwargs else ""
        )

        inputs_text = ", ".join(inputs)
        outputs_text = ", ".join(outputs)
        processed_label = (
            lang["processed_by_function_plural"]
            if len(data["inputs"]) > 1
            else lang["processed_by_function_singular"]
        )
        generated_label = (
            lang["and_generated_object_plural"]
            if len(data["outputs"]) > 1
            else lang["and_generated_object_singular"]
        )

        process_lines.append(
            f"{step_counter} - {inputs_text} {processed_label} {func_name} (id: {function_id}){kwargs_text} {generated_label} {outputs_text}."
        )
        step_counter += 1

    # Third part: Functions used (with their code)
    function_lines = []
    
    for function_id in functions_used:
        func_obj = id_to_obj.get(function_id)
        if not func_obj:
            continue
            
        func_metadata = func_obj.get("SharedObjectMetadata", {})
        func_name = func_metadata.get("name", lang["function_without_name"])
        func_description = func_metadata.get("description", lang["no_description"])
        func_code = func_obj.get("object", lang["no_code_available"])
        
        function_lines.append(f"- {func_name} (id: {function_id})")
        function_lines.append(f"  {lang['description']}: {func_description}")
        function_lines.append(f"  {lang['function_code']}\n```python\n{func_code}\n```")

    # Include section headers only if there is content
    sections = []
    sections.append("\n\n".join(object_lines))
    
    if process_lines:
        process_lines.insert(0, lang["processing_steps"])
        sections.append("\n\n".join(process_lines))
    
    if function_lines:
        function_lines.insert(0, lang["functions_section"])
        sections.append("\n\n".join(function_lines))

    return "\n\n".join(sections)


# Classes
class SharedObjectMetadata(BaseModel):
    description: str = Field(..., description="Description of the object")
    object_type: str = Field(..., description="Type of the object")
    name: str = Field(..., description="Name of the object")
    disk_path: str = Field(..., description="Path where the object is stored on disk")
    object_id: str = Field(..., description="Unique ID of the object")
    function_generator: Optional[str] = Field(
        default=None, description="ID of the python-callable that generated this object"
    )
    objects_args: Optional[List[str]] = Field(
        default=None, description="List of object IDs used as arguments to generate this object"
    )
    function_generator_kwargs: Optional[Dict[str, Any]] = Field(
        default=None, description="Dictionary of keyword arguments used to generate this object"
    )
    created_at: int = Field(..., description="Timestamp when the object was created (milliseconds since epoch)")

class SharedObject(BaseModel):
    metadata: SharedObjectMetadata
    object: Any 

class SharedObjectsMemory(SharedMemoryBase):
    """
    Memory class for managing shared objects.

    This class allows persisting Python objects with metadata for semantic search.
    """

    MEMORY_NAME = "eagle-shared-objects-memory"

    EMBEDDED_FIELDS = ["description", "object_type"]

    VALUE_EXAMPLE = {
        "description": "Example description",
        "object_type": "Example type",
        "name": "Example name",
        "disk_path": "/path/to/object.pkl",
        "object_id": "unique-id",
        "function_generator": "function-id",
        "objects_args": ["argument-object-id-1", "argument-object-id-2"],
        "function_generator_kwargs": '{"param1": "value1", "param2": 42}',
        "created_at": 1746031133388
    }

    ATTRIBUTE_INFO = [
        AttributeInfo(name="value.description", type="string", description="Description of the object"),
        AttributeInfo(name="value.object_type", type="string", description="Type of the object"),
        AttributeInfo(name="value.name", type="string", description="Name of the object"),
        AttributeInfo(name="value.disk_path", type="string", description="Path where the object is stored on disk"),
        AttributeInfo(name="value.object_id", type="string", description="Unique ID of the object"),
        AttributeInfo(name="value.function_generator", type="string", description="ID of the python-callable that generated this object"),
        AttributeInfo(name="value.objects_args", type="list", description="List of object IDs used as arguments to generate this object"),
        AttributeInfo(name="value.function_generator_kwargs", type="string", description="Dictionary of keyword arguments used to generate this object stored as a JSON string"),
        AttributeInfo(name="value.created_at", type="long", description="Timestamp when the object was created (seconds with milliseconds)"),
    ]

    def __init__(self, store_class: type, embedding_model: Embeddings, embedding_dims: int, storage_dir: str) -> None:
        """
        Initialize SharedObjectsMemory.

        Args:
            store_class (type): The class of the store to use for persistence.
            embedding_model (Embeddings): The embedding model to use.
            embedding_dims (int): The dimensions of the embedding model.
            storage_dir (str): Directory to store serialized objects.
        """
        super().__init__(store_class, embedding_model, embedding_dims)
        self.storage_dir = storage_dir
        os.makedirs(storage_dir, exist_ok=True)

    def search_memories(
        self,
        chat_id: str,
        query: Optional[str] = None,
        limit: int = 10,
        offset: int = 0,
        filter_object_types: Optional[List[str]] = None,
    ) -> List[SharedObject]:
        """
        Search for shared objects by metadata.

        Args:
            chat_id (str): The chat ID.
            query (Optional[str]): A query string for semantic search.
            limit (int): Maximum number of results to return.
            offset (int): Number of results to skip.
            filter_object_types: Optional[List[str]]: List of object types to filter by.

        Returns:
            List[SharedObject]: A list of matching objects with metadata.
        """
        namespace_prefix = (self.MEMORY_NAME, chat_id)
        filter_list = []
        if filter_object_types:
            filter_object_types = 'or(' + ','.join([f'eq("value.object_type", "{obj_type}")' for obj_type in filter_object_types]) + ')'
            filter_list.append(filter_object_types)
        filter = 'and(' + ','.join(filter_list) + ')' if filter_list else None
        items = super().search_memories(
            namespace_prefix=namespace_prefix,
            query=query,
            filter=filter,
            limit=limit,
            offset=offset,
        )
        results = []
        for item in items:
            item.value['function_generator_kwargs'] = json.loads(item.value['function_generator_kwargs'])
            metadata = SharedObjectMetadata(**item.value)
            with open(metadata.disk_path, "rb") as file:
                obj = pickle.load(file)
            results.append(SharedObject(metadata=metadata, object=obj))
        return results

    def put_memory(
        self,
        chat_id: str,
        object_name: str,
        obj: Any,
        description: str,
        object_type: str,
        function_generator: Optional[str] = None,
        objects_args: Optional[List[str]] = None,
        function_generator_kwargs: Optional[Dict[str, Any]] = None,
        ttl: Optional[float] = None,
    ) -> str:
        """
        Add a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_name (str): The name of the object.
            obj: Any: The object to store.
            description (str): A description of the object.
            object_type (str): The type of the object.
            function_generator (Optional[str]): ID of the python-callable that generated this object.
            objects_args (Optional[List[str]]): List of object IDs used as arguments to generate this object.
            function_generator_kwargs (Optional[Dict[str, Any]]): Dictionary of keyword arguments used to generate this object.
            ttl (Optional[float]): Time-to-live for the memory in minutes.

        Returns:
            str: The generated object ID.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        object_id = str(uuid.uuid4())
        disk_path = os.path.join(self.storage_dir, f"{object_id}.pkl")
        
        if object_type == "python-callable":
            obj_to_store = inspect.getsource(obj) if inspect.isfunction(obj) else obj
            # For python-callables, we don't want to set function_generator
            function_generator = None
            objects_args = None
            function_generator_kwargs = None
        else:
            obj_to_store = obj

        with open(disk_path, "wb") as file:
            pickle.dump(obj_to_store, file, protocol=PICKLE_PROTOCOL_VERSION)
        created_at = int(datetime.now(timezone.utc).timestamp() * 1000)  # Precise timestamp in milliseconds
        value = {
            "description": description,
            "object_type": object_type,
            "name": object_name,
            "disk_path": disk_path,
            "object_id": object_id,
            "function_generator": function_generator,
            "objects_args": objects_args or [],
            "function_generator_kwargs": json.dumps(function_generator_kwargs) or {},
            "created_at": created_at,
        }
        super().put_memory(namespace=namespace, key=object_id, value=value, ttl=ttl)
        # Wait for the memory to be indexed
        sleep(2)
        return object_id

    async def aput_memory(
        self,
        chat_id: str,
        object_name: str,
        obj: Any,
        description: str,
        object_type: str,
        function_generator: Optional[str] = None,
        objects_args: Optional[List[str]] = None,
        function_generator_kwargs: Optional[Dict[str, Any]] = None,
        ttl: Optional[float] = None,
    ) -> str:
        """
        Asynchronously add a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_name: str: The name of the object.
            obj: Any: The object to store.
            description: str: A description of the object.
            object_type: str: The type of the object.
            function_generator: Optional[str]: ID of the python-callable that generated this object.
            objects_args: Optional[List[str]]: List of object IDs used as arguments to generate this object.
            function_generator_kwargs: Optional[Dict[str, Any]]: Dictionary of keyword arguments used to generate this object.
            ttl: Optional[float]: Time-to-live for the memory in minutes.

        Returns:
            str: The generated object ID.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        object_id = str(uuid.uuid4())
        disk_path = os.path.join(self.storage_dir, f"{object_id}.pkl")
        
        if object_type == "python-callable":
            obj_to_store = inspect.getsource(obj) if inspect.isfunction(obj) else obj
            # For python-callables, we don't want to set function_generator
            function_generator = None
            objects_args = None
            function_generator_kwargs = None
        else:
            obj_to_store = obj

        with open(disk_path, "wb") as file:
            pickle.dump(obj_to_store, file, protocol=PICKLE_PROTOCOL_VERSION)
        created_at = int(datetime.now().timestamp() * 1000)  # Precise timestamp in milliseconds
        value = {
            "description": description,
            "object_type": object_type,
            "name": object_name,
            "disk_path": disk_path,
            "object_id": object_id,
            "function_generator": function_generator,
            "objects_args": objects_args or [],
            "function_generator_kwargs": json.dumps(function_generator_kwargs) or {},
            "created_at": created_at,
        }
        await super().aput_memory(namespace=namespace, key=object_id, value=value, ttl=ttl)
        # Wait for the memory to be indexed
        asyncio.sleep(2)
        return object_id

    def get_memory(self, chat_id: str, object_id: str) -> Optional[SharedObject]:
        """
        Retrieve a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_id (str): The unique ID of the object.

        Returns:
            Optional[SharedObject]: The retrieved memory or None if not found.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        item = super().get_memory(namespace=namespace, key=object_id)
        if not item:
            return None
        item.value['function_generator_kwargs'] = json.loads(item.value['function_generator_kwargs'])
        metadata = SharedObjectMetadata(**item.value)
        with open(metadata.disk_path, "rb") as file:
            obj = pickle.load(file)
        return SharedObject(metadata=metadata, object=obj)

    async def aget_memory(self, chat_id: str, object_id: str) -> Optional[SharedObject]:
        """
        Asynchronously retrieve a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_id (str): The unique ID of the object.

        Returns:
            Optional[SharedObject]: The retrieved memory or None if not found.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        item = await super().aget_memory(namespace=namespace, key=object_id)
        if not item:
            return None
        item.value['function_generator_kwargs'] = json.loads(item.value['function_generator_kwargs'])
        metadata = SharedObjectMetadata(**item.value)
        with open(metadata.disk_path, "rb") as file:
            obj = pickle.load(file)
        return SharedObject(metadata=metadata, object=obj)

    def delete_memory(self, chat_id: str, object_id: str) -> None:
        """
        Delete a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_id (str): The unique ID of the object.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        item = super().get_memory(namespace=namespace, key=object_id)
        if item:
            os.remove(item.value["disk_path"])
        super().delete_memory(namespace=namespace, key=object_id)

    async def adelete_memory(self, chat_id: str, object_id: str) -> None:
        """
        Asynchronously delete a memory for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_id (str): The unique ID of the object.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        item = await super().aget_memory(namespace=namespace, key=object_id)
        if item:
            os.remove(item.value["disk_path"])
        await super().adelete_memory(namespace=namespace, key=object_id)

    def delete_memories_by_namespace(self, chat_id: str) -> None:
        """
        Delete all shared object memories in the given namespace.

        Args:
            chat_id (str): The chat ID.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        items = super().search_memories(namespace_prefix=namespace)
        for item in items:
            os.remove(item.value["disk_path"])
        super().delete_memories_by_namespace(namespace)

    def get_last_memories_metadata(
        self,
        chat_id: str,
        k: int,
        filter: Optional[str] = None,
    ) -> List[SharedObjectMetadata]:
        """
        Retrieve the last k memories sorted by creation time.

        Args:
            chat_id (str): The chat ID.
            k (int): Number of recent memories to retrieve.
            filter (Optional[str]): String-based filter for the search.

        Returns:
            List[SharedObjectMetadata]: A list of the most recent shared objects.
        """
        namespace_prefix = (self.MEMORY_NAME, chat_id)
        items = super().search_memories(
            namespace_prefix=namespace_prefix,
            filter=filter,
            limit=k,
            sort={"value.created_at": "desc"},
        )
        results = []
        for item in items:
            item.value['function_generator_kwargs'] = json.loads(item.value['function_generator_kwargs'])
            metadata = SharedObjectMetadata(**item.value)
            results.append(metadata)
        return results

    def get_memory_metadata(self, chat_id: str, object_id: str) -> Optional[SharedObjectMetadata]:
        """
        Retrieve metadata for a shared object.

        Args:
            chat_id (str): The chat ID.
            object_id (str): The unique ID of the object.

        Returns:
            Optional[SharedObjectMetadata]: The metadata of the object or None if not found.
        """
        namespace = (self.MEMORY_NAME, chat_id)
        item = super().get_memory(namespace=namespace, key=object_id)
        if not item:
            return None
        item.value['function_generator_kwargs'] = json.loads(item.value['function_generator_kwargs'])
        return SharedObjectMetadata(**item.value)

    def generate_summary_from_object_ids(self, chat_id: str, object_ids: List[str], language: str = "pt_br") -> str:
        """
        Generate a descriptive summary of selected objects and their relationships.

        Args:
            chat_id (str): The chat ID.
            object_ids (List[str]): List of object IDs to include in the summary.
            language (str): Language for the summary ("pt_br" or "en").

        Returns:
            str: A descriptive summary of the objects and their relationships.
        """
        processed_ids = set()
        objects = []

        def add_object_and_dependencies(object_id: str):
            if object_id in processed_ids:
                return
            metadata = self.get_memory_metadata(chat_id, object_id)
            if metadata:
                processed_ids.add(object_id)
                obj_content = None
                if metadata.object_type == "python-callable":
                    # Load the full content for Python callables
                    obj = self.get_memory(chat_id, object_id)
                    obj_content = obj.object if obj else None
                objects.append({
                    "object_id": metadata.object_id,
                    "object_type": metadata.object_type,
                    "SharedObjectMetadata": metadata.dict(),
                    "function_generator": metadata.function_generator,
                    "objects_args": metadata.objects_args or [],
                    "function_generator_kwargs": metadata.function_generator_kwargs or {},
                    "object": obj_content,  # Include the full content for callables
                })
                # Process function generator if it exists
                if metadata.function_generator:
                    add_object_and_dependencies(metadata.function_generator)
                # Process all args used to generate this object
                if metadata.objects_args:
                    for arg_id in metadata.objects_args:
                        add_object_and_dependencies(arg_id)

        for object_id in object_ids:
            add_object_and_dependencies(object_id)

        return generate_summary_with_processing_history(objects, language=language) if objects else ""
