from typing import Optional, List, Dict, Any
from pydantic import BaseModel, Field
from langchain_core.tools import BaseTool
from langchain_core.language_models.chat_models  import BaseChatModel
from eagle.chains.plan_qa import create_qa_chain
from eagle.memory.planning.possible_plans_memory import PossiblePlansMemory

class PossiblePlansToolInput(BaseModel):
    question: str = Field(description="A request of a plan to do something. ('How to...?', 'What is the best way to...?' etc.)")

class PossiblePlansTool(BaseTool):
    name: str = "possible_plans_tool"
    description: str = (
        "A tool that retrieves the most relevant plans from memory and uses a QA chain "
        "to generate a response to a given question, citing the sources used."
    )
    memory: PossiblePlansMemory
    prompt_language: str
    chain_llm: Optional[BaseChatModel] = Field(default=None, description="Language model for processing and checking")
    limit: int
    plans_set_id: str
    filter_sources: Optional[List[str]]

    args_schema = PossiblePlansToolInput

    def _format_plan_texts(self, plans: List[Dict[str, Any]], prompt_language: str) -> List[str]:
        """
        Format the plan texts based on the prompt language.

        Args:
            plans (List[Dict[str, Any]]): List of plans retrieved from memory.
            prompt_language (str): The language of the prompt ('pt-br' or 'en').

        Returns:
            List[str]: Formatted plan texts.
        """
        formatted_plans = []
        for plan in plans:
            if prompt_language == "pt-br":
                formatted_plans.append(
                    f"Pergunta: {plan['question']}\nPlano: {plan['plan']}\nFonte: {plan['source']}"
                )
            elif prompt_language == "en":
                formatted_plans.append(
                    f"Question: {plan['question']}\nPlan: {plan['plan']}\nSource: {plan['source']}"
                )
            else:
                raise ValueError(
                    f"Unsupported prompt language: {prompt_language}. Supported languages are 'pt-br' and 'en'."
                )
        return formatted_plans

    def _run(self, **_inputs: PossiblePlansToolInput) -> Dict[str, Any]:
        """
        Run the tool synchronously.

        Args:
            inputs (PossiblePlansToolInput): Input object containing the question and optional filter_sources.

        Returns:
            Dict[str, Any]: The response and sources generated by the QA chain.
        """
        # Extract arguments from inputs
        inputs = PossiblePlansToolInput(**_inputs)

        # Search for relevant plans in memory
        plans = self.memory.search_memories(
            plans_set_id=self.plans_set_id,
            query=inputs.question,
            limit=self.limit,
            filter_sources=self.filter_sources
        )

        # Format the plan texts based on the prompt language
        plan_texts = self._format_plan_texts(plans, self.prompt_language)

        # Create the QA chain
        qa_chain = create_qa_chain(
            prompt_language=self.prompt_language,
            llm=self.chain_llm
        )

        # Run the QA chain
        input_data = {
            "question": inputs.question,
            "plans": "\n".join(plan_texts)
        }
        result = qa_chain.invoke(input_data)

        return {
            "response": result.response,
            "sources": result.sources
        }

    async def _arun(self, **_inputs: PossiblePlansToolInput) -> Dict[str, Any]:
        """
        Run the tool asynchronously.

        Args:
            inputs (PossiblePlansToolInput): Input object containing the question and optional filter_sources.

        Returns:
            Dict[str, Any]: The response and sources generated by the QA chain.
        """
        # Extract arguments from inputs
        inputs = PossiblePlansToolInput(**_inputs)

        # Search for relevant plans in memory
        plans = await self.memory.asearch_memories(
            plans_set_id=self.plans_set_id,
            query=inputs.question,
            limit=self.limit,
            filter_sources=self.filter_sources
        )

        # Format the plan texts based on the prompt language
        plan_texts = self._format_plan_texts(plans, self.prompt_language)

        # Create the QA chain
        qa_chain = create_qa_chain(
            prompt_language=self.prompt_language,
            llm=self.chain_llm
        )

        # Run the QA chain
        input_data = {
            "question": inputs.question,
            "plans": "\n".join(plan_texts)
        }
        result = await qa_chain.ainvoke(input_data)

        return {
            "response": result.response,
            "sources": result.sources
        }
