"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CronJob = exports.ConcurrencyPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const workload = require("./workload");
/**
 * Concurrency policy for CronJobs.
 */
var ConcurrencyPolicy;
(function (ConcurrencyPolicy) {
    /**
     * This policy allows to run job concurrently.
     */
    ConcurrencyPolicy["ALLOW"] = "Allow";
    /**
     * This policy does not allow to run job concurrently. It does not let a new job to be scheduled if the previous one is not finished yet.
     */
    ConcurrencyPolicy["FORBID"] = "Forbid";
    /**
     * This policy replaces the currently running job if a new job is being scheduled.
     */
    ConcurrencyPolicy["REPLACE"] = "Replace";
})(ConcurrencyPolicy = exports.ConcurrencyPolicy || (exports.ConcurrencyPolicy = {}));
/**
 * A CronJob is responsible for creating a Job and scheduling it based on provided cron schedule. This helps running Jobs in a recurring manner.
 */
class CronJob extends workload.Workload {
    constructor(scope, id, props) {
        super(scope, id, {
            restartPolicy: pod_1.RestartPolicy.NEVER,
            select: false,
            ...props,
        });
        /**
         * Represents the resource type.
         */
        this.resourceType = 'cronjob';
        this.apiObject = new k8s.KubeCronJob(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        if (props.startingDeadline != undefined && props.startingDeadline.toSeconds() < 10) {
            throw new Error(`The 'startingDeadline' property cannot be less than 10 seconds since the Kubernetes CronJobController checks things every 10 seconds and hence the CronJob may not be scheduled. The value passed is: ${props.startingDeadline}`);
        }
        if (props.ttlAfterFinished != undefined && (props.successfulJobsRetained != undefined || props.failedJobsRetained != undefined)) {
            throw new Error('The \'ttlAfterFinished\' property cannot be set if \'successfulJobsRetained\' property or \'failedJobsRetained\' property is set. This would cause the retention of jobs to not work properly since it would delete the job based on its value.');
        }
        if (this.isolate) {
            this.connections.isolate();
        }
        this.schedule = props.schedule;
        this.concurrencyPolicy = props.concurrencyPolicy ?? ConcurrencyPolicy.FORBID;
        this.startingDeadline = props.startingDeadline ?? cdk8s_1.Duration.seconds(10);
        this.suspend = props.suspend ?? false;
        this.successfulJobsRetained = props.successfulJobsRetained ?? 3;
        this.failedJobsRetained = props.failedJobsRetained ?? 1;
        this.jobProperties = {
            activeDeadline: props.activeDeadline,
            backoffLimit: props.backoffLimit,
            ttlAfterFinished: props.ttlAfterFinished,
        };
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            concurrencyPolicy: this._toIoK8SApiBatchV1CronJobSpecConcurrencyPolicy(this.concurrencyPolicy),
            failedJobsHistoryLimit: this.failedJobsRetained,
            jobTemplate: {
                spec: this._toJobSpec(),
            },
            schedule: this.schedule.expressionString,
            startingDeadlineSeconds: this.startingDeadline.toSeconds(),
            successfulJobsHistoryLimit: this.successfulJobsRetained,
            suspend: this.suspend,
        };
    }
    /**
     * Returns the job spec.
     */
    _toJobSpec() {
        return {
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            activeDeadlineSeconds: this.jobProperties.activeDeadline?.toSeconds(),
            backoffLimit: this.jobProperties.backoffLimit,
            ttlSecondsAfterFinished: this.jobProperties.ttlAfterFinished?.toSeconds(),
        };
    }
    /**
     * Returns policy value as k8s.IoK8SApiBatchV1CronJobSpecConcurrencyPolicy type
     * @param policy Concurrency policy value
     * @returns k8s.IoK8SApiBatchV1CronJobSpecConcurrencyPolicy
     */
    _toIoK8SApiBatchV1CronJobSpecConcurrencyPolicy(policy) {
        const enumPolicy = policy.toUpperCase();
        return k8s.IoK8SApiBatchV1CronJobSpecConcurrencyPolicy[enumPolicy];
    }
}
exports.CronJob = CronJob;
_a = JSII_RTTI_SYMBOL_1;
CronJob[_a] = { fqn: "cdk8s-plus-23.CronJob", version: "2.0.54" };
//# sourceMappingURL=data:application/json;base64,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