"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.hasAutoscaler = false;
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
        if (this.isolate) {
            this.connections.isolate();
        }
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const myPorts = container.extractContainerPorts(this);
        const myPortNumbers = myPorts.map(p => p.number);
        const ports = options.ports ?? myPorts.map(p => ({ port: p.number, targetPort: p.number, protocol: p.protocol }));
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        // validate the ports are owned by our containers
        for (const port of ports) {
            const targetPort = port.targetPort ?? port.port;
            if (!myPortNumbers.includes(targetPort)) {
                throw new Error(`Unable to expose deployment ${this.name} via a service: Port ${targetPort} is not exposed by any container`);
            }
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.hasAutoscaler ? undefined : (this.replicas ?? 2),
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
    /**
     * @see IScalable.markHasAutoscaler()
     */
    markHasAutoscaler() {
        this.hasAutoscaler = true;
    }
    /**
     * @see IScalable.toScalingTarget()
     */
    toScalingTarget() {
        return {
            kind: this.apiObject.kind,
            apiVersion: this.apiObject.apiVersion,
            name: this.name,
            containers: this.containers,
            replicas: this.replicas,
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-23.Deployment", version: "2.0.54" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-23.PercentOrAbsolute", version: "2.0.54" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.RECREATE,
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.ROLLING_UPDATE,
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-23.DeploymentStrategy", version: "2.0.54" };
//# sourceMappingURL=data:application/json;base64,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