"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    //
    // Actions
    //
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error(`Cannot add 'Actions' to policy statement if 'NotActions' have been added`);
        }
        this.action.push(...actions);
    }
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error(`Cannot add 'NotActions' to policy statement if 'Actions' have been added`);
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error(`Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added`);
        }
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error(`Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added`);
        }
        for (const notPrincipal of notPrincipals) {
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error(`Cannot add 'Resources' to policy statement if 'NotResources' have been added`);
        }
        this.resource.push(...arns);
    }
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error(`Cannot add 'NotResources' to policy statement if 'Resources' have been added`);
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement'
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
}
exports.PolicyStatement = PolicyStatement;
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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