#!/usr/bin/env python

import json
import os
import sys
import time
from datetime import datetime

import redis
from packaging.version import parse

from clue.common.logging import get_logger
from clue.common.uid import get_random_id
from clue.constants.env import TESTING

logger = get_logger(__file__)

# Add a version warning if redis python client is < 2.10.0. Older versions
# have a connection bug that can manifest with the dispatcher.
if parse(redis.__version__) <= parse("2.10.0"):
    import warnings

    warnings.warn(
        "%s works best with redis > 2.10.0. You're running"
        " redis %s. You should upgrade." % (__name__, redis.__version__)
    )


pool: dict[tuple[str, int], redis.ConnectionPool] = {}


def now_as_iso():
    s = datetime.utcfromtimestamp(time.time()).isoformat()
    return f"{s}Z"


def reply_queue_name(prefix=None, suffix=None):
    if prefix:
        components = [prefix]
    else:
        components = []

    components.append(get_random_id())

    if suffix:
        components.append(str(suffix))

    return "-".join(components)


def retry_call(func, *args, **kw):
    max_attempts = 10
    maximum = 2
    exponent = -7

    attempts = 0
    while True:
        try:
            ret_val = func(*args, **kw)

            if exponent != -7:
                logger.info("Reconnected to Redis!")

            return ret_val

        except (redis.ConnectionError, ConnectionResetError):
            attempts += 1

            if attempts > max_attempts:
                logger.exception("Redis connection failed.")
                raise
            else:
                logger.exception("No connection to Redis, reconnecting...")
                time.sleep(2**exponent)
                exponent = exponent + 1 if exponent < maximum else exponent


def _redis_ssl_kwargs(host: str) -> dict:
    return dict(ssl_ca_certs=os.environ.get(f"{host.upper()}_ROOT_CA_PATH", "/etc/clue/ssl/clue_root-ca.crt"))


def get_client(host, port, private, password=None):
    # In case a structure is passed a client as host
    if isinstance(host, (redis.Redis, redis.StrictRedis)):
        return host

    if not host or not port or not password:
        from clue.config import config

        host = host or config.core.redis.host
        port = int(port or config.core.redis.port)
        password = config.core.redis.password

    if password:
        logger.debug("Connecting to redis with password")
    elif "pytest" not in sys.modules and not TESTING:
        logger.warning("Connecting to redis without authentication.")

    ssl_kwargs = {}

    # Automatically detect if encryption was enabled
    tmp_ssl_kwargs = _redis_ssl_kwargs(host)
    if os.path.exists(tmp_ssl_kwargs["ssl_ca_certs"]):
        ssl_kwargs = tmp_ssl_kwargs
        ssl_kwargs["ssl"] = True

    if private:
        return redis.StrictRedis(host=host, port=port, socket_keepalive=True, password=password, **ssl_kwargs)
    else:
        return redis.StrictRedis(
            connection_pool=get_pool(host, port, ssl=ssl_kwargs.get("ssl", False), password=password),
            socket_keepalive=True,
            password=password,
        )


def get_pool(host: str, port: int, ssl: bool = False, password: str | None = None):
    key = (host, port)
    connection_class = redis.connection.Connection
    connection_kwargs = {}

    if password:
        connection_kwargs["password"] = password

    if ssl:
        connection_class = redis.connection.SSLConnection  # type: ignore[assignment]
        connection_kwargs = _redis_ssl_kwargs(host)

    connection_pool = pool.get(key, None)
    if not connection_pool:
        connection_pool = redis.BlockingConnectionPool(
            host=host, port=port, max_connections=200, connection_class=connection_class, **connection_kwargs
        )
        pool[key] = connection_pool

    return connection_pool


def decode(data):
    try:
        return json.loads(data)
    except ValueError:
        logger.warning("Invalid data on queue: %s", str(data))
        return None
