"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSqs(stack, 'lambda-to-sqs-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const queue = pattern.sqsQueue;
    expect(queue).toBeDefined();
    // Assertion 3
    const dlq = pattern.deadLetterQueue;
    expect(dlq).toBeDefined();
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqs(stack, "lambda-to-sqs-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqsstackReplaceDefaultSecurityGroupsecuritygroupAED1D1EE",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqs(stack, "lambda-to-sqs-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "192.68.0.0/16",
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqsstackReplaceDefaultSecurityGroupsecuritygroupAED1D1EE",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSqs(stack, "lambda-to-sqs-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: testVpc,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqsstackReplaceDefaultSecurityGroupsecuritygroupAED1D1EE",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPC
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSqs(stack, "lambda-to-sqs-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSqs(stack, "lambda-to-sqs-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqs(stack, 'lambda-to-sqs-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        queueEnvironmentVariableName: 'CUSTOM_QUEUE_NAME'
    });
    // Assertion
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_QUEUE_NAME: {
                    Ref: 'lambdatosqsstackqueueFDDEE3DB'
                }
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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