from pathlib import Path
from typing import Dict

from _nebari.provider.terraform import (
    Data,
    Provider,
    TerraformBackend,
    tf_render_objects,
)
from _nebari.utils import deep_merge


def NebariAWSProvider(nebari_config: Dict):
    return Provider("aws", region=nebari_config["amazon_web_services"]["region"])


def NebariGCPProvider(nebari_config: Dict):
    return Provider(
        "google",
        project=nebari_config["google_cloud_platform"]["project"],
        region=nebari_config["google_cloud_platform"]["region"],
    )


def NebariAzureProvider(nebari_config: Dict):
    return Provider("azurerm", features={})


def NebariDigitalOceanProvider(nebari_config: Dict):
    return Provider("digitalocean")


def NebariKubernetesProvider(nebari_config: Dict):
    if nebari_config["provider"] == "aws":
        cluster_name = f"{nebari_config['project_name']}-{nebari_config['namespace']}"
        # The AWS provider needs to be added, as we are using aws related resources #1254
        return deep_merge(
            Data("aws_eks_cluster", "default", name=cluster_name),
            Data("aws_eks_cluster_auth", "default", name=cluster_name),
            Provider("aws", region=nebari_config["amazon_web_services"]["region"]),
            Provider(
                "kubernetes",
                experiments={"manifest_resource": True},
                host="${data.aws_eks_cluster.default.endpoint}",
                cluster_ca_certificate="${base64decode(data.aws_eks_cluster.default.certificate_authority[0].data)}",
                token="${data.aws_eks_cluster_auth.default.token}",
            ),
        )
    return Provider(
        "kubernetes",
        experiments={"manifest_resource": True},
    )


def NebariHelmProvider(nebari_config: Dict):
    if nebari_config["provider"] == "aws":
        cluster_name = f"{nebari_config['project_name']}-{nebari_config['namespace']}"

        return deep_merge(
            Data("aws_eks_cluster", "default", name=cluster_name),
            Data("aws_eks_cluster_auth", "default", name=cluster_name),
            Provider(
                "helm",
                kubernetes=dict(
                    host="${data.aws_eks_cluster.default.endpoint}",
                    cluster_ca_certificate="${base64decode(data.aws_eks_cluster.default.certificate_authority[0].data)}",
                    token="${data.aws_eks_cluster_auth.default.token}",
                ),
            ),
        )
    return Provider("helm")


def NebariTerraformState(directory: str, nebari_config: Dict):
    if nebari_config["terraform_state"]["type"] == "local":
        return {}
    elif nebari_config["terraform_state"]["type"] == "existing":
        return TerraformBackend(
            nebari_config["terraform_state"]["backend"],
            **nebari_config["terraform_state"]["config"],
        )
    elif nebari_config["provider"] == "aws":
        return TerraformBackend(
            "s3",
            bucket=f"{nebari_config['project_name']}-{nebari_config['namespace']}-terraform-state",
            key=f"terraform/{nebari_config['project_name']}-{nebari_config['namespace']}/{directory}.tfstate",
            region=nebari_config["amazon_web_services"]["region"],
            encrypt=True,
            dynamodb_table=f"{nebari_config['project_name']}-{nebari_config['namespace']}-terraform-state-lock",
        )
    elif nebari_config["provider"] == "gcp":
        return TerraformBackend(
            "gcs",
            bucket=f"{nebari_config['project_name']}-{nebari_config['namespace']}-terraform-state",
            prefix=f"terraform/{nebari_config['project_name']}/{directory}",
        )
    elif nebari_config["provider"] == "do":
        return TerraformBackend(
            "s3",
            endpoint=f"{nebari_config['digital_ocean']['region']}.digitaloceanspaces.com",
            region="us-west-1",  # fake aws region required by terraform
            bucket=f"{nebari_config['project_name']}-{nebari_config['namespace']}-terraform-state",
            key=f"terraform/{nebari_config['project_name']}-{nebari_config['namespace']}/{directory}.tfstate",
            skip_credentials_validation=True,
            skip_metadata_api_check=True,
        )
    elif nebari_config["provider"] == "azure":
        return TerraformBackend(
            "azurerm",
            resource_group_name=f"{nebari_config['project_name']}-{nebari_config['namespace']}-state",
            # storage account must be globally unique
            storage_account_name=f"{nebari_config['project_name']}{nebari_config['namespace']}{nebari_config['azure']['storage_account_postfix']}",
            container_name=f"{nebari_config['project_name']}-{nebari_config['namespace']}-state",
            key=f"terraform/{nebari_config['project_name']}-{nebari_config['namespace']}/{directory}",
        )
    elif nebari_config["provider"] == "existing":
        optional_kwargs = {}
        if "kube_context" in nebari_config["existing"]:
            optional_kwargs["confix_context"] = nebari_config["existing"][
                "kube_context"
            ]
        return TerraformBackend(
            "kubernetes",
            secret_suffix=f"{nebari_config['project_name']}-{nebari_config['namespace']}-{directory}",
            load_config_file=True,
            **optional_kwargs,
        )
    elif nebari_config["provider"] == "local":
        optional_kwargs = {}
        if "kube_context" in nebari_config["local"]:
            optional_kwargs["confix_context"] = nebari_config["local"]["kube_context"]
        return TerraformBackend(
            "kubernetes",
            secret_suffix=f"{nebari_config['project_name']}-{nebari_config['namespace']}-{directory}",
            load_config_file=True,
            **optional_kwargs,
        )
    else:
        raise NotImplementedError("state not implemented")


def stage_01_terraform_state(config):
    if config["provider"] == "gcp":
        return {
            Path("stages")
            / "01-terraform-state"
            / "gcp"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariGCPProvider(config),
                ]
            )
        }
    elif config["provider"] == "aws":
        return {
            Path("stages")
            / "01-terraform-state"
            / "aws"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariAWSProvider(config),
                ]
            )
        }
    else:
        return {}


def stage_02_infrastructure(config):
    if config["provider"] == "gcp":
        return {
            Path("stages")
            / "02-infrastructure"
            / "gcp"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariGCPProvider(config),
                    NebariTerraformState("02-infrastructure", config),
                ]
            )
        }
    elif config["provider"] == "do":
        return {
            Path("stages")
            / "02-infrastructure"
            / "do"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariTerraformState("02-infrastructure", config),
                ]
            )
        }
    elif config["provider"] == "azure":
        return {
            Path("stages")
            / "02-infrastructure"
            / "azure"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariTerraformState("02-infrastructure", config),
                ]
            ),
        }
    elif config["provider"] == "aws":
        return {
            Path("stages")
            / "02-infrastructure"
            / "aws"
            / "_nebari.tf.json": tf_render_objects(
                [
                    NebariAWSProvider(config),
                    NebariTerraformState("02-infrastructure", config),
                ]
            )
        }
    else:
        return {}


def stage_03_kubernetes_initialize(config):
    return {
        Path("stages")
        / "03-kubernetes-initialize"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("03-kubernetes-initialize", config),
                NebariKubernetesProvider(config),
                NebariHelmProvider(config),
            ]
        ),
    }


def stage_04_kubernetes_ingress(config):
    return {
        Path("stages")
        / "04-kubernetes-ingress"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("04-kubernetes-ingress", config),
                NebariKubernetesProvider(config),
                NebariHelmProvider(config),
            ]
        ),
    }


def stage_05_kubernetes_keycloak(config):
    return {
        Path("stages")
        / "05-kubernetes-keycloak"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("05-kubernetes-keycloak", config),
                NebariKubernetesProvider(config),
                NebariHelmProvider(config),
            ]
        ),
    }


def stage_06_kubernetes_keycloak_configuration(config):
    return {
        Path("stages")
        / "06-kubernetes-keycloak-configuration"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("06-kubernetes-keycloak-configuration", config),
            ]
        ),
    }


def stage_07_kubernetes_services(config):
    return {
        Path("stages")
        / "07-kubernetes-services"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("07-kubernetes-services", config),
                NebariKubernetesProvider(config),
                NebariHelmProvider(config),
            ]
        ),
    }


def stage_08_nebari_tf_extensions(config):
    return {
        Path("stages")
        / "08-nebari-tf-extensions"
        / "_nebari.tf.json": tf_render_objects(
            [
                NebariTerraformState("08-nebari-tf-extensions", config),
                NebariKubernetesProvider(config),
                NebariHelmProvider(config),
            ]
        ),
    }
