# Core System - Project Setup Orchestrator

## Purpose

Master orchestrator that sets up your project infrastructure after SpecKit creates the spec. Uses intelligent agents to analyze projects and coordinate subsystem setup automatically.

## What It Does

1. **Analyzes project structure** - Detects type, stack, and prerequisites intelligently
2. **Orchestrates subsystems** - Coordinates security, deployment, testing, docs setup
3. **Manages upgrades** - Handles incremental project type upgrades (landing-page → ai-app)
4. **Provides utilities** - Scripts and templates for project analysis and configuration

## Agents Used

Core uses 3 specialized agents for intelligent orchestration:

- **project-analyzer** - Analyzes project structure to detect type, stack, and prerequisites
  - Runs detection scripts (detect-project-type.sh, analyze-project-structure.sh)
  - Validates prerequisites before setup begins
  - Generates JSON analysis reports

- **project-setup-orchestrator** - Coordinates complete project setup across all subsystems
  - Creates `.multiagent/config.json` with type-specific settings
  - Invokes subsystem commands in proper sequence
  - Configures MCP servers, workflows, and dependencies
  - Generates SETUP_COMPLETE.md report

- **upgrade-orchestrator** - Manages project type upgrades while preserving existing code
  - Validates upgrade paths (no downgrades)
  - Backs up configuration before changes
  - Adds new infrastructure incrementally (backend, AI, observability)
  - Generates UPGRADE_COMPLETE.md report

## Commands

### Core Commands (via Slash Commands)

**`/core:project-setup [spec-directory]`**
- **Purpose**: Orchestrate complete project setup after spec creation
- **Agent**: project-setup-orchestrator
- **What it does**:
  - Analyzes project type and stack
  - Creates `.multiagent/config.json`
  - Configures subsystems (security, deployment, testing, docs)
  - Sets up MCP servers and GitHub workflows
- **Output**: `SETUP_COMPLETE.md` report

**`/core:build [--sync-all]`**
- **Purpose**: Build multiagent-core package from templates
- **What it does**:
  - Syncs templates from ~/.multiagent/ to package
  - Builds Python package (wheel + sdist)
  - Optionally syncs to all registered projects
- **Output**: `dist/*.whl`, `dist/*.tar.gz`
- **Note**: Framework development only

**`/core:upgrade-to <project-type>`**
- **Purpose**: Upgrade project to more complex type
- **Agent**: upgrade-orchestrator
- **What it does**:
  - Validates upgrade path (landing-page → website → web-app → ai-app → saas)
  - Backs up configuration
  - Adds new infrastructure incrementally
  - Preserves existing code
- **Output**: `UPGRADE_COMPLETE.md` report
- **Examples**:
  - `/core:upgrade-to web-app` - Add backend capabilities
  - `/core:upgrade-to ai-app` - Add AI infrastructure
  - `/core:upgrade-to saas` - Add full production stack

### Commands It Invokes

During project setup, core orchestrates these subsystem commands:
- `/security:setup` - Configure security and environment
- `/deployment:deploy-prepare` - Generate deployment configs
- `/testing:test-generate` - Create test structure
- `/mcp:add <servers>` - Configure MCP servers
- `/docs:init` - Initialize documentation

## Directory Structure

```
.multiagent/core/
├── scripts/                          # Utility scripts for mechanical operations
│   ├── analyze-project-structure.sh  # Detects stack, framework, directories (JSON output)
│   ├── detect-project-type.sh        # Infers project type from spec.md keywords
│   ├── validate-prerequisites.sh     # Checks prerequisites before setup
│   └── backup-configuration.sh       # Creates timestamped config backups
├── templates/
│   ├── github-workflows/             # CI/CD workflow templates
│   │   ├── agent-pr-review.yml       # PR review automation
│   │   ├── auto-update-projects.yml  # Auto-update mechanism
│   │   ├── version-management.yml    # Semantic versioning
│   │   └── feedback-webhook-handler.yml  # Webhook handling
│   └── github-config/                # GitHub configuration
│       ├── ISSUE_TEMPLATE/           # Issue templates
│       ├── copilot-instructions.md   # Copilot config
│       └── labels.yml                # GitHub labels
└── README.md                         # This file
```

## Outputs

### 1. GitHub Workflows (`.github/workflows/`)

Generated by `configure-workflows.sh` from templates:

| File | Purpose | Variables Filled |
|------|---------|-----------------|
| `ci.yml` | Run tests, lint, build | NODE_VERSION, TEST_COMMAND, BUILD_COMMAND |
| `deploy.yml` | Deploy to cloud | DEPLOYMENT_TARGET (vercel/aws/docker) |
| `security.yml` | Security scanning | Language-specific scanners |

### 2. Git Hooks (`.git/hooks/`)

| File | Purpose | When Run |
|------|---------|----------|
| `pre-commit` | Lint and format | Before each commit |
| `pre-push` | Run tests | Before push to remote |

### 3. Configuration Files (Root Directory)

| File | Purpose | Content |
|------|---------|---------|
| `.env.example` | Environment template | Variables detected from spec |
| `.gitmessage` | Commit template | Standard commit format |

## Workflow

```mermaid
graph TD
    A[multiagent init] -->|Copies templates| B[.multiagent/core]
    B --> C[/specify]
    C -->|Creates spec| D[specs/001-*/spec.md]
    D --> E[/plan]
    E -->|Triggers hook| F[post-plan-setup.sh]
    F -->|Runs| G[/project-setup]
    G --> H[configure-workflows.sh]
    H -->|Creates| I[.github/workflows/*.yml]
    G --> J[/deploy-prepare]
    J -->|Creates| K[deployment/*]
    G --> L[/testing-workflow]
    L -->|Creates| M[tests/*]
    G --> N[setup-checklist.sh]
    N -->|Validates| O[Setup Complete]
```

## Template Variables

Templates use `{{VARIABLE|default}}` format:

### Common Variables
- `{{PROJECT_NAME|my-project}}`
- `{{NODE_VERSION|20}}` or `{{PYTHON_VERSION|3.11}}`
- `{{TEST_COMMAND|npm test}}` or `{{TEST_COMMAND|pytest}}`
- `{{BUILD_COMMAND|npm run build}}`
- `{{DEPLOYMENT_TARGET|docker}}`

### Conditional Blocks
```yaml
{{#IF_NODE}}
  - name: Run npm audit
    run: npm audit
{{/IF_NODE}}
```

## How Templates Work

1. **Spec Analysis**: Script reads `specs/001-*/spec.md`
2. **Detection**: Identifies project type (Node.js, Python, etc.)
3. **Variable Setting**: Sets appropriate values
4. **Template Processing**: Replaces variables and conditionals
5. **Output Generation**: Creates final workflow files

## Integration Points

### With Deployment System
- Core invokes `/deploy-prepare`
- Deployment creates `deployment/` directory
- Core doesn't duplicate deployment functionality

### With Testing System
- Core invokes `/testing-workflow --generate`
- Testing creates `tests/` directory
- Core doesn't create test files

### With PR Review System
- PR review workflows remain untouched
- Core preserves specialized workflows like `claude-code-review.yml`

## Validation

Run setup checklist to verify:
```bash
.multiagent/core/scripts/validation/setup-checklist.sh
```

Output:
```
✅ multiagent init completed
✅ Spec directory exists
✅ GitHub workflows exist
✅ Deployment configs exist (via deployment system)
✅ Test structure exists (via testing system)
✅ Git hooks configured
✅ Environment template exists
```

## Manual Post-Setup

After automatic setup:

1. **Add GitHub Secrets**:
   ```bash
   gh secret set VERCEL_TOKEN
   gh secret set CLAUDE_API_KEY
   ```

2. **Configure Environment**:
   ```bash
   cp .env.example .env
   # Edit with actual values
   ```

3. **Test Everything**:
   ```bash
   npm test  # or pytest
   ```

## Runtime Utilities

Available to all systems:
```python
from .multiagent.core.templates.runtime import Logger

logger = Logger("my-script")
logger.info("Operation complete")
```

## Key Points

- **Core orchestrates, doesn't duplicate**
- **Each system owns its domain** (deployment owns deployment/, testing owns tests/)
- **Templates are spec-driven** (not generic)
- **Workflows are project-specific** (Node.js gets different CI than Python)

## Related Documentation

- **Build System**: `../../template-management/README.md` - How templates get packaged and deployed
- **Initialization Flow**: `../ README.md` - User-facing initialization explanation
- **Deployment System**: `../deployment/README.md` - Deployment config generation
- **Testing System**: `../testing/README.md` - Test structure generation