# API Client Template
# Purpose: Generate HTTP client setup with authentication, error handling, and request/response interceptors
# Variables: {{PROJECT_NAME}}, {{API_BASE_URL}}, {{AUTH_TYPE}}, {{API_VERSION}}, {{CURRENT_DATE}}, {{API_BASE_URL_ENV}}
# Generated by: frontend-developer agent
# Template location: multiagent_core/templates/.multiagent/frontend/templates/API_CLIENT.md

# API Client Architecture

## Overview

**Project**: {{PROJECT_NAME}}
**Base URL**: {{API_BASE_URL}}
**Auth Type**: {{AUTH_TYPE}} (JWT/OAuth/API Key/Session)
**Version**: {{API_VERSION}}
**Last Updated**: {{CURRENT_DATE}}

## API Client Setup

### Base Configuration

```typescript
// src/lib/api/client.ts
import axios from 'axios';

const apiClient = axios.create({
  baseURL: process.env.{{API_BASE_URL_ENV}},
  timeout: {{TIMEOUT}},
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

export default apiClient;
```

### Environment Variables

```bash
# .env.local
{{API_BASE_URL_ENV}}={{API_BASE_URL}}
{{API_KEY_ENV}}={{API_KEY}}
{{AUTH_TOKEN_ENV}}={{AUTH_TOKEN}}
```

## Authentication

### Token Management

```typescript
// src/lib/api/auth.ts
export const AuthService = {
  // Get token from storage
  getToken(): string | null {
    return localStorage.getItem('{{TOKEN_KEY}}');
  },

  // Set token in storage
  setToken(token: string): void {
    localStorage.setItem('{{TOKEN_KEY}}', token);
  },

  // Remove token
  clearToken(): void {
    localStorage.removeItem('{{TOKEN_KEY}}');
  },

  // Check if authenticated
  isAuthenticated(): boolean {
    const token = this.getToken();
    if (!token) return false;

    // Check token expiry
    const decoded = {{DECODE_JWT}}(token);
    return decoded.exp > Date.now() / 1000;
  },
};
```

### Request Interceptors

```typescript
// Add auth token to requests
apiClient.interceptors.request.use(
  (config) => {
    const token = AuthService.getToken();
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }
    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);
```

### Response Interceptors

```typescript
// Handle 401 unauthorized
apiClient.interceptors.response.use(
  (response) => response,
  async (error) => {
    const originalRequest = error.config;

    if (error.response?.status === 401 && !originalRequest._retry) {
      originalRequest._retry = true;

      try {
        // Refresh token logic
        const newToken = await {{REFRESH_TOKEN_FUNCTION}}();
        AuthService.setToken(newToken);

        originalRequest.headers.Authorization = `Bearer ${newToken}`;
        return apiClient(originalRequest);
      } catch (refreshError) {
        // Redirect to login
        AuthService.clearToken();
        window.location.href = '/login';
        return Promise.reject(refreshError);
      }
    }

    return Promise.reject(error);
  }
);
```

## API Endpoints

### Endpoint Registry

```typescript
// src/lib/api/endpoints.ts
export const API_ENDPOINTS = {
  // Auth
  AUTH: {
    LOGIN: '/auth/login',
    LOGOUT: '/auth/logout',
    REGISTER: '/auth/register',
    REFRESH: '/auth/refresh',
    VERIFY: '/auth/verify',
  },

  // Users
  USERS: {
    LIST: '/users',
    GET: (id: string) => `/users/${id}`,
    CREATE: '/users',
    UPDATE: (id: string) => `/users/${id}`,
    DELETE: (id: string) => `/users/${id}`,
  },

  // {{RESOURCE_1}}
  {{RESOURCE_1}}: {
    LIST: '/{{RESOURCE_1_PATH}}',
    GET: (id: string) => `/{{RESOURCE_1_PATH}}/${id}`,
    CREATE: '/{{RESOURCE_1_PATH}}',
    UPDATE: (id: string) => `/{{RESOURCE_1_PATH}}/${id}`,
    DELETE: (id: string) => `/{{RESOURCE_1_PATH}}/${id}`,
  },

  // {{RESOURCE_2}}
  {{RESOURCE_2}}: {
    LIST: '/{{RESOURCE_2_PATH}}',
    GET: (id: string) => `/{{RESOURCE_2_PATH}}/${id}`,
    CREATE: '/{{RESOURCE_2_PATH}}',
    UPDATE: (id: string) => `/{{RESOURCE_2_PATH}}/${id}`,
    DELETE: (id: string) => `/{{RESOURCE_2_PATH}}/${id}`,
  },
} as const;
```

## Type Definitions

### API Response Types

```typescript
// src/lib/api/types.ts

// Generic API Response
export interface ApiResponse<T> {
  data: T;
  message: string;
  success: boolean;
  timestamp: string;
}

// Paginated Response
export interface PaginatedResponse<T> {
  data: T[];
  pagination: {
    page: number;
    pageSize: number;
    total: number;
    totalPages: number;
  };
}

// Error Response
export interface ApiError {
  message: string;
  code: string;
  details?: Record<string, string[]>;
  statusCode: number;
}

// Resource Types
export interface {{RESOURCE_1_TYPE}} {
  id: string;
  {{FIELD_1}}: {{TYPE_1}};
  {{FIELD_2}}: {{TYPE_2}};
  createdAt: string;
  updatedAt: string;
}

export interface {{RESOURCE_2_TYPE}} {
  id: string;
  {{FIELD_1}}: {{TYPE_1}};
  {{FIELD_2}}: {{TYPE_2}};
  createdAt: string;
  updatedAt: string;
}
```

## Service Layer

### Generic CRUD Service

```typescript
// src/lib/api/services/base.service.ts
export class BaseService<T> {
  constructor(private endpoint: string) {}

  async list(params?: Record<string, any>): Promise<T[]> {
    const response = await apiClient.get<ApiResponse<T[]>>(
      this.endpoint,
      { params }
    );
    return response.data.data;
  }

  async get(id: string): Promise<T> {
    const response = await apiClient.get<ApiResponse<T>>(
      `${this.endpoint}/${id}`
    );
    return response.data.data;
  }

  async create(data: Partial<T>): Promise<T> {
    const response = await apiClient.post<ApiResponse<T>>(
      this.endpoint,
      data
    );
    return response.data.data;
  }

  async update(id: string, data: Partial<T>): Promise<T> {
    const response = await apiClient.put<ApiResponse<T>>(
      `${this.endpoint}/${id}`,
      data
    );
    return response.data.data;
  }

  async delete(id: string): Promise<void> {
    await apiClient.delete(`${this.endpoint}/${id}`);
  }
}
```

### Resource-Specific Services

```typescript
// src/lib/api/services/{{RESOURCE_1}}.service.ts
import { BaseService } from './base.service';
import { {{RESOURCE_1_TYPE}} } from '../types';
import { API_ENDPOINTS } from '../endpoints';

class {{RESOURCE_1}}ServiceClass extends BaseService<{{RESOURCE_1_TYPE}}> {
  constructor() {
    super(API_ENDPOINTS.{{RESOURCE_1}}.LIST);
  }

  // Custom methods specific to this resource
  async {{CUSTOM_METHOD_1}}(id: string, data: any): Promise<{{RESOURCE_1_TYPE}}> {
    const response = await apiClient.post<ApiResponse<{{RESOURCE_1_TYPE}}>>(
      `${this.endpoint}/${id}/{{CUSTOM_ENDPOINT}}`,
      data
    );
    return response.data.data;
  }

  async {{CUSTOM_METHOD_2}}(params: any): Promise<{{RESOURCE_1_TYPE}}[]> {
    const response = await apiClient.get<ApiResponse<{{RESOURCE_1_TYPE}}[]>>(
      `${this.endpoint}/{{CUSTOM_ENDPOINT}}`,
      { params }
    );
    return response.data.data;
  }
}

export const {{RESOURCE_1}}Service = new {{RESOURCE_1}}ServiceClass();
```

## React Query Integration

### Query Hooks

```typescript
// src/lib/api/hooks/use{{RESOURCE_1}}.ts
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { {{RESOURCE_1}}Service } from '../services/{{RESOURCE_1}}.service';
import { {{RESOURCE_1_TYPE}} } from '../types';

// Query Keys
export const {{RESOURCE_1}}_KEYS = {
  all: ['{{RESOURCE_1}}'] as const,
  lists: () => [...{{RESOURCE_1}}_KEYS.all, 'list'] as const,
  list: (filters: string) => [...{{RESOURCE_1}}_KEYS.lists(), filters] as const,
  details: () => [...{{RESOURCE_1}}_KEYS.all, 'detail'] as const,
  detail: (id: string) => [...{{RESOURCE_1}}_KEYS.details(), id] as const,
};

// List Hook
export function use{{RESOURCE_1}}List(params?: Record<string, any>) {
  return useQuery({
    queryKey: {{RESOURCE_1}}_KEYS.list(JSON.stringify(params)),
    queryFn: () => {{RESOURCE_1}}Service.list(params),
  });
}

// Detail Hook
export function use{{RESOURCE_1}}(id: string) {
  return useQuery({
    queryKey: {{RESOURCE_1}}_KEYS.detail(id),
    queryFn: () => {{RESOURCE_1}}Service.get(id),
    enabled: !!id,
  });
}

// Create Mutation
export function useCreate{{RESOURCE_1}}() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (data: Partial<{{RESOURCE_1_TYPE}}>) =>
      {{RESOURCE_1}}Service.create(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: {{RESOURCE_1}}_KEYS.lists() });
    },
  });
}

// Update Mutation
export function useUpdate{{RESOURCE_1}}() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, data }: { id: string; data: Partial<{{RESOURCE_1_TYPE}}> }) =>
      {{RESOURCE_1}}Service.update(id, data),
    onSuccess: (_, { id }) => {
      queryClient.invalidateQueries({ queryKey: {{RESOURCE_1}}_KEYS.detail(id) });
      queryClient.invalidateQueries({ queryKey: {{RESOURCE_1}}_KEYS.lists() });
    },
  });
}

// Delete Mutation
export function useDelete{{RESOURCE_1}}() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: string) => {{RESOURCE_1}}Service.delete(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: {{RESOURCE_1}}_KEYS.lists() });
    },
  });
}
```

## Error Handling

### Error Handler Utility

```typescript
// src/lib/api/error-handler.ts
import { AxiosError } from 'axios';
import { ApiError } from './types';

export function handleApiError(error: unknown): ApiError {
  if (error instanceof AxiosError) {
    const apiError: ApiError = {
      message: error.response?.data?.message || 'An error occurred',
      code: error.response?.data?.code || 'UNKNOWN_ERROR',
      details: error.response?.data?.details,
      statusCode: error.response?.status || 500,
    };
    return apiError;
  }

  return {
    message: 'An unexpected error occurred',
    code: 'UNEXPECTED_ERROR',
    statusCode: 500,
  };
}

export function isApiError(error: unknown): error is ApiError {
  return (
    typeof error === 'object' &&
    error !== null &&
    'message' in error &&
    'code' in error
  );
}
```

### Error Boundary Component

```typescript
// src/components/ErrorBoundary.tsx
export function ApiErrorDisplay({ error }: { error: ApiError }) {
  return (
    <div className="error-container">
      <h3>Error {error.statusCode}</h3>
      <p>{error.message}</p>
      {error.details && (
        <ul>
          {Object.entries(error.details).map(([field, errors]) => (
            <li key={field}>
              <strong>{field}:</strong> {errors.join(', ')}
            </li>
          ))}
        </ul>
      )}
    </div>
  );
}
```

## Caching Strategy

### React Query Configuration

```typescript
// src/lib/api/query-client.ts
import { QueryClient } from '@tanstack/react-query';

export const queryClient = new QueryClient({
  defaultOptions: {
    queries: {
      staleTime: {{STALE_TIME}}, // 5 minutes
      cacheTime: {{CACHE_TIME}}, // 30 minutes
      retry: {{RETRY_COUNT}},
      refetchOnWindowFocus: {{REFETCH_ON_FOCUS}},
    },
  },
});
```

### Cache Invalidation

```typescript
// Invalidate specific resource
queryClient.invalidateQueries({ queryKey: {{RESOURCE_1}}_KEYS.lists() });

// Invalidate all
queryClient.invalidateQueries();

// Remove from cache
queryClient.removeQueries({ queryKey: {{RESOURCE_1}}_KEYS.detail(id) });
```

## Request/Response Logging

### Development Logger

```typescript
// src/lib/api/logger.ts
apiClient.interceptors.request.use((config) => {
  if (process.env.NODE_ENV === 'development') {
    console.log('🚀 Request:', {
      method: config.method?.toUpperCase(),
      url: config.url,
      params: config.params,
      data: config.data,
    });
  }
  return config;
});

apiClient.interceptors.response.use(
  (response) => {
    if (process.env.NODE_ENV === 'development') {
      console.log('✅ Response:', {
        status: response.status,
        url: response.config.url,
        data: response.data,
      });
    }
    return response;
  },
  (error) => {
    if (process.env.NODE_ENV === 'development') {
      console.error('❌ Error:', {
        status: error.response?.status,
        url: error.config?.url,
        message: error.message,
        data: error.response?.data,
      });
    }
    return Promise.reject(error);
  }
);
```

## Testing

### Mock Service Worker Setup

```typescript
// src/mocks/handlers.ts
import { rest } from 'msw';
import { API_ENDPOINTS } from '@/lib/api/endpoints';

export const handlers = [
  // List
  rest.get(API_ENDPOINTS.{{RESOURCE_1}}.LIST, (req, res, ctx) => {
    return res(
      ctx.status(200),
      ctx.json({
        data: [{{MOCK_DATA}}],
        success: true,
      })
    );
  }),

  // Get by ID
  rest.get(API_ENDPOINTS.{{RESOURCE_1}}.GET(':id'), (req, res, ctx) => {
    return res(
      ctx.status(200),
      ctx.json({
        data: {{MOCK_SINGLE_DATA}},
        success: true,
      })
    );
  }),

  // Create
  rest.post(API_ENDPOINTS.{{RESOURCE_1}}.CREATE, async (req, res, ctx) => {
    const body = await req.json();
    return res(
      ctx.status(201),
      ctx.json({
        data: { id: '{{MOCK_ID}}', ...body },
        success: true,
      })
    );
  }),
];
```

### API Test Utilities

```typescript
// src/__tests__/utils/api-test-utils.ts
import { renderHook, waitFor } from '@testing-library/react';
import { QueryClient, QueryClientProvider } from '@tanstack/react-query';

export function createTestQueryClient() {
  return new QueryClient({
    defaultOptions: {
      queries: { retry: false },
      mutations: { retry: false },
    },
  });
}

export function createWrapper() {
  const testQueryClient = createTestQueryClient();
  return ({ children }: { children: React.ReactNode }) => (
    <QueryClientProvider client={testQueryClient}>
      {children}
    </QueryClientProvider>
  );
}
```

## Best Practices

### 1. Type Safety
- Always define TypeScript interfaces for request/response
- Use generic types for reusable code
- Validate data at runtime with Zod or similar

### 2. Error Handling
- Catch and transform all errors
- Provide user-friendly error messages
- Log errors for debugging

### 3. Performance
- Use React Query for caching
- Implement pagination for large lists
- Debounce search queries
- Use optimistic updates for better UX

### 4. Security
- Never store sensitive data in localStorage without encryption
- Implement CSRF protection
- Use HTTPS only
- Sanitize user inputs

### 5. Testing
- Mock API calls in tests
- Test error scenarios
- Test loading states
- Test cache invalidation

## Migration Guide

### From Fetch to Axios

```typescript
// Before (fetch)
const response = await fetch(url, {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify(data),
});
const json = await response.json();

// After (axios)
const response = await apiClient.post(url, data);
const json = response.data;
```

### From useEffect to React Query

```typescript
// Before (useEffect)
const [data, setData] = useState(null);
const [loading, setLoading] = useState(true);

useEffect(() => {
  fetchData().then(setData).finally(() => setLoading(false));
}, []);

// After (React Query)
const { data, isLoading } = use{{RESOURCE}}();
```

## Resources

- [Axios Documentation](https://axios-http.com/)
- [React Query Documentation](https://tanstack.com/query)
- [API Endpoint Registry]({{ENDPOINT_DOCS_URL}})
- [Authentication Flow]({{AUTH_DOCS_URL}})

---

**Maintainer**: {{MAINTAINER_NAME}}
**Last Updated**: {{CURRENT_DATE}}
**API Version**: {{API_VERSION}}
