# State Management Template
# Purpose: Define state management strategy, store structure, and data flow patterns
# Variables: {{PROJECT_NAME}}, {{STATE_LIBRARY}}, {{VERSION}}, {{CURRENT_DATE}}
# Generated by: frontend-developer agent
# Template location: multiagent_core/templates/.multiagent/frontend/templates/STATE_ARCHITECTURE.md

# State Management Architecture

## Overview

**Project**: {{PROJECT_NAME}}
**State Management**: {{STATE_LIBRARY}} (Redux/Zustand/Context/Jotai/Recoil)
**Version**: {{VERSION}}
**Last Updated**: {{CURRENT_DATE}}

## State Management Strategy

### State Categories

1. **Server State**: Data fetched from APIs (managed by React Query/SWR)
2. **UI State**: Component-local state, modals, dropdowns (React useState)
3. **Global State**: Shared across components (Redux/Zustand/Context)
4. **URL State**: Router params, query strings (React Router/Next.js Router)
5. **Form State**: Form data, validation (React Hook Form/Formik)

### Decision Matrix

| State Type | Tool | Reason |
|------------|------|--------|
| API Data | React Query | Caching, revalidation, loading states |
| Global App State | {{STATE_LIBRARY}} | Theme, user, preferences |
| Component State | useState | Local, not shared |
| Form State | {{FORM_LIBRARY}} | Validation, submission |
| URL State | Router | Shareable, bookmarkable |

## Global State Setup

### Store Configuration ({{STATE_LIBRARY}})

#### Redux Toolkit

```typescript
// src/store/index.ts
import { configureStore } from '@reduxjs/toolkit';
import {{SLICE_1}}Reducer from './slices/{{SLICE_1}}';
import {{SLICE_2}}Reducer from './slices/{{SLICE_2}}';

export const store = configureStore({
  reducer: {
    {{SLICE_1}}: {{SLICE_1}}Reducer,
    {{SLICE_2}}: {{SLICE_2}}Reducer,
  },
  middleware: (getDefaultMiddleware) =>
    getDefaultMiddleware({
      serializableCheck: {{SERIALIZABLE_CHECK}},
    }),
});

export type RootState = ReturnType<typeof store.getState>;
export type AppDispatch = typeof store.dispatch;
```

#### Zustand

```typescript
// src/store/index.ts
import { create } from 'zustand';
import { devtools, persist } from 'zustand/middleware';

interface {{STORE_NAME}}State {
  // State
  {{STATE_FIELD_1}}: {{TYPE_1}};
  {{STATE_FIELD_2}}: {{TYPE_2}};

  // Actions
  {{ACTION_1}}: ({{PARAMS}}) => void;
  {{ACTION_2}}: ({{PARAMS}}) => void;
}

export const use{{STORE_NAME}} = create<{{STORE_NAME}}State>()(
  devtools(
    persist(
      (set) => ({
        // Initial state
        {{STATE_FIELD_1}}: {{INITIAL_VALUE_1}},
        {{STATE_FIELD_2}}: {{INITIAL_VALUE_2}},

        // Actions
        {{ACTION_1}}: ({{PARAMS}}) =>
          set((state) => ({ {{STATE_UPDATE}} })),

        {{ACTION_2}}: ({{PARAMS}}) =>
          set((state) => ({ {{STATE_UPDATE}} })),
      }),
      {
        name: '{{STORE_NAME}}-storage',
      }
    )
  )
);
```

#### React Context

```typescript
// src/contexts/{{CONTEXT_NAME}}Context.tsx
import { createContext, useContext, useState, ReactNode } from 'react';

interface {{CONTEXT_NAME}}ContextType {
  {{STATE_FIELD_1}}: {{TYPE_1}};
  {{STATE_FIELD_2}}: {{TYPE_2}};
  {{ACTION_1}}: ({{PARAMS}}) => void;
  {{ACTION_2}}: ({{PARAMS}}) => void;
}

const {{CONTEXT_NAME}}Context = createContext<{{CONTEXT_NAME}}ContextType | undefined>(
  undefined
);

export function {{CONTEXT_NAME}}Provider({ children }: { children: ReactNode }) {
  const [{{STATE_FIELD_1}}, set{{STATE_FIELD_1}}] = useState<{{TYPE_1}}>({{INITIAL}});
  const [{{STATE_FIELD_2}}, set{{STATE_FIELD_2}}] = useState<{{TYPE_2}}>({{INITIAL}});

  const {{ACTION_1}} = ({{PARAMS}}) => {
    // Action logic
  };

  const {{ACTION_2}} = ({{PARAMS}}) => {
    // Action logic
  };

  return (
    <{{CONTEXT_NAME}}Context.Provider
      value={{
        {{STATE_FIELD_1}},
        {{STATE_FIELD_2}},
        {{ACTION_1}},
        {{ACTION_2}},
      }}
    >
      {children}
    </{{CONTEXT_NAME}}Context.Provider>
  );
}

export function use{{CONTEXT_NAME}}() {
  const context = useContext({{CONTEXT_NAME}}Context);
  if (context === undefined) {
    throw new Error('use{{CONTEXT_NAME}} must be used within {{CONTEXT_NAME}}Provider');
  }
  return context;
}
```

## State Slices

### Slice Structure (Redux Toolkit)

```typescript
// src/store/slices/{{SLICE_NAME}}.ts
import { createSlice, PayloadAction } from '@reduxjs/toolkit';

export interface {{SLICE_NAME}}State {
  {{FIELD_1}}: {{TYPE_1}};
  {{FIELD_2}}: {{TYPE_2}};
  loading: boolean;
  error: string | null;
}

const initialState: {{SLICE_NAME}}State = {
  {{FIELD_1}}: {{INITIAL_1}},
  {{FIELD_2}}: {{INITIAL_2}},
  loading: false,
  error: null,
};

const {{SLICE_NAME}}Slice = createSlice({
  name: '{{SLICE_NAME}}',
  initialState,
  reducers: {
    {{ACTION_1}}: (state, action: PayloadAction<{{PAYLOAD_TYPE}}>) => {
      // Update state
      state.{{FIELD_1}} = action.payload;
    },

    {{ACTION_2}}: (state, action: PayloadAction<{{PAYLOAD_TYPE}}>) => {
      // Update state
      state.{{FIELD_2}} = action.payload;
    },

    setLoading: (state, action: PayloadAction<boolean>) => {
      state.loading = action.payload;
    },

    setError: (state, action: PayloadAction<string | null>) => {
      state.error = action.payload;
    },

    reset: (state) => {
      Object.assign(state, initialState);
    },
  },
});

export const {
  {{ACTION_1}},
  {{ACTION_2}},
  setLoading,
  setError,
  reset,
} = {{SLICE_NAME}}Slice.actions;

export default {{SLICE_NAME}}Slice.reducer;
```

### Async Actions (Redux Thunk)

```typescript
// src/store/slices/{{SLICE_NAME}}.ts
import { createAsyncThunk } from '@reduxjs/toolkit';
import { {{SERVICE_NAME}} } from '@/lib/api/services';

export const fetch{{RESOURCE}} = createAsyncThunk(
  '{{SLICE_NAME}}/fetch{{RESOURCE}}',
  async (params: {{PARAMS_TYPE}}, { rejectWithValue }) => {
    try {
      const data = await {{SERVICE_NAME}}.list(params);
      return data;
    } catch (error) {
      return rejectWithValue(handleApiError(error));
    }
  }
);

// Add to slice
extraReducers: (builder) => {
  builder
    .addCase(fetch{{RESOURCE}}.pending, (state) => {
      state.loading = true;
      state.error = null;
    })
    .addCase(fetch{{RESOURCE}}.fulfilled, (state, action) => {
      state.loading = false;
      state.{{FIELD}} = action.payload;
    })
    .addCase(fetch{{RESOURCE}}.rejected, (state, action) => {
      state.loading = false;
      state.error = action.payload as string;
    });
},
```

## Selectors

### Basic Selectors

```typescript
// src/store/selectors/{{SLICE_NAME}}.selectors.ts
import { RootState } from '../index';

export const select{{FIELD_1}} = (state: RootState) =>
  state.{{SLICE_NAME}}.{{FIELD_1}};

export const select{{FIELD_2}} = (state: RootState) =>
  state.{{SLICE_NAME}}.{{FIELD_2}};

export const selectLoading = (state: RootState) =>
  state.{{SLICE_NAME}}.loading;

export const selectError = (state: RootState) =>
  state.{{SLICE_NAME}}.error;
```

### Memoized Selectors (Reselect)

```typescript
// src/store/selectors/{{SLICE_NAME}}.selectors.ts
import { createSelector } from '@reduxjs/toolkit';
import { RootState } from '../index';

const select{{SLICE_NAME}} = (state: RootState) => state.{{SLICE_NAME}};

export const select{{DERIVED_VALUE}} = createSelector(
  [select{{SLICE_NAME}}],
  ({{SLICE_NAME}}) => {
    // Compute derived value
    return {{COMPUTATION}};
  }
);

export const select{{FILTERED_LIST}} = createSelector(
  [select{{FIELD_1}}, (state: RootState, filter: string) => filter],
  ({{FIELD_1}}, filter) => {
    return {{FIELD_1}}.filter(item => {{FILTER_LOGIC}});
  }
);
```

## Custom Hooks

### Typed Redux Hooks

```typescript
// src/store/hooks.ts
import { TypedUseSelectorHook, useDispatch, useSelector } from 'react-redux';
import type { RootState, AppDispatch } from './index';

export const useAppDispatch = () => useDispatch<AppDispatch>();
export const useAppSelector: TypedUseSelectorHook<RootState> = useSelector;
```

### Abstraction Hooks

```typescript
// src/hooks/use{{FEATURE}}.ts
import { useAppDispatch, useAppSelector } from '@/store/hooks';
import { {{ACTIONS}} } from '@/store/slices/{{SLICE_NAME}}';
import { {{SELECTORS}} } from '@/store/selectors/{{SLICE_NAME}}.selectors';

export function use{{FEATURE}}() {
  const dispatch = useAppDispatch();
  const {{STATE_1}} = useAppSelector({{SELECTOR_1}});
  const {{STATE_2}} = useAppSelector({{SELECTOR_2}});

  const {{ACTION_1}} = ({{PARAMS}}) => {
    dispatch({{ACTION}}({{PARAMS}}));
  };

  const {{ACTION_2}} = ({{PARAMS}}) => {
    dispatch({{ACTION}}({{PARAMS}}));
  };

  return {
    {{STATE_1}},
    {{STATE_2}},
    {{ACTION_1}},
    {{ACTION_2}},
  };
}
```

## State Persistence

### Local Storage Persistence

```typescript
// src/store/middleware/persistence.ts
import { Middleware } from '@reduxjs/toolkit';

const STORAGE_KEY = '{{APP_NAME}}-state';

// Save middleware
export const saveStateMiddleware: Middleware = (store) => (next) => (action) => {
  const result = next(action);
  const state = store.getState();

  try {
    const serializedState = JSON.stringify({
      {{SLICE_1}}: state.{{SLICE_1}},
      {{SLICE_2}}: state.{{SLICE_2}},
    });
    localStorage.setItem(STORAGE_KEY, serializedState);
  } catch (error) {
    console.error('Failed to save state:', error);
  }

  return result;
};

// Load state
export function loadState() {
  try {
    const serializedState = localStorage.getItem(STORAGE_KEY);
    if (serializedState === null) {
      return undefined;
    }
    return JSON.parse(serializedState);
  } catch (error) {
    console.error('Failed to load state:', error);
    return undefined;
  }
}
```

### Zustand Persistence

```typescript
// Persistence is built-in
export const useStore = create<State>()(
  persist(
    (set) => ({
      // State and actions
    }),
    {
      name: 'app-storage',
      // Optionally specify which fields to persist
      partialize: (state) => ({
        {{FIELD_1}}: state.{{FIELD_1}},
        {{FIELD_2}}: state.{{FIELD_2}},
      }),
    }
  )
);
```

## State Normalization

### Normalized Structure

```typescript
// Instead of nested arrays
interface BadState {
  posts: {
    id: string;
    title: string;
    comments: Array<{
      id: string;
      text: string;
    }>;
  }[];
}

// Use normalized structure
interface GoodState {
  posts: {
    byId: Record<string, Post>;
    allIds: string[];
  };
  comments: {
    byId: Record<string, Comment>;
    allIds: string[];
  };
}
```

### Normalization with createEntityAdapter

```typescript
// src/store/slices/{{RESOURCE}}.ts
import { createEntityAdapter, createSlice } from '@reduxjs/toolkit';

const {{RESOURCE}}Adapter = createEntityAdapter<{{TYPE}}>({
  selectId: ({{RESOURCE}}) => {{RESOURCE}}.id,
  sortComparer: (a, b) => a.{{FIELD}}.localeCompare(b.{{FIELD}}),
});

const {{RESOURCE}}Slice = createSlice({
  name: '{{RESOURCE}}',
  initialState: {{RESOURCE}}Adapter.getInitialState(),
  reducers: {
    {{RESOURCE}}Added: {{RESOURCE}}Adapter.addOne,
    {{RESOURCE}}Updated: {{RESOURCE}}Adapter.updateOne,
    {{RESOURCE}}Removed: {{RESOURCE}}Adapter.removeOne,
    {{RESOURCE}}sReceived: {{RESOURCE}}Adapter.setAll,
  },
});

// Export selectors
export const {
  selectAll: selectAll{{RESOURCE}}s,
  selectById: select{{RESOURCE}}ById,
  selectIds: select{{RESOURCE}}Ids,
} = {{RESOURCE}}Adapter.getSelectors((state: RootState) => state.{{RESOURCE}});
```

## State Synchronization

### Sync with URL

```typescript
// src/hooks/useSyncWithUrl.ts
import { useEffect } from 'react';
import { useSearchParams } from 'react-router-dom';
import { useAppDispatch, useAppSelector } from '@/store/hooks';

export function useSyncWithUrl() {
  const [searchParams, setSearchParams] = useSearchParams();
  const dispatch = useAppDispatch();
  const {{STATE}} = useAppSelector({{SELECTOR}});

  // Read from URL on mount
  useEffect(() => {
    const {{PARAM}} = searchParams.get('{{PARAM_NAME}}');
    if ({{PARAM}}) {
      dispatch({{ACTION}}({{PARAM}}));
    }
  }, []);

  // Write to URL when state changes
  useEffect(() => {
    if ({{STATE}}) {
      setSearchParams({ {{PARAM_NAME}}: {{STATE}} });
    }
  }, [{{STATE}}]);
}
```

### Sync with localStorage

```typescript
// src/hooks/useSyncWithLocalStorage.ts
import { useEffect } from 'react';

export function useSyncWithLocalStorage<T>(
  key: string,
  value: T,
  setter: (value: T) => void
) {
  // Load from localStorage on mount
  useEffect(() => {
    const stored = localStorage.getItem(key);
    if (stored) {
      try {
        setter(JSON.parse(stored));
      } catch (error) {
        console.error('Failed to parse stored value:', error);
      }
    }
  }, []);

  // Save to localStorage when value changes
  useEffect(() => {
    localStorage.setItem(key, JSON.stringify(value));
  }, [value, key]);
}
```

## Performance Optimization

### Prevent Unnecessary Re-renders

```typescript
// Use selectors wisely
// ❌ Bad: Selecting entire slice
const slice = useAppSelector(state => state.{{SLICE}});

// ✅ Good: Selecting specific fields
const {{FIELD}} = useAppSelector(state => state.{{SLICE}}.{{FIELD}});

// ✅ Better: Memoized selector
const {{DERIVED}} = useAppSelector(select{{DERIVED}});
```

### Shallow Equality

```typescript
// Use shallowEqual for object/array selectors
import { shallowEqual } from 'react-redux';

const {{ARRAY}} = useAppSelector(
  state => state.{{SLICE}}.{{ARRAY}},
  shallowEqual
);
```

### Split Large Slices

```typescript
// Instead of one large slice
interface AppState {
  user: UserState;
  posts: PostsState;
  comments: CommentsState;
  ui: UIState;
}

// Split into focused slices
const store = configureStore({
  reducer: {
    user: userReducer,
    posts: postsReducer,
    comments: commentsReducer,
    ui: uiReducer,
  },
});
```

## Testing

### Testing Reducers

```typescript
// src/store/slices/__tests__/{{SLICE}}.test.ts
import reducer, { {{ACTION}} } from '../{{SLICE}}';

describe('{{SLICE}} reducer', () => {
  it('should handle {{ACTION}}', () => {
    const previousState = {{INITIAL_STATE}};
    const action = {{ACTION}}({{PAYLOAD}});
    const newState = reducer(previousState, action);

    expect(newState.{{FIELD}}).toEqual({{EXPECTED}});
  });
});
```

### Testing with Mock Store

```typescript
// src/__tests__/utils/test-utils.tsx
import { configureStore } from '@reduxjs/toolkit';
import { Provider } from 'react-redux';
import { render } from '@testing-library/react';

export function renderWithProviders(
  ui: React.ReactElement,
  {
    preloadedState = {},
    store = configureStore({
      reducer: {{REDUCERS}},
      preloadedState,
    }),
    ...renderOptions
  } = {}
) {
  function Wrapper({ children }: { children: React.ReactNode }) {
    return <Provider store={store}>{children}</Provider>;
  }

  return { store, ...render(ui, { wrapper: Wrapper, ...renderOptions }) };
}
```

## Best Practices

### 1. State Organization
- Keep state flat and normalized
- Avoid deeply nested structures
- Use entity adapters for collections

### 2. Action Naming
- Use past tense: `userLoggedIn`, not `loginUser`
- Be specific: `postCreated`, not `create`
- Include entity: `user/loggedIn`, not just `loggedIn`

### 3. Selector Usage
- Always use selectors, never access state directly
- Memoize expensive computations
- Keep selectors pure (no side effects)

### 4. Performance
- Split state into logical slices
- Use shallow equality for arrays/objects
- Avoid selecting entire state objects

### 5. Type Safety
- Type all state interfaces
- Type all action payloads
- Use typed hooks (useAppDispatch, useAppSelector)

## Migration Guide

### Context to Zustand

```typescript
// Before (Context)
const { user, setUser } = useUser();

// After (Zustand)
const { user, setUser } = useUserStore();
```

### Redux to Redux Toolkit

```typescript
// Before (Redux)
const SET_USER = 'SET_USER';
function userReducer(state = {}, action) {
  switch (action.type) {
    case SET_USER:
      return { ...state, user: action.payload };
    default:
      return state;
  }
}

// After (Redux Toolkit)
const userSlice = createSlice({
  name: 'user',
  initialState: {},
  reducers: {
    setUser: (state, action) => {
      state.user = action.payload;
    },
  },
});
```

## Resources

- [{{STATE_LIBRARY}} Documentation]({{DOCS_URL}})
- [React Query for Server State](https://tanstack.com/query)
- [State Management Best Practices]({{BEST_PRACTICES_URL}})

---

**Maintainer**: {{MAINTAINER_NAME}}
**Last Updated**: {{CURRENT_DATE}}
**State Library**: {{STATE_LIBRARY}}
