#!/usr/bin/env bash
# Script: validate-template-vars.sh
# Purpose: Check template placeholders and rendering (mechanical only)
# Subsystem: validation
# Called by: /validation:validate command
# Outputs: JSON validation report

set -euo pipefail

# --- Configuration ---
TEMPLATE_FILE="${1:-}"
OUTPUT_FILE="${2:-/tmp/template-validation.json}"

# --- Usage ---
if [ -z "$TEMPLATE_FILE" ]; then
  echo "Usage: $0 <template-file> [output-file]"
  echo "Example: $0 multiagent_core/templates/.multiagent/testing/templates/unit_template.test.js"
  exit 1
fi

# --- Validation Checks ---

echo "[INFO] Validating template: $TEMPLATE_FILE"

if [ ! -f "$TEMPLATE_FILE" ]; then
  cat > "$OUTPUT_FILE" <<EOF
{
  "status": "failed",
  "template": "$TEMPLATE_FILE",
  "error": "File does not exist",
  "checks": {}
}
EOF
  exit 1
fi

ISSUES=()
WARNINGS=()
CHECKS_PASSED=0
CHECKS_TOTAL=0

# Check 1: PURPOSE comment header
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
if head -10 "$TEMPLATE_FILE" | grep -qiE "^[#/\*]+ Purpose:|^Purpose:" ; then
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
else
  WARNINGS+=("Missing PURPOSE comment header")
fi

# Check 2: Find all placeholders {{VARIABLE}}
PLACEHOLDERS=$(grep -oE '\{\{[A-Z_]+\}\}' "$TEMPLATE_FILE" | sort -u || true)
PLACEHOLDER_COUNT=$(echo "$PLACEHOLDERS" | grep -c . || true)

# Check 3: Variables documented in header
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
if head -10 "$TEMPLATE_FILE" | grep -qiE "Variables?:|Placeholders?:" ; then
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
else
  if [ "$PLACEHOLDER_COUNT" -gt 0 ]; then
    WARNINGS+=("Found $PLACEHOLDER_COUNT placeholders but no 'Variables:' header documenting them")
  else
    CHECKS_PASSED=$((CHECKS_PASSED + 1))
  fi
fi

# Check 4: Agent attribution
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
if head -10 "$TEMPLATE_FILE" | grep -qiE "Generated by:|Created by:" ; then
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
else
  WARNINGS+=("Missing agent attribution (Generated by:)")
fi

# Check 5: Template location documented
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
if head -10 "$TEMPLATE_FILE" | grep -qE "Template location:|Location:" ; then
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
else
  WARNINGS+=("Missing template location documentation")
fi

# Check 6: No unfilled placeholders (check for common mistakes)
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
# Look for unfilled variables like: {{UNFILLED}}, {{TODO}}, {{FIXME}}
if grep -qE '\{\{(UNFILLED|TODO|FIXME|TBD|PLACEHOLDER)\}\}' "$TEMPLATE_FILE"; then
  ISSUES+=("Contains unfilled placeholder markers ({{UNFILLED}}, {{TODO}}, etc.)")
else
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
fi

# Check 7: Valid YAML frontmatter (if applicable)
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
if head -3 "$TEMPLATE_FILE" | grep -q "^---$"; then
  # Has frontmatter, check if it's valid
  FRONTMATTER_END=$(awk '/^---$/{if(++count==2){print NR; exit}}' "$TEMPLATE_FILE")
  if [ -n "$FRONTMATTER_END" ]; then
    CHECKS_PASSED=$((CHECKS_PASSED + 1))
  else
    ISSUES+=("YAML frontmatter appears incomplete (missing closing ---)")
  fi
else
  # No frontmatter expected
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
fi

# Check 8: Placeholder format consistency
CHECKS_TOTAL=$((CHECKS_TOTAL + 1))
# Check for inconsistent formats: {VARIABLE}, ${VARIABLE}, [VARIABLE]
INCONSISTENT=$(grep -oE '\{[A-Z_]+\}|\$\{[A-Z_]+\}|\[[A-Z_]+\]' "$TEMPLATE_FILE" | grep -vE '\{\{[A-Z_]+\}\}' || true)
if [ -n "$INCONSISTENT" ]; then
  WARNINGS+=("Found non-standard placeholder formats (use {{VARIABLE}} format)")
else
  CHECKS_PASSED=$((CHECKS_PASSED + 1))
fi

# --- Calculate Score ---
if [ "$CHECKS_TOTAL" -gt 0 ]; then
  SCORE=$(echo "scale=1; ($CHECKS_PASSED / $CHECKS_TOTAL) * 100" | bc)
else
  SCORE=0
fi

# Determine status
if [ "${#ISSUES[@]}" -eq 0 ]; then
  STATUS="passed"
elif [ "$CHECKS_PASSED" -ge $((CHECKS_TOTAL * 70 / 100)) ]; then
  STATUS="warning"
else
  STATUS="failed"
fi

# --- Generate Output ---
# Convert placeholders to JSON array
PLACEHOLDERS_JSON=$(echo "$PLACEHOLDERS" | jq -R . | jq -s . 2>/dev/null || echo "[]")

cat > "$OUTPUT_FILE" <<EOF
{
  "status": "$STATUS",
  "template": "$TEMPLATE_FILE",
  "score": $SCORE,
  "checks_passed": $CHECKS_PASSED,
  "checks_total": $CHECKS_TOTAL,
  "placeholder_count": $PLACEHOLDER_COUNT,
  "placeholders": $PLACEHOLDERS_JSON,
  "issues": [
$(printf '    "%s"' "${ISSUES[@]}" | paste -sd ',' -)
  ],
  "warnings": [
$(printf '    "%s"' "${WARNINGS[@]}" | paste -sd ',' -)
  ]
}
EOF

echo "✅ Validation complete: $STATUS (${CHECKS_PASSED}/${CHECKS_TOTAL} checks passed)"
echo "📊 Found $PLACEHOLDER_COUNT placeholders"
echo "📄 Report: $OUTPUT_FILE"

if [ "$STATUS" = "failed" ]; then
  exit 1
else
  exit 0
fi
