"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    static getBoolean(value) {
        if (typeof value === 'string') {
            // CloudFormation allows passing strings as boolean
            switch (value) {
                case 'true': return true;
                case 'false': return false;
                default: throw new Error(`Expected 'true' or 'false' for boolean value, got: '${value}'`);
            }
        }
        // in all other cases, just return the value,
        // and let a validator handle if it's not a boolean
        return value;
    }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        if (typeof value === 'string') {
            const parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.version = this.parseValue(resourceAttributes.Version);
        cfnOptions.description = this.parseValue(resourceAttributes.Description);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: FromCloudFormation.getBoolean(policy.EnableVersionUpgrade),
            useOnlineResharding: FromCloudFormation.getBoolean(policy.UseOnlineResharding),
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: FromCloudFormation.getBoolean(p.WaitOnResourceSignals),
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                const value = object[key];
                let logicalId, attributeName, stringForm;
                // Fn::GetAtt takes as arguments either a string...
                if (typeof value === 'string') {
                    // ...in which case the logical ID and the attribute name are separated with '.'
                    const dotIndex = value.indexOf('.');
                    if (dotIndex === -1) {
                        throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${value}'`);
                    }
                    logicalId = value.substr(0, dotIndex);
                    attributeName = value.substr(dotIndex + 1); // the +1 is to skip the actual '.'
                    stringForm = true;
                }
                else {
                    // ...or a 2-element list
                    logicalId = value[0];
                    attributeName = value[1];
                    stringForm = false;
                }
                const target = this.finder.findResource(logicalId);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${logicalId}' not found`);
                }
                return cfn_reference_1.CfnReference.for(target, attributeName, stringForm ? cfn_reference_1.ReferenceRendering.GET_ATT_STRING : undefined);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef !== undefined) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', cfn_reference_1.ReferenceRendering.FN_SUB).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, cfn_reference_1.ReferenceRendering.FN_SUB).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
//# sourceMappingURL=data:application/json;base64,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