"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const lib_1 = require("../lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const defaults = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-kinesisstreams-lambda';
const lambdaRole = new iam.Role(stack, 'test-role', {
    assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    inlinePolicies: {
        LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
            statements: [new iam.PolicyStatement({
                    actions: [
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents'
                    ],
                    resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                })]
        })
    }
});
(0, core_1.addCfnGuardSuppressRules)(lambdaRole, ["IAM_NO_INLINE_POLICY_CHECK"]);
const lambdaFn = new lambda.Function(stack, 'test-fn', {
    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
    handler: 'index.handler',
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    role: lambdaRole,
});
const cfnLambdafunction = lambdaFn.node.findChild('Resource');
cfnLambdafunction.cfnOptions.metadata = {
    cfn_nag: {
        rules_to_suppress: [{
                id: 'W89',
                reason: `This is not a rule for the general case, just for specific use cases/industries`
            },
            {
                id: 'W92',
                reason: `Impossible for us to define the correct concurrency for clients`
            }]
    }
};
const stream = new kinesis.Stream(stack, 'test-stream', {
    shardCount: 2,
    encryption: kinesis.StreamEncryption.MANAGED
});
// Definitions
const props = {
    existingStreamObj: stream,
    existingLambdaObj: lambdaFn,
    kinesisEventSourceProps: {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 1
    },
};
new lib_1.KinesisStreamsToLambda(stack, 'test-ks-lambda', props);
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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