import os
import logging
from typing import Dict, Any, Optional
from dotenv import load_dotenv

from .agent import BlaskAPIAgent
from .planner import PlannerTool
from .controller import ControllerTool
from .prompts import enhancement_prompt


logging.basicConfig(
    level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


load_dotenv()


class BlaskAPIAgentManager:
    """Manager class for integrating API agents with the RAG system."""

    def __init__(self, username: Optional[str] = None, password: Optional[str] = None):
        """Initialize the BlaskAPIAgentManager.

        Args:
            username: Username for API authentication
            password: Password for API authentication
        """
        self.username = username or os.getenv("BLASK_USERNAME")
        self.password = password or os.getenv("BLASK_PASSWORD")

        self.api_agent = BlaskAPIAgent(username=self.username, password=self.password)
        self.planner = PlannerTool(api_agent=self.api_agent)
        self.controller = ControllerTool(api_agent=self.api_agent)

    def process_query(self, query: str) -> Dict[str, Any]:
        """Process a query through the API agents.

        Args:
            query: The user query to process

        Returns:
            Dict[str, Any]: Results of processing the query
        """
        if not self.api_agent.authenticate():
            return {
                "error": "Authentication failed",
                "data": None,
                "summary": "Could not authenticate with Blask API",
            }

        self.api_agent.load_swagger_spec()

        actions, explanation = self.planner.get_api_plan(query)

        if not actions:
            return {
                "error": None,
                "data": None,
                "summary": "No relevant API endpoints found for this query",
            }

        results = self.controller.execute_api_actions(query, actions)

        enriched_results = {
            "original_query": query,
            "api_plan": {"actions": actions, "explanation": explanation},
            "results": results,
        }

        summary = self.controller.synthesize_results(
            query, enriched_results, explanation
        )

        return {
            "error": None,
            "data": enriched_results,
            "summary": summary,
            "plan": {"actions": actions, "explanation": explanation},
        }

    def enhance_rag_answer(self, query: str, rag_answer: str) -> str:
        """Enhance the RAG answer with additional information from the API.

        Args:
            query: The user query
            rag_answer: The answer generated by the RAG system

        Returns:
            str: Enhanced answer
        """
        api_results = self.process_query(query)

        if api_results.get("error") or not api_results.get("summary"):
            return rag_answer

        formatted_prompt = enhancement_prompt.format(
            query=query, rag_answer=rag_answer, api_results=api_results
        )
        enhanced_answer = self.api_agent.llm.invoke(formatted_prompt).content

        return enhanced_answer

    def get_api_data(self, query: str) -> str:
        """Process a query through the API data pipeline.

        This function:
        1. Uses the existing authenticated API tools
        2. Plans the API calls needed to answer the query
        3. Executes the planned API calls
        4. Synthesizes the results into a coherent answer

        Args:
            query: The user's question or query

        Returns:
            str: A synthesized answer based on the API data
        """
        if not self.api_agent.authenticate():
            return "Could not authenticate with Blask API"

        self.api_agent.load_swagger_spec()

        actions, explanation = self.planner.get_api_plan(query)

        if not actions:
            logger.info("No API actions planned for this query")
            return "No relevant API data found for this query."

        results = self.controller.execute_api_actions(query, actions)
        logger.info(f"Executed {len(results)} API calls")

        summary = self.controller.synthesize_results(query, results, explanation)

        return summary
