"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static bucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static inline(code) {
        return new InlineCode(code);
    }
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static asset(path) {
        return new AssetCode(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static cfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to
     * bind to the stack, add resources and have fun.
     */
    bind(_construct) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            s3Bucket: this.bucketName,
            s3Key: this.key,
            s3ObjectVersion: this.objectVersion
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length > 4096) {
            throw new Error("Lambda source is too large, must be <= 4096 but is " + code.length);
        }
    }
    bind(construct) {
        const runtime = construct.runtime;
        if (!runtime.supportsInlineCode) {
            throw new Error(`Inline source not allowed for ${runtime && runtime.name}`);
        }
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            zipFile: this.code
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path) {
        super();
        this.path = path;
        this.isInline = false;
    }
    bind(construct) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(construct, 'Code', { path: this.path });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
    }
    /**
     * @internal
     */
    _toJSON(resource) {
        if (resource) {
            // https://github.com/awslabs/aws-cdk/issues/1432
            this.asset.addResourceMetadata(resource, 'Code');
        }
        return {
            s3Bucket: this.asset.s3BucketName,
            s3Key: this.asset.s3ObjectKey
        };
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(construct) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(construct, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(construct, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    /** @internal */
    _toJSON(_) {
        return {
            s3Bucket: this._bucketNameParam.valueAsString,
            s3Key: this._objectKeyParam.valueAsString,
        };
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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