"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const path = require("path");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.asset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NODEJS_8_10,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        provider.addToRolePolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                RetentionInDays: props.retention === Infinity ? undefined : props.retention
            }
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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