"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'version and aliases'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('1');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.beASupersetOfTemplate({
            MyLambdaVersion16CDE3C40: {
                Type: "AWS::Lambda::Version",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" }
                }
            },
            Alias325C5727: {
                Type: "AWS::Lambda::Alias",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" },
                    FunctionVersion: stack.resolve(version.version),
                    Name: "prod"
                }
            }
        }));
        test.done();
    },
    'can create an alias to $LATEST'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'latest',
            version: fn.latestVersion,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
            FunctionVersion: '$LATEST',
            Name: 'latest',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Version'));
        test.done();
    },
    'can use newVersion to create a new Version'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('NewVersion');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Version', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
            Name: "prod"
        }));
        test.done();
    },
    'can add additional versions to alias'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version1 = fn.addVersion('1');
        const version2 = fn.addVersion('2');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version: version1,
            additionalVersions: [{ version: version2, weight: 0.1 }]
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionVersion: stack.resolve(version1.version),
            RoutingConfig: {
                AdditionalVersionWeights: [
                    {
                        FunctionVersion: stack.resolve(version2.version),
                        FunctionWeight: 0.1
                    }
                ]
            }
        }));
        test.done();
    },
    'sanity checks on version weights'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('1');
        // WHEN: Individual weight too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias1', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 5 }]
            });
        });
        // WHEN: Sum too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias2', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 0.5 }, { version, weight: 0.6 }]
            });
        });
        test.done();
    },
    'metric adds Resource: aliasArn to dimensions'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        new cloudwatch.Alarm(stack, 'Alarm', {
            metric: alias.metric('Test'),
            alarmName: 'Test',
            threshold: 1,
            evaluationPeriods: 1
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            Dimensions: [{
                    Name: "FunctionName",
                    Value: {
                        Ref: "MyLambdaCCE802FB"
                    }
                }, {
                    Name: "Resource",
                    Value: {
                        'Fn::Join': [
                            '',
                            [
                                {
                                    "Fn::GetAtt": [
                                        "MyLambdaCCE802FB",
                                        "Arn"
                                    ]
                                },
                                ':prod'
                            ]
                        ]
                    }
                }]
        }));
        test.done();
    },
    'alias exposes real Lambdas role'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // THEN
        test.equals(alias.role, fn.role);
        test.done();
    },
    'functionName is derived from the aliasArn so that dependencies are sound'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_8_10,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        test.deepEqual(stack.resolve(alias.functionName), {
            "Fn::Join": [
                "",
                [
                    {
                        "Fn::Select": [
                            6,
                            {
                                "Fn::Split": [
                                    ":",
                                    {
                                        Ref: "Alias325C5727"
                                    }
                                ]
                            }
                        ]
                    },
                    ":prod"
                ]
            ]
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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