"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const path = require("path");
const lambda = require("../lib");
function defineFunction(code, runtime = lambda.Runtime.NODEJS_8_10) {
    const stack = new cdk.Stack();
    return new lambda.Function(stack, 'Func', {
        handler: 'foom',
        code, runtime
    });
}
function generateRandomString(bytes) {
    let s = '';
    for (let i = 0; i < bytes; ++i) {
        s += String.fromCharCode(Math.round(Math.random() * 256));
    }
    return s;
}
module.exports = {
    'lambda.Code.inline': {
        'fails if used with unsupported runtimes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.GO_1_X), /Inline source not allowed for go1\.x/);
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.JAVA_8), /Inline source not allowed for java8/);
            test.done();
        },
        'fails if larger than 4096 bytes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline(generateRandomString(4097)), lambda.Runtime.NODEJS_8_10), /Lambda source is too large, must be <= 4096 but is 4097/);
            test.done();
        }
    },
    'lambda.Code.asset': {
        'fails if a non-zip asset is used'(test) {
            // GIVEN
            const fileAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler', 'index.py'));
            // THEN
            test.throws(() => defineFunction(fileAsset), /Asset must be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetCode'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler'));
            // WHEN
            new lambda.Function(stack, 'Func1', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_8_10,
                code: directoryAsset
            });
            new lambda.Function(stack, 'Func2', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_8_10,
                code: directoryAsset
            });
            // THEN
            const assembly = app.synth();
            const synthesized = assembly.stacks[0];
            // Func1 has an asset, Func2 does not
            const metadata = synthesized.manifest.metadata || {};
            test.ok(metadata['/MyStack/Func1/Code']);
            test.deepEqual(metadata['/MyStack/Func1/Code'].length, 1);
            test.deepEqual(metadata['/MyStack/Func1/Code'][0].type, 'aws:cdk:asset');
            test.deepEqual(metadata['/MyStack/Func2/Code'], undefined);
            test.done();
        },
        'adds code asset metadata'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
            const location = path.join(__dirname, 'my-lambda-handler');
            // WHEN
            new lambda.Function(stack, 'Func1', {
                code: lambda.Code.asset(location),
                runtime: lambda.Runtime.NODEJS_8_10,
                handler: 'foom',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Metadata: {
                    [cxapi.ASSET_RESOURCE_METADATA_PATH_KEY]: 'asset.9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232',
                    [cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY]: 'Code'
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }
    },
    'lambda.Code.cfnParameters': {
        "automatically creates the Bucket and Key parameters when it's used in a Function"(test) {
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode();
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_8_10,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "FunctionLambdaSourceBucketNameParameter9E9E108F",
                    },
                    S3Key: {
                        Ref: "FunctionLambdaSourceObjectKeyParameter1C7AED11",
                    },
                },
            }));
            test.equal(stack.resolve(code.bucketNameParam), 'FunctionLambdaSourceBucketNameParameter9E9E108F');
            test.equal(stack.resolve(code.objectKeyParam), 'FunctionLambdaSourceObjectKeyParameter1C7AED11');
            test.done();
        },
        'does not allow accessing the Parameter properties before being used in a Function'(test) {
            const code = new lambda.CfnParametersCode();
            test.throws(() => {
                test.notEqual(code.bucketNameParam, undefined);
            }, /bucketNameParam/);
            test.throws(() => {
                test.notEqual(code.objectKeyParam, undefined);
            }, /objectKeyParam/);
            test.done();
        },
        'allows passing custom Parameters when creating it'(test) {
            const stack = new cdk.Stack();
            const bucketNameParam = new cdk.CfnParameter(stack, 'BucketNameParam', {
                type: 'String',
            });
            const bucketKeyParam = new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                type: 'String',
            });
            const code = lambda.Code.cfnParameters({
                bucketNameParam,
                objectKeyParam: bucketKeyParam,
            });
            test.equal(stack.resolve(code.bucketNameParam), 'BucketNameParam');
            test.equal(stack.resolve(code.objectKeyParam), 'ObjectKeyParam');
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_8_10,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: "BucketNameParam",
                    },
                    S3Key: {
                        Ref: "ObjectKeyParam",
                    },
                },
            }));
            test.done();
        },
        'can assign parameters'(test) {
            // given
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode({
                bucketNameParam: new cdk.CfnParameter(stack, 'BucketNameParam', {
                    type: 'String',
                }),
                objectKeyParam: new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                    type: 'String',
                }),
            });
            // when
            const overrides = stack.resolve(code.assign({
                bucketName: 'SomeBucketName',
                objectKey: 'SomeObjectKey',
            }));
            // then
            test.equal(overrides['BucketNameParam'], 'SomeBucketName');
            test.equal(overrides['ObjectKeyParam'], 'SomeObjectKey');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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