"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'creating a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // WHEN
        new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Content: {
                S3Bucket: stack.resolve(bucket.bucketName),
                S3Key: 'ObjectKey',
            },
            CompatibleRuntimes: ['nodejs8.10']
        }));
        test.done();
    },
    'granting access to a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const layer = new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10]
        });
        // WHEN
        layer.addPermission('GrantUsage-123456789012', { accountId: '123456789012' });
        layer.addPermission('GrantUsage-o-123456', { accountId: '*', organizationId: 'o-123456' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '123456789012',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '*',
            OrganizationId: 'o-123456'
        }));
        test.done();
    },
    'creating a layer with no runtimes compatible'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // THEN
        test.throws(() => new lambda.LayerVersion(stack, 'LayerVersion', { code, compatibleRuntimes: [] }), /supports no runtime/);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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