"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const os_1 = require("os");
const s3 = require("../lib");
const lib_1 = require("../lib");
module.exports = {
    'default bucket'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'CFN properties are type-validated during resolution'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            bucketName: cdk.Token.asString(5) // Oh no
        });
        test.throws(() => {
            assert_1.SynthUtils.synthesize(stack);
        }, /bucketName: 5 should be a string/);
        test.done();
    },
    'bucket without encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.UNENCRYPTED
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with managed encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'valid bucket names'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'abc.xyz-34ab',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '124.pp--33',
        }));
        test.done();
    },
    'bucket validation skips tokenized values'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: cdk.Lazy.stringValue({ produce: () => '_BUCKET' }),
        }));
        test.done();
    },
    'fails with message on invalid bucket names'(test) {
        const stack = new cdk.Stack();
        const bucket = `-buckEt.-${new Array(65).join('$')}`;
        const expectedErrors = [
            `Invalid S3 bucket name (value: ${bucket})`,
            'Bucket name must be at least 3 and no more than 63 characters',
            'Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) (offset: 5)',
            'Bucket name must start and end with a lowercase character or number (offset: 0)',
            `Bucket name must start and end with a lowercase character or number (offset: ${bucket.length - 1})`,
            'Bucket name must not have dash next to period, or period next to dash, or consecutive periods (offset: 7)',
        ].join(os_1.EOL);
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: bucket,
        }), function (err) {
            return expectedErrors === err.message;
        });
        test.done();
    },
    'fails if bucket name has less than 3 or more than 63 characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'a',
        }), /at least 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: new Array(65).join('x'),
        }), /no more than 63/);
        test.done();
    },
    'fails if bucket name has invalid characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'b@cket',
        }), /offset: 1/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucKet',
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'bučket',
        }), /offset: 2/);
        test.done();
    },
    'fails if bucket name does not start or end with lowercase character or number'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '-ucket',
        }), /offset: 0/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucke.',
        }), /offset: 5/);
        test.done();
    },
    'fails only if bucket name has the consecutive symbols (..), (.-), (-.)'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'buc..ket',
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'buck.-et',
        }), /offset: 4/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'b-.ucket',
        }), /offset: 1/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: 'bu--cket',
        }));
        test.done();
    },
    'fails only if bucket name resembles IP address'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '1.2.3.4',
        }), /must not resemble an IP address/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '1.2.3',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: '1.2.3.a',
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: '1000.2.3.4',
        }));
        test.done();
    },
    'fails if encryption key is used with managed encryption'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'fails if encryption key is used with encryption set to unencrypted'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.UNENCRYPTED,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'encryptionKey can specify kms key'(test) {
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'MyKey', { description: 'hello, world' });
        new s3.Bucket(stack, 'MyBucket', { encryptionKey, encryption: s3.BucketEncryption.KMS });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyKey6AB29FA6": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "hello, world",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": [
                                                "MyKey6AB29FA6",
                                                "Arn"
                                            ]
                                        },
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with versioning turned on'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            versioned: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "VersioningConfiguration": {
                            "Status": "Enabled"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAll'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "BlockPublicPolicy": true,
                            "IgnorePublicAcls": true,
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAcls'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ACLS,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "IgnorePublicAcls": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with custom block public access setting'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: new s3.BlockPublicAccess({ restrictPublicBuckets: true })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with custom canned access control'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            accessControl: s3.BucketAccessControl.LOG_DELIVERY_WRITE
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "AccessControl": "LogDeliveryWrite"
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'permissions': {
        'addPermission creates a bucket policy'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            bucket.addToResourcePolicy(new iam.PolicyStatement({ resources: ['foo'], actions: ['bar:baz'] }));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "UpdateReplacePolicy": "Retain",
                    },
                    "MyBucketPolicyE7FBAC7B": {
                        "Type": "AWS::S3::BucketPolicy",
                        "Properties": {
                            "Bucket": {
                                "Ref": "MyBucketF68F3FF0"
                            },
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "bar:baz",
                                        "Effect": "Allow",
                                        "Resource": "foo"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                    }
                }
            });
            test.done();
        },
        'forBucket returns a permission statement associated with the bucket\'s ARN'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const x = new iam.PolicyStatement({ resources: [bucket.bucketArn], actions: ['s3:ListBucket'] });
            test.deepEqual(stack.resolve(x.toStatementJson()), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }
            });
            test.done();
        },
        'arnForObjects returns a permission statement associated with objects in the bucket'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const p = new iam.PolicyStatement({ resources: [bucket.arnForObjects('hello/world')], actions: ['s3:GetObject'] });
            test.deepEqual(stack.resolve(p.toStatementJson()), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/hello/world']
                    ]
                }
            });
            test.done();
        },
        'arnForObjects accepts multiple arguments and FnConcats them'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.UNENCRYPTED });
            const user = new iam.User(stack, 'MyUser');
            const team = new iam.Group(stack, 'MyTeam');
            const resource = bucket.arnForObjects(`home/${team.groupName}/${user.userName}/*`);
            const p = new iam.PolicyStatement({ resources: [resource], actions: ['s3:GetObject'] });
            test.deepEqual(stack.resolve(p.toStatementJson()), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [
                            { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                            '/home/',
                            { Ref: 'MyTeam01DD6685' },
                            '/',
                            { Ref: 'MyUserDC45028B' },
                            '/*'
                        ]
                    ]
                }
            });
            test.done();
        }
    },
    'removal policy can be used to specify behavior upon delete'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            encryption: s3.BucketEncryption.UNENCRYPTED
        });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyBucketF68F3FF0: {
                    Type: 'AWS::S3::Bucket',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain'
                }
            }
        });
        test.done();
    },
    'import/export': {
        'static import(ref) allows importing an external/existing bucket'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn });
            // this is a no-op since the bucket is external
            bucket.addToResourcePolicy(new iam.PolicyStatement({ resources: ['foo'], actions: ['bar:baz'] }));
            const p = new iam.PolicyStatement({ resources: [bucket.bucketArn], actions: ['s3:ListBucket'] });
            // it is possible to obtain a permission statement for a ref
            test.deepEqual(p.toStatementJson(), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: 'arn:aws:s3:::my-bucket'
            });
            test.deepEqual(bucket.bucketArn, bucketArn);
            test.deepEqual(stack.resolve(bucket.bucketName), 'my-bucket');
            test.deepEqual(assert_1.SynthUtils.synthesize(stack).template, {}, 'the ref is not a real resource');
            test.done();
        },
        'import can also be used to import arbitrary ARNs'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement({ resources: ['*'], actions: ['*'] }));
            // at this point we technically didn't create any resources in the consuming stack.
            assert_1.expect(stack).toMatch({});
            // but now we can reference the bucket
            // you can even use the bucket name, which will be extracted from the arn provided.
            const user = new iam.User(stack, 'MyUser');
            user.addToPolicy(new iam.PolicyStatement({
                resources: [bucket.arnForObjects(`my/folder/${bucket.bucketName}`)],
                actions: ['s3:*']
            }));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "s3:*",
                                        "Effect": "Allow",
                                        "Resource": "arn:aws:s3:::my-bucket/my/folder/my-bucket"
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        },
                    }
                }
            });
            test.done();
        },
    },
    'grantRead'(test) {
        const stack = new cdk.Stack();
        const reader = new iam.User(stack, 'Reader');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.grantRead(reader);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ReaderF7BF189D": {
                    "Type": "AWS::IAM::User"
                },
                "ReaderDefaultPolicy151F3818": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::GetAtt": [
                                                "MyBucketF68F3FF0",
                                                "Arn"
                                            ]
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::GetAtt": [
                                                            "MyBucketF68F3FF0",
                                                            "Arn"
                                                        ]
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "ReaderDefaultPolicy151F3818",
                        "Users": [
                            {
                                "Ref": "ReaderF7BF189D"
                            }
                        ]
                    }
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
            }
        });
        test.done();
    },
    'grantReadWrite': {
        'can be used to grant reciprocal permissions to an identity'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "UpdateReplacePolicy": "Retain",
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'grant permissions to non-identity principal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
            // WHEN
            bucket.grantRead(new iam.OrganizationPrincipal('o-1234'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                PolicyDocument: {
                    "Version": "2012-10-17",
                    "Statement": [
                        {
                            "Action": ["s3:GetObject*", "s3:GetBucket*", "s3:List*"],
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": [
                                { "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] },
                                { "Fn::Join": ["", [{ "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] }, "/*"]] }
                            ]
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                            "Action": ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*",
                                "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion", "kms:GenerateDataKey"],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": ["", [
                                            "arn:", { "Ref": "AWS::Partition" }, ":iam::", { "Ref": "AWS::AccountId" }, ":root"
                                        ]]
                                }
                            },
                            "Resource": "*"
                        },
                        {
                            "Action": ["kms:Decrypt", "kms:DescribeKey"],
                            "Effect": "Allow",
                            "Resource": "*",
                            "Principal": "*",
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                        }
                    ],
                    "Version": "2012-10-17"
                },
            }));
            test.done();
        },
        'if an encryption key is included, encrypt/decrypt permissions are also added both ways'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketKeyC17130CF": {
                        "Type": "AWS::KMS::Key",
                        "Properties": {
                            "Description": "Created by MyBucket",
                            "KeyPolicy": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kms:Create*",
                                            "kms:Describe*",
                                            "kms:Enable*",
                                            "kms:List*",
                                            "kms:Put*",
                                            "kms:Update*",
                                            "kms:Revoke*",
                                            "kms:Disable*",
                                            "kms:Get*",
                                            "kms:Delete*",
                                            "kms:ScheduleKeyDeletion",
                                            "kms:CancelKeyDeletion",
                                            "kms:GenerateDataKey"
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":iam::",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":root"
                                                    ]
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::GetAtt": [
                                                    "MyUserDC45028B",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                        "DeletionPolicy": "Retain",
                        "UpdateReplacePolicy": "Retain"
                    },
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "UpdateReplacePolicy": "Retain",
                        "Properties": {
                            "BucketEncryption": {
                                "ServerSideEncryptionConfiguration": [
                                    {
                                        "ServerSideEncryptionByDefault": {
                                            "KMSMasterKeyID": {
                                                "Fn::GetAtt": [
                                                    "MyBucketKeyC17130CF",
                                                    "Arn"
                                                ]
                                            },
                                            "SSEAlgorithm": "aws:kms"
                                        }
                                    }
                                ]
                            }
                        }
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyBucketKeyC17130CF",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
    },
    'more grants'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.KMS });
        const putter = new iam.User(stack, 'Putter');
        const writer = new iam.User(stack, 'Writer');
        const deleter = new iam.User(stack, 'Deleter');
        bucket.grantPut(putter);
        bucket.grantWrite(writer);
        bucket.grantDelete(deleter);
        const resources = assert_1.SynthUtils.synthesize(stack).template.Resources;
        const actions = (id) => resources[id].Properties.PolicyDocument.Statement[0].Action;
        test.deepEqual(actions('WriterDefaultPolicyDC585BCE'), ['s3:DeleteObject*', 's3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('PutterDefaultPolicyAB138DD3'), ['s3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('DeleterDefaultPolicyCD33B8A0'), 's3:DeleteObject*');
        test.done();
    },
    'cross-stack permissions': {
        'in the same account and region'(test) {
            const app = new cdk.App();
            const stackA = new cdk.Stack(app, 'stackA');
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket');
            const stackB = new cdk.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            bucketFromStackA.grantRead(user);
            assert_1.expect(stackA).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "UpdateReplacePolicy": "Retain"
                    }
                },
                "Outputs": {
                    "ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyBucketF68F3FF0",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                        }
                    }
                }
            });
            assert_1.expect(stackB).toMatch({
                "Resources": {
                    "UserWhoNeedsAccessF8959C3D": {
                        "Type": "AWS::IAM::User"
                    },
                    "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                            "Users": [
                                {
                                    "Ref": "UserWhoNeedsAccessF8959C3D"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'in different accounts'(test) {
            // given
            const stackA = new cdk.Stack(undefined, 'StackA', { env: { account: '123456789012' } });
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket', {
                bucketName: 'my-bucket-physical-name',
            });
            const stackB = new cdk.Stack(undefined, 'StackB', { env: { account: '234567890123' } });
            const roleFromStackB = new iam.Role(stackB, 'MyRole', {
                assumedBy: new iam.AccountPrincipal('234567890123'),
                roleName: 'MyRolePhysicalName',
            });
            // when
            bucketFromStackA.grantRead(roleFromStackB);
            // then
            assert_1.expect(stackA).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":iam::234567890123:role/MyRolePhysicalName",
                                        ],
                                    ],
                                },
                            },
                        },
                    ],
                },
            }));
            assert_1.expect(stackB).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::my-bucket-physical-name",
                                        ],
                                    ],
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":s3:::my-bucket-physical-name/*",
                                        ],
                                    ],
                                },
                            ],
                        },
                    ],
                },
            }));
            test.done();
        },
        'in different accounts, with a KMS Key'(test) {
            // given
            const stackA = new cdk.Stack(undefined, 'StackA', { env: { account: '123456789012' } });
            const key = new kms.Key(stackA, 'MyKey');
            const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket', {
                bucketName: 'my-bucket-physical-name',
                encryptionKey: key,
                encryption: s3.BucketEncryption.KMS,
            });
            const stackB = new cdk.Stack(undefined, 'StackB', { env: { account: '234567890123' } });
            const roleFromStackB = new iam.Role(stackB, 'MyRole', {
                assumedBy: new iam.AccountPrincipal('234567890123'),
                roleName: 'MyRolePhysicalName',
            });
            // when
            bucketFromStackA.grantRead(roleFromStackB);
            // then
            assert_1.expect(stackA).to(assert_1.haveResourceLike('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                        // grant to the root of the owning account
                        },
                        {
                            "Action": [
                                "kms:Decrypt",
                                "kms:DescribeKey",
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition",
                                            },
                                            ":iam::234567890123:role/MyRolePhysicalName",
                                        ],
                                    ],
                                },
                            },
                        },
                    ],
                },
            }));
            assert_1.expect(stackB).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                        // Bucket grant
                        },
                        {
                            "Action": [
                                "kms:Decrypt",
                                "kms:DescribeKey",
                            ],
                            "Effect": "Allow",
                            "Resource": "*",
                        },
                    ],
                },
            }));
            test.done();
        },
    },
    'urlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketURL', { value: bucket.urlForObject() });
        new cdk.CfnOutput(stack, 'MyFileURL', { value: bucket.urlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileURL', { value: bucket.urlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                }
            },
            "Outputs": {
                "BucketURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                }
                            ]
                        ]
                    },
                },
                "MyFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/my/file.txt"
                            ]
                        ]
                    }
                },
                "YourFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/your/file.txt"
                            ]
                        ]
                    },
                }
            }
        });
        test.done();
    },
    'grantPublicAccess': {
        'by default, grants s3:GetObject to all objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"keyPrefix" can be used to only grant access to certain objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('only/access/these/*');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/only/access/these/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"allowedActions" can be used to specify actions explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('*', 's3:GetObject', 's3:PutObject');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": ["s3:GetObject", "s3:PutObject"],
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'returns the PolicyStatement which can be then customized'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            const result = bucket.grantPublicAccess();
            result.resourceStatement.addCondition('IpAddress', { "aws:SourceIp": "54.240.143.0/24" });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] },
                            "Condition": {
                                "IpAddress": { "aws:SourceIp": "54.240.143.0/24" }
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'throws when blockPublicPolicy is set to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', {
                blockPublicAccess: new s3.BlockPublicAccess({ blockPublicPolicy: true })
            });
            // THEN
            test.throws(() => bucket.grantPublicAccess(), /blockPublicPolicy/);
            test.done();
        }
    },
    'website configuration': {
        'only index doc'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html'
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html"
                }
            }));
            test.done();
        },
        'fails if only error doc is specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteErrorDocument: 'error.html'
                });
            }, /"websiteIndexDocument" is required if "websiteErrorDocument" is set/);
            test.done();
        },
        'error and index docs'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
                websiteErrorDocument: 'error.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html",
                    ErrorDocument: "error.html"
                }
            }));
            test.done();
        },
        'exports the WebsiteURL'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html'
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] });
            test.done();
        },
        'exports the WebsiteDomain'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html'
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), {
                'Fn::Select': [
                    2,
                    {
                        'Fn::Split': ['/', { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] }]
                    }
                ]
            });
            test.done();
        },
        'exports the WebsiteURL for imported buckets'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketName(stack, 'Website', 'my-test-bucket');
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), {
                'Fn::Join': [
                    '',
                    [
                        'http://my-test-bucket.s3-website-',
                        { Ref: 'AWS::Region' },
                        '.',
                        { Ref: 'AWS::URLSuffix' }
                    ]
                ]
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), {
                'Fn::Join': [
                    '',
                    [
                        'my-test-bucket.s3-website-',
                        { Ref: 'AWS::Region' },
                        '.',
                        { Ref: 'AWS::URLSuffix' }
                    ]
                ]
            });
            test.done();
        },
        'exports the WebsiteURL for imported buckets with url'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'Website', {
                bucketName: 'my-test-bucket',
                bucketWebsiteUrl: 'http://my-test-bucket.my-test.suffix',
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), 'http://my-test-bucket.my-test.suffix');
            test.deepEqual(stack.resolve(bucket.bucketWebsiteDomainName), 'my-test-bucket.my-test.suffix');
            test.done();
        },
        'adds RedirectAllRequestsTo property'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteRedirect: {
                    hostName: 'www.example.com',
                    protocol: s3.RedirectProtocol.HTTPS
                }
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    RedirectAllRequestsTo: {
                        HostName: 'www.example.com',
                        Protocol: 'https'
                    }
                }
            }));
            test.done();
        },
        'fails if websiteRedirect and websiteIndex and websiteError are specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteIndexDocument: 'index.html',
                    websiteErrorDocument: 'error.html',
                    websiteRedirect: {
                        hostName: 'www.example.com'
                    }
                });
            }, /"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used/);
            test.done();
        },
        'fails if websiteRedirect and websiteRoutingRules are specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteRoutingRules: [],
                    websiteRedirect: {
                        hostName: 'www.example.com'
                    }
                });
            }, /"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used/);
            test.done();
        },
        'adds RedirectRules property'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteRoutingRules: [{
                        hostName: 'www.example.com',
                        httpRedirectCode: '302',
                        protocol: s3.RedirectProtocol.HTTPS,
                        replaceKey: s3.ReplaceKey.prefixWith('test/'),
                        condition: {
                            httpErrorCodeReturnedEquals: '200',
                            keyPrefixEquals: 'prefix',
                        }
                    }]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    RoutingRules: [{
                            RedirectRule: {
                                HostName: 'www.example.com',
                                HttpRedirectCode: '302',
                                Protocol: 'https',
                                ReplaceKeyPrefixWith: 'test/'
                            },
                            RoutingRuleCondition: {
                                HttpErrorCodeReturnedEquals: '200',
                                KeyPrefixEquals: 'prefix'
                            }
                        }]
                }
            }));
            test.done();
        },
        'fails if routingRule condition object is empty'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteRoutingRules: [{
                            httpRedirectCode: '303',
                            condition: {}
                        }]
                });
            }, /The condition property cannot be an empty object/);
            test.done();
        },
    },
    'Bucket.fromBucketArn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketArn(stack, 'my-bucket', 'arn:aws:s3:::my_corporate_bucket');
        // THEN
        test.deepEqual(bucket.bucketName, 'my_corporate_bucket');
        test.deepEqual(bucket.bucketArn, 'arn:aws:s3:::my_corporate_bucket');
        test.done();
    },
    'Bucket.fromBucketName'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketName(stack, 'imported-bucket', 'my-bucket-name');
        // THEN
        test.deepEqual(bucket.bucketName, 'my-bucket-name');
        test.deepEqual(stack.resolve(bucket.bucketArn), {
            'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket-name']]
        });
        test.done();
    },
    'if a kms key is specified, it implies bucket is encrypted with kms (dah)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const key = new kms.Key(stack, 'k');
        // THEN
        new lib_1.Bucket(stack, 'b', { encryptionKey: key });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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