"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const openapi_helper_1 = require("../lib/openapi-helper");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
const defaults = require("@aws-solutions-constructs/core");
const inlineJsonApiDefinition = require("./openapi/apiDefinition.json");
test('Simple deployment works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from Asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from S3 Bucket and Key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const apiDefinitionKey = 'api.yaml';
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionBucket,
        apiDefinitionKey,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from an apiDefinitionJson ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionJson: inlineJsonApiDefinition,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
const incorrectDefinitionMessage = 'Exactly one of apiDefinitionAsset, apiDefinitionJson or (apiDefinitionBucket/apiDefinitionKey) must be provided';
test('Throws error when both api definition asset and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Multiple Lambda Functions can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
});
test('Existing lambda function can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    functionName: 'NewLambdaFunction',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'ExistingLambdaFunction'
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'NewLambdaFunction'
    });
});
test('Throws error when neither existingLambdaObj or lambdaFunctionProps is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: [
                {
                    id: 'MessagesHandler'
                }
            ]
        });
    };
    expect(app).toThrowError('One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: MessagesHandler');
});
test('Two Constructs create APIs with different names', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const secondApiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'SecondOpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition-withCognitoAuth.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'second-test', {
        apiDefinitionAsset: secondApiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ApiGateway::RestApi", 2);
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "test-apigateway-lambda-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "second-test-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
});
test('Confirm API definition uri is added to function name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    const resources = JSON.parse(JSON.stringify(template)).Resources;
    expect(Object.keys(resources).find((element) => {
        return element.includes("MessagesHandler");
    })).toBeTruthy();
});
test('Confirm that providing both lambdaFunction and functionProps is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const props = {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                },
                existingLambdaObj
            }
        ]
    };
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test', props);
    };
    expect(app).toThrowError(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: MessagesHandler`);
});
/*
 * openapi-helper.ts tests
*/
test('Throws error when no api definition is specified', () => {
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when no api integration is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('At least one ApiIntegration must be specified in the apiIntegrations property');
});
test('Throws error when api definition s3 bucket is specified but s3 object key is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('apiDefinitionBucket and apiDefinitionKey must be specified together.');
});
test('Throws error when api definition s3 object key is specified but s3 bucket is missing', () => {
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('apiDefinitionBucket and apiDefinitionKey must be specified together.');
});
test('Throws error when api is defined as asset and s3 bucket is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when api is defined as asset and s3 key is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = (0, core_1.CreateScrapBucket)(stack, "scrapBucket");
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when both api definition inline and api definition asset are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionAsset,
            apiDefinitionJson: inlineJsonApiDefinition,
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('Throws error when both api definition inline and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        (0, openapi_helper_1.CheckOpenApiProps)({
            apiDefinitionJson: inlineJsonApiDefinition,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError(incorrectDefinitionMessage);
});
test('ObtainApiDefinition from local asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionAsset,
        tokenToFunctionMap: apiLambdaFunctions
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
});
test('ObtainApiDefinition from inline JSON spec', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiJsonDefinition: inlineJsonApiDefinition,
        tokenToFunctionMap: apiLambdaFunctions
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 0);
    expect(api).toBeDefined();
    expect(api.definition).toBeDefined();
    expect(api.definition.openapi).toEqual("3.0.1");
    expect(api.definition.info).toBeDefined();
    expect(api.definition.paths).toBeDefined();
});
test('ObtainApiDefinition from S3 bucket/key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionBucket: (0, core_1.CreateScrapBucket)(stack, "scrapBucket"),
        apiDefinitionKey: "api.yml",
        tokenToFunctionMap: apiLambdaFunctions
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
});
test('ObtainApiDefinition uses custom properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const testFunction = new lambda.Function(stack, 'test', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiLambdaFunctions = [
        {
            id: 'MessageHandler',
            lambdaFunction: testFunction
        }
    ];
    const api = (0, openapi_helper_1.ObtainApiDefinition)(stack, {
        apiDefinitionAsset,
        tokenToFunctionMap: apiLambdaFunctions,
        internalTransformTimeout: aws_cdk_lib_1.Duration.seconds(385),
        internalTransformMemorySize: 3456
    });
    expect(api).toBeDefined();
    expect(api.bucketName).toBeDefined();
    expect(api.key).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("Custom::TemplateWriter", 1);
    template.hasResource("AWS::Lambda::Function", {
        Properties: {
            Timeout: 385,
            MemorySize: 3456
        }
    });
});
test('Shared lambda function works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const constructTwo = new lib_1.OpenApiGatewayToLambda(stack, 'two-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj: construct.apiLambdaFunctions[0].lambdaFunction
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
    expect(construct.apiLambdaFunctions[0].lambdaFunction.functionArn).toEqual(constructTwo.apiLambdaFunctions[0].lambdaFunction.functionArn);
});
//# sourceMappingURL=data:application/json;base64,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