"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        let parsedValue;
        if (typeof value === 'string') {
            parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const finder = this.options.finder;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef) {
                    return specialRef;
                }
                else {
                    const refElement = this.options.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                // Fn::GetAtt takes a 2-element list as its argument
                const value = object[key];
                const target = this.options.finder.findResource(value[0]);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${value[0]}' not found`);
                }
                return target.getAtt(value[1]);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.options.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.options.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.options.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.options.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', true).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.options.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, true).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.options.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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