"use strict";
const fs = require("fs");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../../lib");
const evaluate_cfn_1 = require("../evaluate-cfn");
const CFN_CONTEXT = {
    'AWS::Region': 'the_region',
    'AWS::AccountId': 'the_account',
    'AWS::URLSuffix': 'domain.aws',
};
let app;
let stack;
/**
 * Evaluate a possibly string-containing value the same way CFN would do
 *
 * (Be invariant to the specific Fn::Sub or Fn::Join we would output)
 */
function evalCFN(value) {
    return evaluate_cfn_1.evaluateCFN(stack.resolve(value), CFN_CONTEXT);
}
function isAssetManifest(x) {
    return x instanceof cxapi.AssetManifestArtifact;
}
function readAssetManifest(asm) {
    const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
    if (!manifestArtifact) {
        throw new Error('no asset manifest in assembly');
    }
    return JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
}
module.exports = {
    'setUp'(cb) {
        app = new lib_1.App({
            context: {
                [cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: 'true',
            },
        });
        stack = new lib_1.Stack(app, 'Stack');
        cb();
    },
    'stack template is in asset manifest'(test) {
        var _a;
        // GIVEN
        new lib_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // WHEN
        const asm = app.synth();
        // THEN -- the S3 url is advertised on the stack artifact
        const stackArtifact = asm.getStackArtifact('Stack');
        const templateHash = '040a6374d4c48c0db867f1d4f95c69b12d28e69c3b8a9903a1db1ec651dcf480';
        test.equals(stackArtifact.stackTemplateAssetObjectUrl, `s3://cdk-hnb659fds-assets-\${AWS::AccountId}-\${AWS::Region}/${templateHash}`);
        // THEN - the template is in the asset manifest
        const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
        test.ok(manifestArtifact);
        const manifest = JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
        const firstFile = (_a = (manifest.files ? manifest.files[Object.keys(manifest.files)[0]] : undefined)) !== null && _a !== void 0 ? _a : {};
        test.deepEqual(firstFile, {
            source: { path: 'Stack.template.json', packaging: 'file' },
            destinations: {
                'current_account-current_region': {
                    bucketName: 'cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}',
                    objectKey: templateHash,
                    assumeRoleArn: 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-file-publishing-role-${AWS::AccountId}-${AWS::Region}',
                },
            },
        });
        test.done();
    },
    'version check is added to template'(test) {
        var _a, _b, _c, _d, _e, _f, _g;
        // GIVEN
        new lib_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // THEN
        const template = app.synth().getStackByName('Stack').template;
        test.deepEqual((_b = (_a = template === null || template === void 0 ? void 0 : template.Parameters) === null || _a === void 0 ? void 0 : _a.BootstrapVersion) === null || _b === void 0 ? void 0 : _b.Type, 'AWS::SSM::Parameter::Value<String>');
        test.deepEqual((_d = (_c = template === null || template === void 0 ? void 0 : template.Parameters) === null || _c === void 0 ? void 0 : _c.BootstrapVersion) === null || _d === void 0 ? void 0 : _d.Default, '/cdk-bootstrap/hnb659fds/version');
        const assertions = (_g = (_f = (_e = template === null || template === void 0 ? void 0 : template.Rules) === null || _e === void 0 ? void 0 : _e.CheckBootstrapVersion) === null || _f === void 0 ? void 0 : _f.Assertions) !== null && _g !== void 0 ? _g : [];
        test.deepEqual(assertions.length, 1);
        test.deepEqual(assertions[0].Assert, {
            'Fn::Not': [
                { 'Fn::Contains': [['1', '2', '3'], { Ref: 'BootstrapVersion' }] },
            ],
        });
        test.done();
    },
    'add file asset'(test) {
        // WHEN
        const location = stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.bucketName), 'cdk-hnb659fds-assets-the_account-the_region');
        test.equals(evalCFN(location.s3Url), 'https://s3.the_region.domain.aws/cdk-hnb659fds-assets-the_account-the_region/abcdef');
        // THEN - object key contains source hash somewhere
        test.ok(location.objectKey.indexOf('abcdef') > -1);
        test.done();
    },
    'add docker image asset'(test) {
        // WHEN
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.repositoryName), 'cdk-hnb659fds-container-assets-the_account-the_region');
        test.equals(evalCFN(location.imageUri), 'the_account.dkr.ecr.the_region.domain.aws/cdk-hnb659fds-container-assets-the_account-the_region:abcdef');
        test.done();
    },
    'synthesis'(test) {
        var _a, _b;
        // GIVEN
        stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // WHEN
        const asm = app.synth();
        // THEN - we have an asset manifest with both assets and the stack template in there
        const manifest = readAssetManifest(asm);
        test.equals(Object.keys(manifest.files || {}).length, 2);
        test.equals(Object.keys(manifest.dockerImages || {}).length, 1);
        // THEN - every artifact has an assumeRoleArn
        for (const file of Object.values((_a = manifest.files) !== null && _a !== void 0 ? _a : {})) {
            for (const destination of Object.values(file.destinations)) {
                test.deepEqual(destination.assumeRoleArn, 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-file-publishing-role-${AWS::AccountId}-${AWS::Region}');
            }
        }
        for (const file of Object.values((_b = manifest.dockerImages) !== null && _b !== void 0 ? _b : {})) {
            for (const destination of Object.values(file.destinations)) {
                test.deepEqual(destination.assumeRoleArn, 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-image-publishing-role-${AWS::AccountId}-${AWS::Region}');
            }
        }
        test.done();
    },
    'customize publishing resources'(test) {
        var _a, _b, _c, _d, _e, _f;
        // GIVEN
        const myapp = new lib_1.App();
        // WHEN
        const mystack = new lib_1.Stack(myapp, 'mystack', {
            synthesizer: new lib_1.DefaultStackSynthesizer({
                fileAssetsBucketName: 'file-asset-bucket',
                fileAssetPublishingRoleArn: 'file:role:arn',
                fileAssetPublishingExternalId: 'file-external-id',
                imageAssetsRepositoryName: 'image-ecr-repository',
                imageAssetPublishingRoleArn: 'image:role:arn',
                imageAssetPublishingExternalId: 'image-external-id',
            }),
        });
        mystack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'file-asset-hash',
        });
        mystack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'docker-asset-hash',
        });
        // THEN
        const asm = myapp.synth();
        const manifest = readAssetManifest(asm);
        test.deepEqual((_c = (_b = (_a = manifest.files) === null || _a === void 0 ? void 0 : _a['file-asset-hash']) === null || _b === void 0 ? void 0 : _b.destinations) === null || _c === void 0 ? void 0 : _c['current_account-current_region'], {
            bucketName: 'file-asset-bucket',
            objectKey: 'file-asset-hash',
            assumeRoleArn: 'file:role:arn',
            assumeRoleExternalId: 'file-external-id',
        });
        test.deepEqual((_f = (_e = (_d = manifest.dockerImages) === null || _d === void 0 ? void 0 : _d['docker-asset-hash']) === null || _e === void 0 ? void 0 : _e.destinations) === null || _f === void 0 ? void 0 : _f['current_account-current_region'], {
            repositoryName: 'image-ecr-repository',
            imageTag: 'docker-asset-hash',
            assumeRoleArn: 'image:role:arn',
            assumeRoleExternalId: 'image-external-id',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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