"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTsconfigCompilerOptions = exports.extractDependencies = exports.tryGetModuleVersionFromPkg = exports.tryGetModuleVersionFromRequire = exports.exec = exports.findUpMultiple = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = new Error().stack?.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    return findUpMultiple([name], directory)[0];
}
exports.findUp = findUp;
/**
 * Find the lowest of multiple files by walking up parent directories. If
 * multiple files exist at the same level, they will all be returned.
 */
function findUpMultiple(names, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const files = [];
    for (const name of names) {
        const file = path.join(directory, name);
        if (fs.existsSync(file)) {
            files.push(file);
        }
    }
    if (files.length > 0) {
        return files;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return [];
    }
    return findUpMultiple(names, path.dirname(absoluteDirectory));
}
exports.findUpMultiple = findUpMultiple;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${proc.stdout?.toString().trim()}\n\n\nstderr: ${proc.stderr?.toString().trim()}`);
        }
        throw new Error(`${cmd} ${args.join(' ')} ${options?.cwd ? `run in directory ${options.cwd}` : ''} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersionFromRequire(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersionFromRequire = tryGetModuleVersionFromRequire;
/**
 * Gets module version from package.json content
 */
function tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath) {
    const dependencies = {
        ...pkgJson.dependencies ?? {},
        ...pkgJson.devDependencies ?? {},
        ...pkgJson.peerDependencies ?? {},
    };
    if (!dependencies[mod]) {
        return undefined;
    }
    // If it's a "file:" version, make it absolute
    const fileMatch = dependencies[mod].match(/file:(.+)/);
    if (fileMatch && !path.isAbsolute(fileMatch[1])) {
        const absoluteFilePath = path.join(path.dirname(pkgPath), fileMatch[1]);
        return `file:${absoluteFilePath}`;
    }
    ;
    return dependencies[mod];
}
exports.tryGetModuleVersionFromPkg = tryGetModuleVersionFromPkg;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    for (const mod of modules) {
        const version = tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath)
            ?? tryGetModuleVersionFromRequire(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
function getTsconfigCompilerOptions(tsconfigPath) {
    const compilerOptions = extractTsConfig(tsconfigPath);
    const excludedCompilerOptions = [
        'composite',
        'tsBuildInfoFile',
    ];
    const options = {
        ...compilerOptions,
        // Overrides
        incremental: false,
        // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next to .ts file.
        rootDir: './',
        outDir: './',
    };
    let compilerOptionsString = '';
    Object.keys(options).sort().forEach((key) => {
        if (excludedCompilerOptions.includes(key)) {
            return;
        }
        const value = options[key];
        const option = '--' + key;
        const type = typeof value;
        if (type === 'boolean') {
            if (value) {
                compilerOptionsString += option + ' ';
            }
        }
        else if (type === 'string') {
            compilerOptionsString += option + ' ' + value + ' ';
        }
        else if (type === 'object') {
            if (Array.isArray(value)) {
                compilerOptionsString += option + ' ' + value.join(',') + ' ';
            }
        }
        else {
            throw new Error(`Missing support for compilerOption: [${key}]: { ${type}, ${value}} \n`);
        }
    });
    return compilerOptionsString.trim();
}
exports.getTsconfigCompilerOptions = getTsconfigCompilerOptions;
function extractTsConfig(tsconfigPath, previousCompilerOptions) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { extends: extendedConfig, compilerOptions } = require(tsconfigPath);
    const updatedCompilerOptions = {
        ...(previousCompilerOptions ?? {}),
        ...compilerOptions,
    };
    if (extendedConfig) {
        return extractTsConfig(path.resolve(tsconfigPath.replace(/[^\/]+$/, ''), extendedConfig), updatedCompilerOptions);
    }
    return updatedCompilerOptions;
}
//# sourceMappingURL=data:application/json;base64,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