"""Abstract base class for section integrators."""

from __future__ import annotations  # To have clean hints of ArrayLike in docs

import abc

from numpy.typing import ArrayLike

from structuralcodes.geometry import CompoundGeometry


class SectionIntegrator(abc.ABC):
    """Abstract base class for section integrators."""

    @abc.abstractmethod
    def prepare_input(
        self, geo: CompoundGeometry, strain: ArrayLike, integrate: str
    ):
        """Prepare general input to the stress integration method.

        Args:
            geo (CompoundGeometry): The geometry to integrate over.
            strain (ArrayLike): The scalar strain components necessary for
                describing the assumed strain distribution over the geometry.
            integrate (str): The integrand, i.e., the quantity to be integrated
                over the section.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def integrate_stress(self, *prepared_input, **kwargs):
        """Integrate stresses over the geometry.

        The input to this method is generated by the prepare_input method. It
        also takes kwargs depending on the concrete implementation.

        Arguments:
            *prepared_input: The input prepared by the prepare_input method.

        Keyword Arguments:
            **kwargs: Keyword arguments depending on the concrete
                implementation.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def integrate_modulus(self, *prepared_input, **kwargs):
        """Integrate material modulus over the geometry.

        The input to this method is generated by the prepare_input method. It
        also takes kwargs depending on the concrete implementation.

        Arguments:
            *prepared_input: The input prepared by the prepare_input method.

        Keyword Arguments:
            **kwargs: Keyword arguments depending on the concrete
                implementation.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def integrate_strain_response_on_geometry(
        self,
        geo: CompoundGeometry,
        strain: ArrayLike,
        integrate: str,
        **kwargs,
    ):
        """Integrate stresses over the geometry to obtain the response due to
        strains.
        """
        raise NotImplementedError
