""" Tests for the PyParser.py file

PyParser is a bit hard to test, as everything depends on the loading
of an initial file by PyModule's '__init__()'. Together with the
testcases' being subclasses of eachother, this results in a lot of
tests which are effectively called a number of times with exactly the
same inputs. Ah well.
"""
import os
import sys
import unittest
# Something dirty. It is assumed that ArchGenXML isn't installed as a
# python module but that it resides "just somewhere" on the
# filesystem.
testDir = os.path.dirname(os.path.abspath(__file__))
parentDir = testDir[:-6] # Strips off '/tests'
# Appends the parent dir to the module search path
sys.path.append(parentDir)
from PyParser import PyModule


class TestPyModule(unittest.TestCase):
    def setUp(self):
        self.emptyFile = testDir + '/pythonfiles/empty.py'
        self.emptyStringFile = '# Empty python file\n# Some more'
        self.realFile = testDir + '/pythonfiles/ParserTest.py'
        self.indentationFile = testDir + '/pythonfiles/ParserTest2.py'
        self.parser = PyModule(self.realFile)
        self.function = self.parser.functions['someMethod']
        self.oneLineFunction = self.parser.functions['oneLineMethod']
        self.klass = self.parser.classes['ParserTest']
        self.method = self.klass.methods['parserMethod']
        self.manualMethod = self.klass.methods['parserMethod2']

    def testReadFile1(self):
        """ Don't barf on an empty file
        """
        self.parser.readFile(self.emptyFile)

    def testReadFile2(self):
        """ Also accept a file pointer instead of a filename
        """
        file = open(self.emptyFile)
        self.parser.readFile(file)
        file.close()

    def testReadFile3(self):
        """ Don't barf on an empty string-file
        """
        parser = PyModule(self.emptyStringFile, mode='string')

    def testReadFile4(self):
        """ Barf on an empty string-file without correct mode
        """
        self.assertRaises(IOError, self.parser.readFile,
                          self.emptyStringFile)

    def testFindClassesAndFunctions(self):
        """ Find the correct number of classes and functions

        tests/pythonfiles/ParserTest.py contains: 1 class + 1 function
         """
        self.assertEquals(len(self.parser.classes), 1)
        self.assertEquals(len(self.parser.functions), 2)

    def testFindProtectedSections(self):
        """ Find the correct number of protected sections

        tests/pythonfiles/ParserTest.py contains 4 protected sections:
        module-header, after-schema, class-header, module-footer.
        """
        self.assertEquals(len(self.parser.protectedSections), 4)

    def testFindProtectionDeclarations(self):
        """ Find the correct number of protection declarations

        tests/pythonfiles/ParserTest.py contains 2 protection
        declarations.
        """
        self.assertEquals(len(self.parser.protectionDeclarations), 2)


class TestPyCodeElement(TestPyModule):
    """ Tests for PyCodeElement class inside PyParser
    """

    def testGetSrc(self):
        # To be overwritten later
        pass

    def testGetName(self):
        # To be overwritten later
        pass


class TestPyFunction(TestPyCodeElement):
    """ Tests for PyFunction class inside PyParser
    """

    def testGetSrc(self):
        """ getSrc should return the source

        Pretty pedantic tests, but we want the source, just the
        source, luke. No messing around with it.
        """
        self.assertEquals(self.function.getSrc(), self.function.src)
        self.assertEquals(self.oneLineFunction.getSrc(),
                          self.oneLineFunction.src)

    def testGetName(self):
        """ getName should return the name

        Also pretty pedantic, but safeguards against messing around!
        """
        self.assertEquals(self.oneLineFunction.getName(), 'oneLineMethod')
        self.assertEquals(self.function.getName(), 'someMethod')

    def testCodeLength(self):
        """ Find the correct length of code for two examples
        """
        self.assertEquals(self.function.codeLength(), 2)
        self.assertEquals(self.oneLineFunction.codeLength(), 1)

    def testExtractCode(self):
        """ Output the correct code from the test files
        """
        desired = 'def oneLineMethod(): pass\n'
        self.assertEquals(self.oneLineFunction.extractCode(), desired)
        desired2 = 'def someMethod():\n    pass\n'
        self.assertEquals(self.function.extractCode(), desired2)

    def testBuildMethod(self):
        """ Test function name extraction

        self.name extraction is the only thing that needs checking,
        extractCode is checked seperately.
        """
        self.assertEquals(self.oneLineFunction.name, 'oneLineMethod')
        self.assertEquals(self.function.name, 'someMethod')


class TestPyMethod(TestPyFunction):
    """ Tests for PyMethod class inside PyParser
    """

    def testGetSrc(self):
        """ getSrc should return the source

        Pretty pedantic tests, but we want the source, just the
        source, luke. No messing around with it.
        """
        self.assertEquals(self.method.getSrc(), self.method.src)
        self.assertEquals(self.manualMethod.getSrc(),
                          self.manualMethod.src)

    def testGetName(self):
        """ getName should return the name

        Also pretty pedantic, but safeguards against messing around!
        """
        self.assertEquals(self.oneLineFunction.getName(), 'oneLineMethod')
        self.assertEquals(self.function.getName(), 'someMethod')

    def _temp_disabled_as_it_fails_testCodeLength(self):
        """ Find the correct length of code for two methods
        """
        self.assertEquals(self.method.codeLength(), 6)
        self.assertEquals(self.manualMethod.codeLength(), 6)

    def testBuildMethod(self):
        """ Test method name extraction
        """
        self.assertEquals(self.method.name, 'parserMethod')
        self.assertEquals(self.manualMethod.name, 'parserMethod2')

    def _temp_disabled_as_it_fails_testStrangeIndentation(self):
        parser = PyModule(self.indentationFile)
        klass = parser.classes['testAtpConnector']
        method = klass.methods['testmethodcorrect']
        self.assertEquals(method.codeLength(), 13)
        method = klass.methods['testmethod']
        self.assertEquals(method.codeLength(), 12)

class TestPyClass(TestPyCodeElement):
    """ Tests for PyClass class inside PyParser
    """
    def testGetDocumentation(self):
        """ Find the docstring of a class
        """
        result = """ Doctest line 1

    Doctest line 2
    """
        # print self.klass.code.co_consts[0]
        self.assertEquals(self.klass.getDocumentation(), result)

    def testBuildMethods(self):
        """ Find correct number of methods in the example file
        """
        self.assertEquals(len(self.klass.methods), 2)
        # The correct names are already tested in above ..._method
        # tests.

    def testGetProtectedSection(self):
        """ Find the class's protected section
        """
        # There should be a 'class-header' protected section in the
        # example file.
        result = self.klass.getProtectedSection('class-header')
        expected = "    someTest = 'Some value'"
        self.assertEquals(result, expected)

    def testGetMethodNames(self):
        """ Find the correct two names
        """
        names = ['parserMethod', 'parserMethod2']
        self.assertEquals(self.klass.getMethodNames().sort(),
                          names.sort())

    def testGetProtectionDeclaration(self):
        """ Find the security.declare... for a manual method
        """
        expected = "    security.declarePublic('parserMethod2')"
        self.assertEquals(self.klass.getProtectionDeclaration('parserMethod2'),
                          expected)

def test_suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestPyModule))
    suite.addTest(unittest.makeSuite(TestPyCodeElement))
    suite.addTest(unittest.makeSuite(TestPyFunction))
    suite.addTest(unittest.makeSuite(TestPyMethod))
    suite.addTest(unittest.makeSuite(TestPyClass))
    return suite

if __name__ == '__main__':
    import utils
    utils.initLog('runalltests.log')
    unittest.main()
