"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = exports.tryGetModuleVersion = exports.exec = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const file = path.join(directory, name);
    if (fs.existsSync(file)) {
        return file;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersion(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersion = tryGetModuleVersion;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a, _b, _c, _d;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    const pkgDependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    for (const mod of modules) {
        const version = (_d = pkgDependencies[mod]) !== null && _d !== void 0 ? _d : tryGetModuleVersion(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
//# sourceMappingURL=data:application/json;base64,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