# Copyright 2025 PhonePe Private Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from dataclasses import dataclass, field
from typing import List

from dataclasses_json import dataclass_json, LetterCase

from phonepe.sdk.pg.common.models.request.instrument_constraint import (
    InstrumentConstraint,
)
from phonepe.sdk.pg.payments.v2.models.request.merchant_urls import MerchantUrls
from phonepe.sdk.pg.common.models.request.meta_info import MetaInfo
from phonepe.sdk.pg.common.models.request.payment_flow import PaymentFlow
from phonepe.sdk.pg.payments.v2.models.request.pg_checkout_payment_flow import (
    PgCheckoutPaymentFlow,
)


@dataclass_json(letter_case=LetterCase.CAMEL)
@dataclass
class CreateSdkOrderRequest:
    merchant_order_id: str = field(default=None)
    amount: int = field(default=None)
    meta_info: MetaInfo = field(default=None)
    payment_flow: PaymentFlow = field(default=None)
    constraints: List[InstrumentConstraint] = field(default=None)
    expire_after: int = field(default=None)
    disable_payment_retry: bool = field(default=None)

    @staticmethod
    def build_standard_checkout_request(
        merchant_order_id: str,
        amount: int,
        redirect_url: str = None,
        message: str = None,
        meta_info: MetaInfo = None,
        expire_after: int = None,
        disable_payment_retry: bool = None
    ):
        """
        Builds SDK order Request for standard checkout client

        Parameters
        ----------
        merchant_order_id: str
            The unique order ID generated by the merchant
        amount: int
            Order amount in paisa
        redirect_url: str
            URL where user will be redirected after success/failed payment
        message: str
            Payment message shown in app for collect requests
        meta_info: MetaInfo
            Merchant defined meta info to store additional information
        expire_after: int
            Order expiry in seconds. If not passed default value will be used
        disable_payment_retry: bool
            disable payment retry parameter for standard checkout allows merchants to control if endUser is allowed to do a payment retry on the payment page

        Returns
        ----------
        CreateSdkOrderRequest:
            Request object
        """

        return CreateSdkOrderRequest(
            merchant_order_id=merchant_order_id,
            amount=amount,
            meta_info=meta_info,
            payment_flow=PgCheckoutPaymentFlow(
                message=message, merchant_urls=MerchantUrls(redirect_url=redirect_url)
            ),
            expire_after=expire_after,
            disable_payment_retry=disable_payment_retry
        )

    @staticmethod
    def build_custom_checkout_request(
        merchant_order_id: str,
        amount: int,
        meta_info: MetaInfo = None,
        constraints: List[InstrumentConstraint] = None,
        expire_after: int = None,
        disable_payment_retry: bool = None
    ):
        """
        Builds SDK order Request for custom checkout client

        Parameters
        ----------
        merchant_order_id: str
            The unique order ID generated by the merchant
        amount: int
            Order amount in paisa
        meta_info: MetaInfo
            Merchant defined meta info to store additional information
        constraints: List[InstrumentConstraints]
            Different type of constraints that must be applied to the payment
        expire_after: int
            Order expiry in seconds. If not passed default value will be used
        disable_payment_retry: bool
            disable payment retry parameter for standard checkout allows merchants to control if endUser is allowed to do a payment retry on the payment page

        Returns
        ----------
        SdkOrderRequest:
            Request object
        """

        return CreateSdkOrderRequest(
            merchant_order_id=merchant_order_id,
            amount=amount,
            meta_info=meta_info,
            payment_flow=PgCheckoutPaymentFlow(),
            constraints=constraints,
            expire_after=expire_after,
            disable_payment_retry=disable_payment_retry
        )
