#!/usr/bin/env python3
import dataclasses
import json
import multiprocessing
import os
import re
import shutil
import subprocess
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import Optional, Any, List, Dict, Tuple, Set
import logging
from types import SimpleNamespace
import tarfile
import json5
import csv
import time
import requests
import urllib3.util
from strenum import StrEnum
from rich.console import Console


scripts_dir_path = Path(__file__).parent.resolve()  # containing directory
sys.path.insert(0, str(scripts_dir_path))

from EVMVerifier.certoraContextValidator import KEY_ENV_VAR
from Mutate import mutateConstants as Constants
from Shared import certoraUtils as Util
from Shared.certoraLogging import LoggingManager
from certoraRun import run_certora, CertoraRunResult
from Shared import certoraValidateFuncs as Vf
from EVMVerifier.Compiler.CompilerCollectorFactory import get_relevant_compiler
from Mutate import mutateAttribute as Attr


DEFAULT_NUM_MUTANTS = 5

mutation_logger = logging.getLogger("mutation")


"""
Class definitions section start
"""


class MutationTestRuleStatus(StrEnum):
    SUCCESS = "SUCCESS"
    FAIL = "FAIL"
    TIMEOUT = "TIMEOUT"
    SANITY_FAIL = "SANITY_FAIL"
    UNKNOWN = "UNKNOWN"


@dataclass
class RuleResult:
    """
    This class represents the verification results of a mutant on a specific rule.

    name - the name of the rule tested
    status - the status of the rule, a string from MutationTestRuleStatus
    """
    name: str
    status: str

    def to_json(self) -> dict:
        return {
            "name": self.name,
            "status": self.status
        }


@dataclass
class Mutant:
    """
    A class that represents a mutant. It can be one of three types:
    1. The original, unmutated code
    2. A random mutation generated by Gambit
    3. A manual mutation carefully crafted by hand

    Example:
    {
        "filename": "gambit_out/mutants/1/C.sol",
        "original_filename": "C.sol",
        "directory": "gambit_out/mutants/1",
        "id": "1",
        "diff": "--- original\n+++ mutant\n@@ -1,5 +1,6 ...",
        "description": "DeleteExpressionMutation"
    }
    """
    filename: str
    original_filename: str
    directory: str
    id: str
    diff: str
    description: str
    name: str = ""

    def __post_init__(self) -> None:
        # Set the 'name' attribute based on the 'filename'. Intended to be used by Gambit mutants
        if not self.name:
            self.name = f"{self.id}_{Path(self.filename).stem}"

    def __str__(self) -> str:
        return self.name


@dataclass
class MutantJob:
    """
    A class that represents a mutation verification job sent to the cloud. It does not include verification results.

    Example:
    {
        "gambit_mutant": {
            "filename": "gambit_out/mutants/1/C.sol",
            "original_filename": "C.sol",
            "directory": "gambit_out/mutants/1",
            "id": "1",
            "diff": "--- original\n+++ mutant\n@@ -1,5 +1,6 ...",
            "description": "DeleteExpressionMutation"
        },
        "link": "https://vaas-stg.certora.com/jobStatus/85695/d7b565e07bb5408bbfd1680aa61f0eb8?anonymousKey=...",
        "success": true,
        "run_directory": ".certora_internal/24_01_29_19_04_42_864/.certora_sources",
        "rule_report_link": "https://vaas-stg.certora.com/output/85695/d7b565e07bb5408bbfd1680a?anonymousKey=..."
    }
    """
    gambit_mutant: Mutant
    link: Optional[str]
    success: bool
    run_directory: Optional[str]
    rule_report_link: Optional[str] = None

    def __post_init__(self) -> None:
        if isinstance(self.gambit_mutant, dict):
            self.gambit_mutant = Mutant(**self.gambit_mutant)

    def __str__(self) -> str:
        return self.gambit_mutant.__str__()


@dataclass
class TestJobs:
    """
    A class that holds the status of all the mutation test's jobs as fetched from the server.
    It has two components:
    1. A link to the run of the original code without any mutations.
    2. A list of verification jobs for the mutations.

    Note: The jobs may not be completed yet. This class does not hold any verification results.

    Example:
    {
       "original": "https://vaas-stg.certora.com/output/85695/ee248ffd4f1a4b8e82dd90915f995a2e?anonymousKey=...",
       "mutants": [
          {
             "gambit_mutant":{
                "filename": "gambit_out/mutants/1/C.sol",
                "original_filename": "C.sol",
                "directory": "gambit_out/mutants/1",
                "id": "1",
                "diff": "--- original\n+++ mutant\n@@ -1,5 +1,6 ...",
                "description": "DeleteExpressionMutation"
             },
             "link": "https://vaas-stg.certora.com/jobStatus/85695/d7b565e07bb5408bbfd1680aa61f0eb8?anonymousKey=...",
             "success": true,
             "run_directory": ".certora_internal/24_01_29_19_04_42_864/.certora_sources",
             "rule_report_link": "https://vaas-stg.certora.com/output/85695/d7b565e07bb5408bbfd1680a?anonymousKey=..."
          },
          ...
       ]
    }
    """
    original: str
    mutants: List[MutantJob]

    def __post_init__(self) -> None:
        self.mutants = [MutantJob(**mutant) for mutant in self.mutants if isinstance(mutant, dict)]


@dataclass
class MutantJobWithResults:
    """
    A class that holds a mutant verification job with its results.
    It has two components:
    1. A link to the run of the original code without any mutations.
    2. A list of verification jobs for the mutations.

    Note: The jobs may not be completed yet. This class does not hold any verification results.

    Example:
    {
        "mutant_job": {
             "gambit_mutant": {
                "filename": "gambit_out/mutants/1/C.sol",
                "original_filename": "C.sol",
                "directory": "gambit_out/mutants/1",
                "id": "1",
                "diff": "--- original\n+++ mutant\n@@ -1,5 +1,6 ...",
                "description": "DeleteExpressionMutation"
             },
             "link": "https://vaas-stg.certora.com/jobStatus/85695/d7b565e07bb5408bbfd1680aa61f0eb8?anonymousKey=...",
             "success": true,
             "run_directory": ".certora_internal/24_01_29_19_04_42_864/.certora_sources",
             "rule_report_link": "https://vaas-stg.certora.com/output/85695/d7b565e07bb5408bbfd1680a?anonymousKey=..."
        },
        "rule_results": [
            {
                "name": 'envfreeFuncsStaticCheck',
                "status": "SUCCESS"
            },
            ...
        ]
    }

    """

    mutant_job: MutantJob
    rule_results: List[RuleResult]

    # These are inferred from rule_results
    rules_with_result: List[RuleResult] = dataclasses.field(init=False)
    SANITY_FAIL: List[RuleResult] = dataclasses.field(init=False)
    UNKNOWN: List[RuleResult] = dataclasses.field(init=False)
    TIMEOUT: List[RuleResult] = dataclasses.field(init=False)

    def all_rule_names(self) -> List[str]:
        return [rule.name for rule in self.rule_results]

    def __validate_unique_rule_names(self) -> None:
        # Check for duplicates
        rule_names: Set[str] = set()
        for rule in self.rule_results:
            if rule.name in rule_names:
                raise RuntimeError(f"Found rule {rule.name} twice for mutant {self}. Malformed server response")
            rule_names.add(rule.name)

    def __post_init__(self) -> None:
        self.__validate_unique_rule_names()
        self.__populate_rules()

    def __populate_rules(self) -> None:
        self.rules_with_result = \
            [r for r in self.rule_results if r.status in
             {MutationTestRuleStatus.SUCCESS.value, MutationTestRuleStatus.FAIL.value}]
        self.SANITY_FAIL = [r for r in self.rule_results if r.status == MutationTestRuleStatus.SANITY_FAIL.value]
        self.UNKNOWN = [r for r in self.rule_results if r.status == MutationTestRuleStatus.UNKNOWN.value]
        self.TIMEOUT = [r for r in self.rule_results if r.status == MutationTestRuleStatus.TIMEOUT.value]

    def get_rule_status(self, rule_name: str) -> str:

        rule_status = next((rule.status for rule in self.rule_results if rule.name == rule_name), None)
        if not rule_status:
            rule_status = next((rule.status for rule in self.SANITY_FAIL if rule.name == rule_name), None)

        if not rule_status:
            return Constants.DEFAULT_CSV_JOB_STATUS

        if rule_status == MutationTestRuleStatus.SUCCESS.value:
            return Constants.UNCAUGHT
        if rule_status == MutationTestRuleStatus.FAIL.value:
            return Constants.CAUGHT
        return rule_status

    def __str__(self) -> str:
        return f"{self.mutant_job.gambit_mutant} {self.mutant_job.link}"

    def to_report_json(self) -> dict:
        """
        Returns a dict representation that is JSON serializable of this class meant for the web report

        Example:
        {
          "description": "mutations/EBTCToken/EBTCToken_3.sol",
          "diff": "274c274,275\n<         require(cachedBalance >= amount...",
          "link": "https://prover.certora.com/jobStatus/93493/2ecb10...f?anonymousKey=5b2b0...",
          "name": "m1_EBTCToken_3",
          "id": "m1",
          "rules": [
            {
              "name": "mingIntegrity",
              "status": "SUCCESS"
            },
            ...
          ],
          "SANITY_FAIL": [],
          "UNKNOWN": [],
          "TIMEOUT": []
        }
        """
        data = {
            'description': self.mutant_job.gambit_mutant.description,
            'diff': self.mutant_job.gambit_mutant.diff,
            'link': self.mutant_job.link or "",
            'name': str(self.mutant_job.gambit_mutant),
            'id': self.mutant_job.gambit_mutant.id,
            'rules': [rule.__dict__ for rule in self.rules_with_result],
            'SANITY_FAIL': [rule.__dict__ for rule in self.SANITY_FAIL],
            'UNKNOWN': [rule.__dict__ for rule in self.UNKNOWN],
            'TIMEOUT': [rule.__dict__ for rule in self.TIMEOUT]
        }
        return data


"""
Class definitions section end
"""


def find_cwd(root_dir: Path) -> Path:
    """
    find the file .cwd under root_dir. checks that there is at most one such file under root_dir. If the file .cwd is
    not found it means root_dir is also the cwd (as it is in most cases)

    :param root_dir:
    :return: the directory of .cwd
    """
    results = []

    for root, _, files in os.walk(root_dir):
        if Util.CWD_FILE in files:
            results.append(root)
    if len(results) == 0:
        return root_dir
    if len(results) > 1:
        raise RuntimeError(f"found multiple {Util.CWD_FILE} files under {root_dir}: {results}")
    return Path(results[0])


def print_separator(txt: str) -> None:
    print(f"\n{'*' * 20} {txt} {'*' * 20}\n")


def wait_for_job(report_url: str) -> bool:
    attempts = 50
    sleep_time_in_sec = 10
    job_status_url = report_url.replace('/output/', '/jobData/')

    for _ in range(attempts):
        try:
            resp = requests.get(job_status_url)
        except requests.exceptions.MissingSchema:
            return False
        except requests.RequestException:
            return False

        if resp.status_code == 200:
            if resp.json()['jobStatus'] == 'SUCCEEDED':
                return True
            if resp.json()['jobStatus'] == 'FAILED':
                return False
        else:
            return False
        time.sleep(sleep_time_in_sec)
    return False


def get_conf_from_certora_metadata(certora_sources: Path) -> Dict:
    metadata_file = certora_sources / ".certora_metadata.json"
    if metadata_file.exists():
        with metadata_file.open() as orig_run_conf:
            metadata = json.load(orig_run_conf)
            if Constants.CONF in metadata:
                return metadata[Constants.CONF]
            else:
                raise Util.CertoraUserInputError(f"{metadata_file} does not have the prover conf entry. Exiting.")
    else:
        raise RuntimeError(f"Could not find .certora_metadata.json in {certora_sources}. "
                           "Try certoraMutate with --conf.")


def get_diff(original: Path, mutant: Path) -> str:
    test_result = subprocess.run(["diff", "--help"], capture_output=True, text=True)
    if test_result.returncode:
        mutation_logger.warning("Unable to get diff for manual mutations. Install 'diff' and try again to "
                                "see more detailed information")
        return ""
    result = subprocess.run(["diff", str(original), str(mutant)], capture_output=True, text=True)
    if result.stdout is None:
        logging.warning(f"Could not get the diff with the mutated file {mutant}")
        logging.debug(f"original file: {original}, diff results: {result}")
    return result.stdout


def get_gambit_exec() -> str:
    exec = Util.get_package_resource(Util.CERTORA_BINS / Constants.GAMBIT)
    # try executing it
    try:
        rc = subprocess.run([exec, "--help"], stdout=subprocess.DEVNULL, stderr=subprocess.PIPE)
        if rc.returncode == 0:
            return str(exec)
        else:
            mutation_logger.info(f"Failed to execute {exec}")
            stderr_lines = rc.stderr.splitlines()
            raise Util.CertoraUserInputError(f"Failed to execute {exec} \n\n{stderr_lines}")
    except Exception:
        # could not run the specialized name, just run gambit
        return Constants.GAMBIT


def valid_link(link: str) -> bool:
    """
    Returns true if the provided link string is either a valid URL or a valid directory path
    """
    return Vf.Util.is_valid_url(urllib3.util.parse_url(link)) or validate_dir(link)


def validate_dir(url: str) -> bool:
    try:
        return Path(url).is_dir()
    except Exception:
        return False


def valid_message(msg: str) -> None:
    if msg:
        pattern = re.compile(r"^[0-9a-zA-Z_=, ':\.\-\/]+$")
        if not re.match(pattern, msg):
            raise Util.CertoraUserInputError("The message includes a forbidden character")


def store_in_a_file(results: List[Any], ui_out: Path) -> None:
    try:
        with ui_out.open('w+') as ui_out_json:
            json.dump(results, ui_out_json, indent=4)
    except Exception as e:
        mutation_logger.warning(f"Failed to output to {ui_out}")
        mutation_logger.debug(f"{e}")


def read_file(ui_out: Path) -> Any:
    if ui_out.exists():
        try:
            with ui_out.open('r') as ui_out_json:
                return json.load(ui_out_json)
        except Exception:
            mutation_logger.warning(f"Failed to read {ui_out}")
    else:
        mutation_logger.warning(f"Couldn't locate the output file ({ui_out})")


class WebUtils:
    def __init__(self, args: SimpleNamespace):
        self.server = self.config_server(args)
        self.max_timeout_attempts_count = \
            args.max_timeout_attempts_count or Constants.DEFAULT_MAX_TIMEOUT_ATTEMPTS_COUNT
        self.request_timeout = args.request_timeout
        if self.server == Constants.STAGING:
            domain = Constants.STAGING_DOTCOM
            mutation_test_domain = Constants.MUTATION_TEST_REPORT_STAGING
        elif self.server == Constants.PRODUCTION:
            domain = Constants.PROVER_DOTCOM
            mutation_test_domain = Constants.MUTATION_TEST_REPORT_PRODUCTION
        elif self.server == Constants.DEV:
            domain = Constants.DEV_DOTCOM
            mutation_test_domain = Constants.MUTATION_TEST_REPORT_DEV
        else:
            raise Util.CertoraUserInputError(f"Invalid server name {self.server}")
        self.mutation_test_id_url = f"https://{domain}/mutationTesting/initiate/"
        self.mutation_test_submit_final_result_url = f"https://{domain}/mutationTesting/getUploadInfo/"
        self.mutation_test_final_result_url = f"https://{mutation_test_domain}"
        mutation_logger.debug(f"Using server {self.server} with mutation_test_id_url {self.mutation_test_id_url}")

    @staticmethod
    def config_server(args: SimpleNamespace) -> str:
        """
        If given a server, it is taken.
        Otherwise, computes from either the conf file or the orig run link.
        """
        # default production
        default = Constants.PRODUCTION
        if args.server:
            return args.server

        server_in_prover_conf = getattr(args, 'prover_dict', {}).get(Constants.SERVER)
        if server_in_prover_conf:
            return server_in_prover_conf

        if args.orig_run is not None:
            if Constants.STAGING_DOTCOM in args.orig_run:
                return Constants.STAGING
            elif Constants.PROVER_DOTCOM in args.orig_run:
                return default
            elif Constants.DEV_DOTCOM in args.orig_run:
                return Constants.DEV
            else:
                raise Util.CertoraUserInputError(f"{args.orig_run} link has an unsupported domain.")
        else:
            return default

    def put_response_with_timeout(self, url: str, data: Any, headers: Dict[str, str]) -> Optional[requests.Response]:
        """
        Executes a put request and returns the response, uses a timeout mechanism

        Args
        ----
            url (str): the URL to send a PUT request to
            data (Any): the data to send
            headers (dict[str, str]): an optional set of headers

        Returns
        -------
            Optional[requests.Response]: if any of the attempt succeeded, returns the response
        """
        for i in range(self.max_timeout_attempts_count):
            try:
                return requests.put(url, data=data, timeout=self.request_timeout,
                                    headers=headers)
            except Exception:
                mutation_logger.debug(f"attempt {i} failed to put url {url}.")
        return None

    def put_json_request_with_timeout(self, url: str, body: Dict[str, Any],
                                      headers: Dict[str, str]) -> Optional[requests.Response]:
        """
        Executes a put request and returns the response, uses a timeout mechanism

        Args
        ----
            url (str): the URL to send a PUT request to
            body (dict[str, Any]): request body
            headers (dict[str, str]): an optional set of headers

        Returns
        -------
            Optional[requests.Response]: if any of the attempt succeeded, returns the response
        """
        for i in range(self.max_timeout_attempts_count):
            try:
                return requests.put(url, json=body, timeout=self.request_timeout,
                                    headers=headers)
            except Exception:
                mutation_logger.debug(f"attempt {i} failed to put url {url}.")
        return None

    def get_response_with_timeout(self, url: str,
                                  cookies: Dict[str, str] = {}, stream: bool = False) -> Optional[requests.Response]:
        """
        Executes a get request and returns the response, uses a timeout mechanism

        Args
        ----
            url (str): the URL to send a GET request to
            cookies (dict[str, str]): an optional set of cookies/request data
            stream (bool): use a lazy way to download large files

        Returns
        -------
            Optional[requests.Response]: if any of the attempt succeeded, returns the response
        """
        for i in range(self.max_timeout_attempts_count):
            try:
                resp = requests.get(url, timeout=self.request_timeout, cookies=cookies, stream=stream)
                return resp
            except Exception:
                mutation_logger.info(f"attempt {i} failed to get url {url}.")
        return None


# SUBMIT PHASE FUNCTIONALITY


def check_key_exists() -> None:
    if KEY_ENV_VAR not in os.environ:
        raise Util.CertoraUserInputError("Cannot run mutation testing without a Certora key.")


class ReportFetcher:
    def get_output_json(self, link: str) -> Optional[Dict[str, Any]]:
        pass

    def get_treeview_json(self, link: str) -> Optional[Dict[str, Any]]:
        pass


class TreeViewStatus(StrEnum):
    RUNNING = "RUNNING"
    VERIFIED = "VERIFIED"
    SKIPPED = "SKIPPED"
    TIMEOUT = "TIMEOUT"
    ERROR = "ERROR"
    UNKNOWN = "UNKNOWN"
    SANITY_FAILED = "SANITY_FAILED"
    VIOLATED = "VIOLATED"


class FinalJobStatus(StrEnum):
    SUCCEEDED = "SUCCEEDED"
    FAILED = "FAILED"
    SERVICE_UNAVAILABLE = "SERVICE_UNAVAILABLE"
    ERROR = "ERROR"
    LAMBDA_ERROR = "LAMBDA_ERROR"

    @classmethod
    def get_statuses(cls) -> List[str]:
        return [s.value for s in cls]


def convert_to_mutation_testing_status(treeview_status: str) -> str:
    if (treeview_status == TreeViewStatus.VERIFIED) or (treeview_status == TreeViewStatus.SKIPPED):
        return MutationTestRuleStatus.SUCCESS.value
    elif treeview_status == TreeViewStatus.VIOLATED:
        return MutationTestRuleStatus.FAIL.value
    elif treeview_status == TreeViewStatus.TIMEOUT:
        return MutationTestRuleStatus.TIMEOUT.value
    elif treeview_status == TreeViewStatus.SANITY_FAILED:
        return MutationTestRuleStatus.SANITY_FAIL.value
    else:
        return MutationTestRuleStatus.UNKNOWN.value


def generate_manual_mutant_object(manual_mutant: Path, orig_file: Path) -> Mutant:
    """
    Generate a Mutant object for a manual mutation.

    Parameters:
    - `manual_mutant`: Path to the manual mutant file.
    - `orig_file`: Path to the original file.

    Returns:
    - A Mutant object representing the parsed manual mutant.

    """
    mutant_id = manual_mutant.stem
    return Mutant(
        filename=str(manual_mutant),
        original_filename=str(orig_file),
        directory=str(manual_mutant.parent),
        id=mutant_id,
        diff=get_diff(orig_file, manual_mutant),
        description=str(manual_mutant),  # NOTE: parse a description from the mutant source
        name=mutant_id
    )


class MutateApp:
    conf: Path
    conf_no_flag: Path
    test: Optional[str]
    orig_run: Optional[str]
    msg: Optional[str]
    server: Optional[str]
    prover_version: Optional[str]
    debug: bool
    orig_run_dir: Optional[Path]
    outdir: Optional[Path]
    gambit_only: bool
    dump_failed_collects: Optional[Path]
    ui_out: Optional[Path]
    dump_link: Optional[Path]
    dump_csv: Optional[Path]
    sync: bool
    wait_for_original_run: bool
    collect_file: Optional[Path]
    poll_timeout: Optional[int]
    max_timeout_attempts_count: Optional[int]
    request_timeout: Optional[int]
    gambit: Optional[Dict]
    manual_mutants: Optional[Dict]
    prover_dict: Dict

    def __init__(self) -> None:
        self.with_split_stats_data = False
        self.manual_mutants_list: List[Mutant] = list()

    def checks_before_settings_defaults(self) -> None:
        # if not self.orig_run:
        #     mutation_logger.warning("using --orig_run_dir without --orig_run has no effect")
        # else:
        #     self.orig_run_dir = Constants.CERTORA_MUTATE_SOURCES
        pass

    def set_defaults(self) -> None:
        MutateApp.__annotations__["orig_run_dir"] = Optional[str]
        if self.orig_run_dir and self.orig_run_dir != Constants.CERTORA_MUTATE_SOURCES:
            if self.orig_run_dir.exists():
                raise FileExistsError(f"orig_run_dir: {self.orig_run_dir} already exists")

        shutil.rmtree(str(Constants.CERTORA_MUTATE_SOURCES), ignore_errors=True)
        if not self.orig_run_dir:
            self.orig_run_dir = Constants.CERTORA_MUTATE_SOURCES

        if not self.dump_failed_collects:
            self.dump_failed_collects = Constants.DEFAULT_DUMP_FAILED_COLLECTS
        if not self.collect_file:
            self.collect_file = Constants.DEFAULT_COLLECT_FILE
        if not self.poll_timeout:
            self.poll_timeout = Constants.DEFAULT_POLL_TIMEOUT_IN_SECS
        if not self.max_timeout_attempts_count:
            self.max_timeout_attempts_count = Constants.DEFAULT_MAX_TIMEOUT_ATTEMPTS_COUNT
        if not self.request_timeout:
            self.request_timeout = Constants.DEFAULT_REQUEST_TIMEOUT_IN_SECS
        if not self.ui_out:
            self.ui_out = Constants.DEFAULT_UI_OUT
        if self.outdir and self.outdir != Constants.GAMBIT_OUT:
            if self.outdir.exists():
                raise FileExistsError(f"outdir: {self.outdir} already exists")
        shutil.rmtree(str(Constants.GAMBIT_OUT), ignore_errors=True)
        shutil.rmtree(str(Constants.APPLIED_MUTANTS_DIR), ignore_errors=True)

        if not self.outdir:
            self.outdir = Constants.GAMBIT_OUT

    def fetch_and_extract_inputs_dir(self, url: Optional[str]) -> Path:
        if not url:
            raise Util.CertoraUserInputError("URL for original run is null")
        web_utils = WebUtils(SimpleNamespace(**vars(self)))
        zip_output_url = url.replace(Constants.OUTPUT, Constants.ZIPOUTPUT)
        response = web_utils.get_response_with_timeout(zip_output_url, stream=True, cookies=default_cookies)
        if not response:
            raise RuntimeError("Could not fetch zip output from previous run."
                               "Try running certoraMutate with --conf.")
        if response.status_code == 200:
            with open(Constants.ZIP_PATH, 'wb') as f:
                for chunk in response.iter_content(chunk_size=128):
                    f.write(chunk)
        else:
            raise RuntimeError(f"Failed to fetch inputs dir from {url}. Got status code: {response.status_code}. "
                               f"Try running certoraMutate with --conf.")
        try:
            extract = tarfile.open(Constants.ZIP_PATH, "r")
            assert self.orig_run_dir
            extract.extractall(self.orig_run_dir)
            return self.orig_run_dir / Constants.TARNAME / Constants.INPUTS
        except Exception:
            raise Util.CertoraUserInputError(f"Failed to extract .certora_source from {url}.")

    def print_notification_msg(self) -> None:
        if self.server == "staging":
            dashboard = Constants.MUTATION_DASHBOARD_STAGING
        else:
            dashboard = Constants.MUTATION_DASHBOARD_PRODUCTION
        print(
            f"You will receive an email notification when this mutation test is completed. It may take several hours.\n"
            f"You can follow the test's progress at {dashboard}"
        )

    def settings_post_parsing(self) -> None:
        if self.debug:
            LoggingManager().set_log_level_and_format(debug=True)

        if self.gambit and not isinstance(self.gambit, list):  # type: ignore
            self.gambit = [self.gambit]  # type: ignore

    def read_conf_from_orig_run(self) -> None:
        """
        If the attribute orig_run is set all prover settings need to be fetched from the cloud based on the
        provided report link. Once we get it we copy it to the CWD all following calls to certoraRun will use
        this conf file. If the conf file contains the mutation object we delete it.
        certoraRun should in any case ignore this object, we delete it only for clarity
        :return: None
        """
        if len(self.prover_dict) != 1 and Constants.MUTATIONS not in self.prover_dict:
            raise Util.CertoraUserInputError("when running from orig run conf file should have just one  "
                                             "mutation object")

        self.fetch_and_extract_inputs_dir(self.orig_run)
        # we need orig_run_dir before we read it, and before we set defaults
        assert self.orig_run_dir
        input_dir = self.orig_run_dir / Constants.TARNAME / Constants.INPUTS
        sources_dir = input_dir / Constants.CERTORA_SOURCES
        self.prover_dict = get_conf_from_certora_metadata(input_dir)  # type: Dict
        with Constants.ORIG_RUN_PROVER_CONF.open('w') as p_conf:
            json.dump(self.prover_dict, p_conf)
        shutil.copy(Constants.ORIG_RUN_PROVER_CONF, sources_dir)
        if Constants.MUTATIONS in self.prover_dict:
            del self.prover_dict[Constants.MUTATIONS]

    def submit(self) -> None:
        mutation_logger.info("Generating mutants and submitting...")

        # ensure .certora_internal exists
        os.makedirs(Util.CERTORA_INTERNAL_ROOT, exist_ok=True)

        local = None
        sources_dir = None

        # call gambit
        generated_mutants = []

        if getattr(self, Constants.GAMBIT, False):
            generated_mutants = self.run_gambit()
        elif self.gambit_only:
            raise Util.CertoraUserInputError("gambit section must exist when running with 'gambit_only'")
        if self.gambit_only:
            sys.exit(0)

        if self.manual_mutants:
            self.manual_mutants_list = self.parse_manual_mutations()
            mutation_logger.debug(f"successfully parsed manual mutants from {self.conf}")

        if not self.orig_run:
            prover_conf = self.conf
            mutation_logger.warning("Running without a link to a previously successful prover run on "
                                    "the original contract. So we will first submit the original Prover configuration. "
                                    "No source mutations...")
            print_separator('PROVER START')
            # run original run. if it fails to compile, nothing to continue with
            success, certora_run_result = self.run_certora_prover(self.conf, msg=Constants.ORIGINAL, is_mutant=False)

            if certora_run_result:
                local = certora_run_result.is_local_link
                sources_dir = certora_run_result.src_dir
                if local:
                    assert certora_run_result.link
                    result_link = certora_run_result.link
                    src = Path(certora_run_result.link) / Constants.REPORTS / Constants.SPLIT_STATS_DATA
                    if src.is_file():
                        try:
                            shutil.copy(src, Path.cwd())
                            self.with_split_stats_data = True
                        except Exception as e:
                            logging.debug(f"did not manage to copy {Constants.SPLIT_STATS_DATA} to current directory: "
                                          f"{e}")
                    else:
                        logging.debug(f"{Constants.SPLIT_STATS_DATA} is not in {src.parent}")
                else:
                    assert certora_run_result.rule_report_link
                    result_link = certora_run_result.rule_report_link
                    if self.wait_for_original_run:
                        if wait_for_job(result_link) and download_report_file(result_link, Constants.SPLIT_STATS_DATA):
                            self.with_split_stats_data = True

            if not success or not certora_run_result or not result_link or not sources_dir:
                try:
                    if self.dump_failed_collects:
                        with self.dump_failed_collects.open('w') as failed_collection:
                            failed_collection.write("failed to collect original run")
                except Exception as e:
                    mutation_logger.debug(f"Couldn't write to collection failures file:{e}")
                raise Util.CertoraUserInputError("Original run was not successful. Cannot run mutation testing.")

            if local:
                try:
                    if not Path(result_link).is_dir():
                        raise Util.CertoraUserInputError("Unexpected path for a local job", result_link)
                except TypeError:
                    raise Util.CertoraUserInputError("link is None")
            else:
                if not Vf.Util.is_valid_url(urllib3.util.parse_url(result_link)):
                    raise Util.CertoraUserInputError(f"Invalid certoraRun result: {result_link}")
            print_separator('PROVER END')
        else:  # orig_run is not None
            # todo vvvv this should be part of preprocessing
            print_separator('USING PREVIOUS RUN LINK')
            result_link = self.orig_run
            local = False
            input_dir = self.fetch_and_extract_inputs_dir(result_link)
            sources_dir = input_dir / Constants.CERTORA_SOURCES
            prover_conf_content = get_conf_from_certora_metadata(input_dir)
            prover_conf = Path(Constants.ORIG_RUN_PROVER_CONF)
            with prover_conf.open('w') as p_conf:
                json.dump(prover_conf_content, p_conf)
            shutil.copy(prover_conf, sources_dir)
            assert self.orig_run_dir
            src = self.orig_run_dir / Constants.TARNAME / Constants.REPORTS / Constants.SPLIT_STATS_DATA
            if src.is_file():
                try:
                    shutil.copy(src, Path.cwd())
                    self.with_split_stats_data = True
                except Exception as e:
                    logging.debug(f"did not manage to copy {Constants.SPLIT_STATS_DATA} from {src.parent} "
                                  f"to current directory {Path.cwd()}: {e}")
            else:
                logging.debug(f"{Constants.SPLIT_STATS_DATA} is not in orig run: {src.parent}")

        # match a generated mutant to a directory where we will apply the diff
        generated_mutants_with_target_dir = []
        for mutant in generated_mutants:
            generated_mutants_with_target_dir.append((mutant, Constants.APPLIED_MUTANTS_DIR / f"mutant{mutant.id}"))
        manual_mutants_with_target_dir = []
        for mutant in self.manual_mutants_list:
            manual_mutants_with_target_dir.append((mutant, Constants.APPLIED_MUTANTS_DIR / f"manual{mutant.id}"))
        all_mutants_with_target_dir = generated_mutants_with_target_dir + manual_mutants_with_target_dir
        mutation_logger.debug("Associated each mutant to a target directory where the mutant will be applied to the "
                              "source code")

        all_mutants = [(mutant, sources_dir, trg_dir) for mutant, trg_dir in all_mutants_with_target_dir]
        for m in all_mutants:
            self.build_mutant_directories(*m)

        self.common_solc_flags = self.get_common_solc_flags()
        for mutant, trg_dir in manual_mutants_with_target_dir:
            self.compile_manual_mutants(mutant, trg_dir)

        web_utils = WebUtils(SimpleNamespace(**vars(self)))
        # get the mutation test id
        mutation_test_id, collect_presigned_url = (
            self.get_mutation_test_id_request(web_utils, len(all_mutants_with_target_dir)))
        mutation_logger.debug(f"Mutation test id: {mutation_test_id}")

        mutation_logger.info("Submit mutations to Prover...")
        print_separator('PROVER START')
        # find out the number of processes. in local runs, we want just one! otherwise,
        # use all CPUs available (set to None)
        if local:
            num_processes_for_mp = 1
            # otherwise, weird things happen locally. this forces us to refresh and get a new executable
            max_task_per_worker = 1
        else:
            num_processes_for_mp = None
            max_task_per_worker = None

        # For debug call run_mutant in same process
        # runs = [(mutant, trg_dir, prover_conf,
        #          mutation_test_id) for mutant, trg_dir in all_mutants_with_target_dir]
        # mutant_runs = []
        # for run in runs:
        #     mutant_runs.append(self.run_mutant(*run))
        with multiprocessing.Pool(processes=num_processes_for_mp, maxtasksperchild=max_task_per_worker) as pool:
            mutant_runs = pool.starmap(self.run_mutant,
                                       [(mutant, trg_dir, prover_conf, mutation_test_id)
                                        for mutant, trg_dir in all_mutants_with_target_dir])

        print_separator('PROVER END')

        mutation_logger.debug("Completed submitting all mutant runs")
        mutation_logger.debug(result_link)
        mutation_logger.debug([dataclasses.asdict(m) for m in mutant_runs])

        # wrap it all up and make the input for the 2nd step: the collector
        # TODO: make sure cannot be None
        assert self.collect_file
        with self.collect_file.open('w+') as collect_file:
            collect_data = {Constants.ORIGINAL: result_link,
                            Constants.MUTANTS: [dataclasses.asdict(m) for m in mutant_runs]}
            json.dump(collect_data, collect_file, indent=4)

        if not self.sync:
            # the new flow. upload the collect_data to the cloud
            self.upload_file_to_cloud_storage(web_utils, collect_presigned_url, collect_data)
            self.print_notification_msg()
        else:
            mutation_logger.info(f"... completed submit phase! Now we poll on {self.collect_file}...")

    def build_mutant_directories(self, mutant: Mutant, src_dir: Path, trg_dir: Path) -> None:
        # first copy src_dir
        Util.safe_copy_folder(src_dir, trg_dir, shutil.ignore_patterns())  # no ignored patterns

        # now apply diff.
        # Remember: we are always running certoraMutate from the project root.
        file_path_to_mutate = trg_dir / Path(mutant.original_filename)
        # 2. apply the mutated file in the newly rooted path
        shutil.copy(mutant.filename, file_path_to_mutate)
        if self.with_split_stats_data and os.environ.get("WITH_AUTOCONFING", False) == '1':
            shutil.copy(Path.cwd() / Constants.SPLIT_STATS_DATA, find_cwd(trg_dir))

    def run_mutant(self, mutant: Mutant, trg_dir: Path, orig_conf: Path, mutation_test_id: str) -> MutantJob:

        with Util.change_working_directory(find_cwd(trg_dir)):
            # we have conf file in sources, let's run from it, it will have proper filepaths
            success, certora_run_result = (
                self.run_certora_prover(orig_conf, mutation_test_id, msg=f"mutant ID: {mutant.id}"))
            if not success or not certora_run_result:
                mutation_logger.warning(f"Failed to run mutant {mutant}")
                return MutantJob(
                    gambit_mutant=mutant,
                    success=success,
                    link=None,
                    run_directory=None,
                    rule_report_link=None
                )

            link = certora_run_result.link
            sources_dir = certora_run_result.src_dir

            return MutantJob(
                gambit_mutant=mutant,
                success=success,
                link=link,
                run_directory=str(sources_dir),
                rule_report_link=certora_run_result.rule_report_link
            )

    def get_solc_version(self, path_to_file: Path) -> str:
        solc = self.prover_dict.get('solc')
        solc_map = self.prover_dict.get('solc_map')
        if solc and solc_map:
            raise Util.CertoraUserInputError("You cannot use both 'solc' and 'solc_map' arguments: "
                                             f"solc is {solc} and solc_map is {solc_map}")
        if solc:
            compiler = solc
        elif solc_map:
            compiler = get_relevant_compiler(path_to_file, Vf.is_solc_file_valid(solc), solc_map)
            if not compiler:
                raise Util.CertoraUserInputError("Cannot resolve Solidity compiler from 'solc' and 'solc_map' for  "
                                                 f"{path_to_file}: solc is {solc} and solc_map: {solc_map}")
        else:
            compiler = Util.DEFAULT_SOLC

        exec_file = shutil.which(compiler)
        if exec_file is None:
            raise Util.CertoraUserInputError(f"{compiler} is not a valid executable")
        return compiler

    def generate_gambit_conf_file(self) -> None:
        """
        the gambit conf file is generated based on the gambit section in the mutation conf file
        plus attributes taken from the prover conf (e.g. packages, solc flags etc)
        :return:
        """
        shared_attributes = {}  # attributes that are shared by all gambit runs
        conf_obj = self.prover_dict
        packages = conf_obj.get('packages', None)
        if packages:
            filtered_packages = \
                [package for package in packages if Path(package.split('=')[1]).exists()]
            shared_attributes['solc_remappings'] = filtered_packages
        if conf_obj.get('solc_optimize', None) or conf_obj.get('solc_optimize_map', None):
            shared_attributes['solc_optimize'] = True  # type: ignore
        if conf_obj.get('solc_allow_path', None):
            shared_attributes['solc_allow_paths'] = True  # type: ignore
        assert self.gambit
        for gambit_obj in self.gambit:
            if Constants.NUM_MUTANTS not in gambit_obj:
                gambit_obj[Constants.NUM_MUTANTS] = DEFAULT_NUM_MUTANTS
            gambit_obj[Constants.SOLC] = self.get_solc_version(Path(gambit_obj[Constants.FILENAME]))
            gambit_obj.update(shared_attributes)
        with Constants.TMP_GAMBIT_PATH.open('w') as f:
            json.dump(self.gambit, f)

    def run_gambit(self) -> List[Mutant]:
        assert self.gambit
        self.generate_gambit_conf_file()

        gambit_exec = get_gambit_exec()
        gambit_args = [gambit_exec, "mutate", "--json", str(Constants.TMP_GAMBIT_PATH),
                       "-o", str(self.outdir)]
        mutation_logger.debug(f"Running gambit: {gambit_args}")
        run_result = \
            subprocess.run(gambit_args, shell=False, universal_newlines=True, stderr=subprocess.PIPE,
                           stdout=subprocess.PIPE)

        if run_result.returncode or run_result.stderr:
            mutation_logger.info(run_result.stdout)
            mutation_logger.info(run_result.stderr)
            raise RuntimeError("Gambit run failed")

        print("Completed gambit run successfully.")

        # read gambit_results.json
        ret_mutants: List[Mutant] = []
        assert self.outdir
        with (self.outdir / "gambit_results.json").open() as gambit_output_json:
            gambit_output = json.load(gambit_output_json)
            for gambit_mutant_data in gambit_output:
                ret_mutants.append(
                    Mutant(
                        filename=str(self.outdir / gambit_mutant_data[Constants.NAME]),
                        original_filename=gambit_mutant_data[Constants.ORIGINAL],
                        # should be relative to re-root in target dir
                        directory=str(self.outdir / Constants.MUTANTS / gambit_mutant_data[Constants.ID]),
                        id=gambit_mutant_data[Constants.ID],
                        diff=gambit_mutant_data[Constants.DIFF],
                        description=gambit_mutant_data[Constants.DESCRIPTION]
                    )
                )

        if Constants.TMP_GAMBIT_PATH.exists() and not self.debug:
            os.remove(Constants.TMP_GAMBIT_PATH)
        if not ret_mutants:
            raise RuntimeError("Could not generate Gambit mutants")
        mutation_logger.debug("Got mutant information")
        return ret_mutants

    def parse_manual_mutations(self) -> List[Mutant]:
        """
        Parse manual mutations from a configuration file.

        Returns:
        - A list of Mutant objects representing the parsed manual mutations.

        Raises:
        - `FileNotFoundError`: If the mutation configuration file does not exist.
        """

        ret_mutants = []
        assert self.manual_mutants
        for mutant in self.manual_mutants:
            file_to_mutate = os.path.normpath(mutant[Constants.FILE_TO_MUTATE])
            path_to_orig = Path(file_to_mutate).resolve()
            if not path_to_orig.exists():
                mutation_logger.info(f"Original file '{path_to_orig}' for manual mutations does not exist"
                                     "Skipping verification.")
                continue
            mutants_location = Path(mutant[Constants.MUTANTS_LOCATION]).resolve()

            if not mutants_location.is_dir():
                raise Util.CertoraUserInputError(
                    f"Manual mutations directory '{mutant[Constants.MUTANTS_LOCATION]}' "
                    f"was not found at {mutants_location}")

            manual_mutants_files = [mm for mm in sorted(mutants_location.glob("*.sol")) if mm.is_file()]
            if not manual_mutants_files:
                raise Util.CertoraUserInputError(
                    f"Could not find any manual mutation files at {mutant[Constants.MUTANTS_LOCATION]}")
            for file in manual_mutants_files:
                manual_mutant_obj = generate_manual_mutant_object(file, file_to_mutate)
                ret_mutants.append(manual_mutant_obj)
        return ret_mutants

    def run_certora_prover(self, conf_file: Path,
                           mutation_test_id: str = "", msg: str = "",
                           is_mutant:  bool = True) -> Tuple[bool, Optional[CertoraRunResult]]:

        if "run_source" in self.prover_dict:
            assert self.prover_dict[Constants.RUN_SOURCE]
            mutation_logger.debug(
                f"Conf object already has a run source: {self.prover_dict[Constants.RUN_SOURCE]}")  # significance?

        certora_args = [str(conf_file), "--run_source", "MUTATION", "--msg", msg,]
        if self.wait_for_original_run:
            certora_args.append("--wait_for_results")
        if self.with_split_stats_data and os.environ.get("WITH_AUTOCONFING", False) == '1':
            certora_args += ['--prover_resource_files', f"ac:{Constants.SPLIT_STATS_DATA}"]
        if mutation_test_id:
            certora_args.extend(["--mutation_test_id", mutation_test_id])
        if self.server:
            certora_args.extend(["--server", self.server])
        if self.prover_version:
            certora_args.extend(["--prover_version", self.prover_version])

        if is_mutant:
            certora_args.extend(["--disable_local_typechecking"])
        mutation_logger.debug(f"Running the Prover: {certora_args}")
        try:
            certora_run_result = run_certora(certora_args, True)
        except Exception as e:
            mutation_logger.warning(f"Failed to run with {e}")
            return False, None

        return True, certora_run_result

    def load_test_results(self) -> TestJobs:
        """
        Load mutation testing results from a collection file.

        Returns:
        - A tuple containing the original URL and a list of mutants' results.

        Raises:
        - `Util.CertoraUserInputError`: Raised if the collection file is missing, or if the original URL or mutants
          information is not found in the collection file.
        """
        if not self.collect_file or not self.collect_file.exists():
            raise Util.CertoraUserInputError(
                f"Cannot collect results, as file to collect from {self.collect_file} does not exist.")

        with self.collect_file.open() as collect_handle:
            results_work = json.load(collect_handle)

        if Constants.ORIGINAL not in results_work:
            raise Util.CertoraUserInputError(f"Could not find original url in {self.collect_file}.")

        if Constants.MUTANTS not in results_work:
            raise Util.CertoraUserInputError(f"Could not find mutants in {self.collect_file}.")

        mutation_logger.info(f"Collecting results from {self.collect_file}...")

        test_results = TestJobs(**results_work)

        return test_results

    def get_report_fetcher(self, url: str) -> ReportFetcher:
        """
        :param url: Either a file path or a web url
        :return: FileFetcher if the url is a file path, WebFetcher otherwise
        """
        if validate_dir(url):
            return FileFetcher()
        web_utils = WebUtils(SimpleNamespace(**vars(self)))
        return WebFetcher(web_utils, self.debug)

    def fetch_mutant_results(self, mutants_jobs: List[MutantJob], fetcher: ReportFetcher) \
            -> Optional[List[MutantJobWithResults]]:
        """
        Retrieve mutant verification results for a list of mutant job objects.

        Parameters:
        - `mutants_jobs`: A list of mutant jobs to fetch results for.
        - `fetcher`: An instance of the ReportFetcher class used to fetch mutant results.

        Returns:
        - A list of mutant results - tuples containing mutant objects and their corresponding results.
        - None if at least one of the mutant verification jobs did not terminate
        """
        all_mutants_results: List[MutantJobWithResults] = []
        for mutant_job in mutants_jobs:
            mutant_link = mutant_job.link

            mutant_run_results = self.get_results(mutant_link, fetcher)
            if mutant_run_results is None:
                return None

            mutant_results = MutantJobWithResults(mutant_job, mutant_run_results)

            all_mutants_results.append(mutant_results)
            mutation_logger.info(f"Successfully retrieved results for mutant {mutant_job}")
        return all_mutants_results

    def collect(self) -> bool:
        """
        Returns true if finished collecting.
        Returns false if not, but there's still a chance something will return.
        Will exit with exitcode 1 if something is broken in the collect file.
        """
        orig_collect_success = True
        test_results = self.load_test_results()

        if test_results.original is None or (not valid_link(test_results.original)):
            mutation_logger.info("There is no original URL - nothing to collect.")
            orig_collect_success = False

        fetcher = self.get_report_fetcher(test_results.original)

        original_results = self.get_results(test_results.original, fetcher)
        original_run_results: Optional[MutantJobWithResults] = None

        if original_results:
            # Construct a MutantJobWithResults for the original run
            orig_mutant = Mutant(
                description="The original file without any mutation",
                diff="",
                id=Constants.ORIGINAL,
                directory="",
                filename="",
                original_filename="",
                name=Constants.ORIGINAL
            )
            mutant_job = MutantJob(
                gambit_mutant=orig_mutant,
                link=test_results.original,
                success=True,
                run_directory="",
                rule_report_link=test_results.original.replace(Constants.JOB_STATUS, Constants.OUTPUT)
            )
            original_run_results = MutantJobWithResults(
                mutant_job=mutant_job,
                rule_results=original_results
            )
        else:
            orig_collect_success = False
            if not self.sync:
                mutation_logger.warning("Failed to get results for original. The report will not be generated.")

        # check the mutant URLs
        mutant_collect_success = True

        if any(not ((mutant.link is None or valid_link(mutant.link)) and mutant.success)
               for mutant in test_results.mutants
               ):
            mutation_logger.warning(f"There are some bad mutant URLs. Check {self.collect_file} to see if some are "
                                    "null or invalid.")
            mutant_collect_success = False

        # build mutants object with the rule results
        mutants_results = self.fetch_mutant_results(test_results.mutants, fetcher)

        if mutants_results is None:  # At least one run did not terminate yet
            mutant_collect_success = False

        else:
            if self.dump_failed_collects:
                with self.dump_failed_collects.open('w+') as failed_collection:
                    # add mutants
                    for mutant_result in mutants_results:
                        if not mutant_result.mutant_job.success:
                            mutant_collect_success = False
                            failed_collection.write(f"{mutant_result}\n\n")

        if not mutant_collect_success and not self.sync:
            mutation_logger.warning(f"Failed to get results for some mutants. See {self.dump_failed_collects} "
                                    f"and try to manually run the prover on them to see the outcome.")

        if orig_collect_success and mutant_collect_success:

            mutants_results_not_none: List[MutantJobWithResults] = mutants_results or []
            assert original_run_results is not None

            if self.ui_out is not None:

                # By the boolean conditions above we know these are not None
                self.write_report_file(original_run_results, mutants_results)  # type: ignore

            if self.dump_csv:
                # By the boolean conditions above we know mutants_results is not None
                all_results = [original_run_results] + mutants_results_not_none  # type: ignore
                self.generate_csv(all_results)
            mutation_logger.info("Done successfully collecting results!")

        return orig_collect_success and mutant_collect_success

    def generate_csv(self, verification_results: List[MutantJobWithResults]) -> None:
        """
        Generate a CSV file based on mutation testing results and verification results of the original unmutated file.
        We also include results of failed runs, or rules that failed in the original run.

        The CSV will be of form:
            RULENAME,Original, 1_C, 2_C, ..., C.m1, C.m2, ...
            rule_name,SUCCESS,FAIL,TIMEOUT/UNKNOWN,....
        """
        if not verification_results:
            mutation_logger.warning("Could not write csv file as mutation test results are empty")
            return

        try:
            assert self.dump_csv
            with self.dump_csv.open(mode='w', newline='') as ui_out_csv:
                wr = csv.writer(ui_out_csv, delimiter=',')

                # Writing csv header - ruleName,Original,1_C,2_C,...,C.m1,C.m2,...
                # Rule names are the same for every mutant
                row1 = [Constants.RULENAME] + [str(mutant.mutant_job) for mutant in verification_results]
                wr.writerow(row1)

                # We construct a row per rule
                rule_names = verification_results[0].all_rule_names()  # Same rule names for all mutants
                for rule_name in rule_names:
                    statuses = [mutant.get_rule_status(rule_name) for mutant in
                                verification_results]  # A list of a status per rule
                    row = [rule_name] + statuses
                    wr.writerow(row)

        except Exception as e:
            mutation_logger.warning(f"Failed to output csv to {self.dump_csv}.")
            mutation_logger.debug(f"{e}")

    def write_report_file(self, original_result: MutantJobWithResults,
                          mutants_results: List[MutantJobWithResults]) -> None:
        """
        Write a report file based on the original and mutated results.

        Parameters:
        - original_result (MutantJobWithResults): The result of the original job.
        - mutants_results (List[MutantJobWithResults]): List of mutated job results.

        This function filters out rules that failed on the original job and generates a report file.
        The report file includes relevant information from the mutated results, and does not include the original.

        Note: The generated report is stored in the 'ui_out' file.
        """
        rule_statuses_to_show = [MutationTestRuleStatus.SUCCESS, MutationTestRuleStatus.SANITY_FAIL]
        valid_rule_names = set([result.name for result in original_result.rule_results
                                if result.status in rule_statuses_to_show])

        filtered_mutation_results: List[dict] = []  # Doesn't include rules that failed on original

        for mutant_result in mutants_results:
            relevant_rule_results = \
                [rule_result for rule_result in mutant_result.rule_results if
                 rule_result.name in valid_rule_names]

            if relevant_rule_results:
                mutant_job_with_result = MutantJobWithResults(
                    mutant_job=mutant_result.mutant_job,
                    rule_results=relevant_rule_results
                )
                filtered_mutation_results.append(mutant_job_with_result.to_report_json())

        assert self.ui_out
        store_in_a_file(filtered_mutation_results, self.ui_out)

    def get_mutation_test_id_request(self, web_utils: WebUtils, mutants_number: int) -> Tuple[str, str]:
        mutation_logger.debug(f"Getting mutation test ID for {mutants_number} mutants")
        url = web_utils.mutation_test_id_url
        body = {"mutants_number": mutants_number}  # type: Dict[str, Any]
        if self.orig_run:
            # we have validated this URL before
            parsed_url = urllib3.util.parse_url(self.orig_run)
            # ignore the query parameters and fragments
            body["original_job_url"] = f"{parsed_url.scheme}://{parsed_url.hostname}{parsed_url.path}"
        if self.msg:
            body["msg"] = self.msg
        mutation_logger.debug("Sending a PUT request with the body:", body)
        resp = web_utils.put_json_request_with_timeout(url, body, headers=auth_headers)
        if resp is None:
            raise Util.CertoraUserInputError("failed to send mutation test to server")
        mutation_logger.debug(f"Got mutation test ID response: {resp.status_code}")
        resp_obj = resp.json()
        if Constants.ID not in resp_obj or Constants.COLLECT_SIGNED_URL not in resp_obj:
            # todo: check how {resp_obj} looks like -- rahav
            raise RuntimeError(f"invalid server response, mutation test failed: {resp_obj}")

        return resp_obj[Constants.ID], resp_obj[Constants.COLLECT_SIGNED_URL]

    def get_mutation_test_final_report_url(self, web_utils: WebUtils) -> Tuple[str, str, str]:
        mutation_logger.debug("Getting mutation test final report URL")
        url = web_utils.mutation_test_submit_final_result_url
        resp = web_utils.get_response_with_timeout(url, cookies=default_cookies)
        if not resp:
            raise Util.CertoraUserInputError("Failed to get the mutation test report URL")
        mutation_logger.debug(f"Got response: {resp.status_code}")
        resp_obj = resp.json()
        if Constants.ID in resp_obj and Constants.PRE_SIGNED_URL in resp_obj and Constants.ANONYMOUS_KEY in resp_obj:
            return resp_obj[Constants.ID], resp_obj[Constants.ANONYMOUS_KEY], resp_obj[Constants.COLLECT_SIGNED_URL]
        else:
            raise RuntimeError(f"Couldn't generate the report URL: {resp_obj}")

    def upload_file_to_cloud_storage(self, web_utils: WebUtils, presigned_url: str, data: Any) -> None:
        mutation_logger.debug("Uploading file")
        headers = {"Content-Type": "application/json"}
        put_resp = web_utils.put_response_with_timeout(presigned_url, json.dumps(data), headers)
        if not put_resp:
            raise ConnectionError(f"Failed to submit to presigned URL. url - {presigned_url}")
        mutation_logger.debug(f"Upload file finished with: {put_resp.status_code}")
        if put_resp.status_code != 200:
            raise ConnectionError(f"Failed to submit to presigned URL, status code {put_resp.status_code}")

    def get_results(self, link: Optional[str], fetcher: ReportFetcher) -> Optional[List[RuleResult]]:
        """
        :param link: A link to get the results from
        :param fetcher:
        :return: - A list of rule verification results. The list will be empty in case of an error.
                   If the job did not terminate yet, returns None.
        """
        if link is None:
            return list()
        if isinstance(fetcher, FileFetcher):
            output_json = fetcher.get_output_json(link)
            if output_json is None:
                mutation_logger.warning(f"failed to get results for {link}")
                return list()

            if Constants.RULES not in output_json:
                mutation_logger.debug(f"Bad format for {Constants.OUTPUTJSON}")
                return list()
        elif isinstance(fetcher, WebFetcher):
            job_data = fetcher.get_job_data(link)
            if job_data is None:
                mutation_logger.warning(f"failed to get job data for {link}")
                return list()

            job_status = job_data.get(Constants.JOB_STATUS, "")
            if job_status not in FinalJobStatus.get_statuses():
                # The job is not completed yet
                return None
        else:
            raise RuntimeError("Unexpected format. Can't proceed with the request. Please contact Certora")

        # now we no longer use output_json

        progress_json = fetcher.get_treeview_json(link)
        if progress_json is None:
            mutation_logger.debug("Could not get progress object")
            return list()
        top_level_rules = get_top_level_rules(progress_json)
        if top_level_rules is None:
            mutation_logger.debug("Could not get tree view object")
            return list()
        rule_results = []

        for rule in top_level_rules:
            # as long as we have children, we need to keep looking.
            # we prioritize failures, then unknown, then timeout, then sanity_fail, and only all success is a success
            if Constants.CHILDREN not in rule:
                mutation_logger.debug(f"Bad format for a rule {rule}")
                return list()

            if Constants.NAME not in rule:
                mutation_logger.debug(f"Bad format for a rule {rule}")
                return list()

            leaf_statuses: List[str] = []
            rec_collect_statuses_children(rule, leaf_statuses)
            name = rule[Constants.NAME]
            if len(leaf_statuses) == 0:
                mutation_logger.warning("Got no rule results")
                return list()
            elif any([s == MutationTestRuleStatus.FAIL for s in leaf_statuses]):
                rule_results.append(RuleResult(name, MutationTestRuleStatus.FAIL.value))
            elif any([s == MutationTestRuleStatus.UNKNOWN for s in leaf_statuses]):
                rule_results.append(RuleResult(name, MutationTestRuleStatus.UNKNOWN.value))
            elif any([s == MutationTestRuleStatus.TIMEOUT for s in leaf_statuses]):
                rule_results.append(RuleResult(name, MutationTestRuleStatus.TIMEOUT.value))
            elif any([s == MutationTestRuleStatus.SANITY_FAIL for s in leaf_statuses]):
                rule_results.append(RuleResult(name, MutationTestRuleStatus.SANITY_FAIL.value))
            elif not all([s == MutationTestRuleStatus.SUCCESS for s in leaf_statuses]):
                mutation_logger.debug("Encountered a new unknown status which isn't FAIL, UNKNOWN, TIMEOUT, "
                                      "SANITY_FAIL, or SUCCESS")
                return list()
            else:
                rule_results.append(RuleResult(name, MutationTestRuleStatus.SUCCESS.value))

        return rule_results

    def print_final_report_url_msg(self, url: str, mutation_id: str, anonymous_key: str) -> None:
        final_url = f"{url}?id={mutation_id}&{Constants.ANONYMOUS_KEY}={anonymous_key}"
        Console().print("\n\n[bold orange4]Final mutation report is available at "
                        f"[link={final_url}]{final_url}[/link]\n")

        if self.dump_link:
            with open(self.dump_link, "w") as file:
                file.write(final_url)

    def poll_collect(self) -> None:
        SECONDS_IN_MINUTE = 60
        assert self.poll_timeout
        poll_timeout_seconds = self.poll_timeout * SECONDS_IN_MINUTE
        start = time.time()
        duration = 0  # seconds
        attempt_number = 1
        retry = 15
        ready = False
        while duration < poll_timeout_seconds:
            mutation_logger.info(f"-------> Trying to poll results... attempt #{attempt_number}")
            ready = self.collect()
            if not ready:
                mutation_logger.info(f"-------> Results are still not ready, trying in {retry} seconds")
                attempt_number += 1
                time.sleep(retry)
            else:
                # upload the results file to the cloud
                assert self.ui_out
                final_report = read_file(self.ui_out)
                web_utils = WebUtils(SimpleNamespace(**vars(self)))
                id, anonymous_key, pre_signed_url = self.get_mutation_test_final_report_url(web_utils)
                self.upload_file_to_cloud_storage(web_utils, pre_signed_url, final_report)
                self.print_final_report_url_msg(web_utils.mutation_test_final_result_url, id, anonymous_key)
                return
            duration = int(time.time() - start)

        if not ready:
            raise Util.CertoraUserInputError(f"Could not get results after {attempt_number} attempts.")

    def get_args(self, args_list: List[str]) -> None:
        attr_dict = Attr.get_args(args_list)
        for key, value in attr_dict.items():
            setattr(self, key, value)

    def set_conf_file(self) -> None:
        confs = [conf for conf in (self.conf, self.conf_no_flag) if conf is not None]
        confs_len = len(confs)
        if confs_len > 1:
            raise Util.CertoraUserInputError(f"Only one conf can be defined. Got: {confs}")
        elif confs_len == 1:
            self.conf = confs[0]

    def read_conf_file(self) -> None:

        with self.conf.open() as conf_file:
            try:
                self.prover_dict = json5.load(conf_file, allow_duplicate_keys=False)
            except Exception as e:
                raise Util.CertoraUserInputError(f"Parsing error in JSON file {conf_file}: {e}")

            try:
                mutation_obj = self.prover_dict[Constants.MUTATIONS]
            except KeyError:
                raise Util.CertoraUserInputError(f"Missing 'Mutations' object in conf file {conf_file}")

            for option in mutation_obj:
                if hasattr(self, option):
                    val = getattr(self, option)
                    if val is None or val is False or val == []:
                        setattr(self, option, mutation_obj[option])
                else:
                    raise Util.CertoraUserInputError(f"Unknown key, {option}, under 'Mutations' in the conf file ")

    def get_common_solc_flags(self) -> List[str]:
        common_flags = []

        if self.prover_dict.get(Constants.SOLC_OPTIMIZE, False) or \
                self.prover_dict.get(Constants.SOLC_OPTIMIZE_MAP, False):
            common_flags.extend(['--optimize'])

        if self.prover_dict.get(Constants.SOLC_ALLOW_PATH):
            common_flags.extend([f'--allow-paths, {self.prover_dict.get(Constants.SOLC_ALLOW_PATH)}'])

        if self.prover_dict.get(Constants.SOLC_EVM_VERSION):
            common_flags.extend(['--evm-version',  self.prover_dict.get(Constants.SOLC_EVM_VERSION, '')])

        if self.prover_dict.get(Constants.PACKAGES):
            common_flags.extend(self.prover_dict.get(Constants.PACKAGES, ''))

        return common_flags

    def compile_manual_mutants(self, mutant: Mutant, trg_dir: Path) -> None:

        if not self.manual_mutants:
            return

        solc = self.get_solc_version(trg_dir)

        via_ir_flag = []
        if self.prover_dict.get(Constants.SOLC_VIA_IR) or \
                self.prover_dict.get(Constants.SOLC_EXPERIMENTAL_VIA_IR):
            via_ir_flag = [Util.get_ir_flag(solc)]

        args = [solc] + self.common_solc_flags + via_ir_flag + [mutant.original_filename]
        if self.test == str(Util.TestValue.CHECK_MANUAL_COMPILATION):
            raise Util.TestResultsReady(' '.join(args))
        with Util.change_working_directory(find_cwd(trg_dir)):
            result = subprocess.run(args, stdout=subprocess.DEVNULL, stderr=subprocess.STDOUT)
        if result.returncode:
            mutation_logger.debug(f"mutation compilation failed: cmd: {' '.join(args)}\n cwd: {os.getcwd()}")
            raise Util.CertoraUserInputError(f"mutation file {mutant.original_filename} failed to compile")


def rec_collect_statuses_children(rule: Dict[str, Any], statuses: List[str]) -> None:
    statuses.append(convert_to_mutation_testing_status(rule[Constants.STATUS]))
    for child in rule[Constants.CHILDREN]:
        rec_collect_statuses_children(child, statuses)


def get_file_url_from_orig_url(url: str, file: str) -> str:
    parsed_url = urllib3.util.parse_url(url)
    file_url = f"{parsed_url.scheme}://{parsed_url.hostname}{parsed_url.path}"
    # ensure there is a single slash
    if not file_url.endswith("/"):
        file_url += "/"
    file_url += f"{file}?{parsed_url.query}"
    return file_url


def get_top_level_rules(progress_json: Dict[str, Any]) -> Optional[List[Dict[str, Any]]]:
    if Constants.VERIFICATION_PROGRESS not in progress_json:
        mutation_logger.debug(f"Could not find {Constants.VERIFICATION_PROGRESS} in progress {progress_json}")
        return None
    # verification progress holds a string which is a json encoding of the latest tree view file
    tree_view_json = json.loads(progress_json[Constants.VERIFICATION_PROGRESS])
    if Constants.RULES not in tree_view_json:
        mutation_logger.debug(f"Could not find rules in tree view file {tree_view_json}")
        return None
    return tree_view_json[Constants.RULES]


certora_key = os.getenv(KEY_ENV_VAR, '')
auth_headers = {"Authorization": f"Bearer {certora_key}", "Content-Type": "application/json"}
default_cookies = {str(Constants.CERTORA_KEY): certora_key}


class FileFetcher(ReportFetcher):

    # in the file fetcher, all links are to the main emv directory
    def get_output_json(self, link: str) -> Optional[Dict[str, Any]]:
        output_path = Path(link) / Constants.REPORTS / Constants.OUTPUTJSON
        if not output_path.is_file():
            mutation_logger.info(f"Got no {Constants.OUTPUTJSON} file")
            return None

        with open(output_path, 'r') as output_handle:
            output_json = json.load(output_handle)

        return output_json

    def get_treeview_json(self, link: str) -> Optional[Dict[str, Any]]:
        # it's a hack, but in web we need to go through the verificationProgress and locally we don't.
        treeview_path = Path(link) / Constants.REPORTS / "treeView"

        # look out for the "latest" tree view file
        candidates = list(treeview_path.glob(r"treeViewStatus_*.json"))
        max = None
        max_no = -1
        for candidate in candidates:
            if candidate.is_file():
                index = int(str(candidate.stem).split("_")[1])
                if index > max_no:
                    max = candidate
                    max_no = index
        # max should contain the latest tree view file
        if max is None:
            mutation_logger.debug("No matching treeview files found")
            return None

        treeview_file = max
        with open(treeview_file, 'r') as treeview_handle:
            treeview_str = json.load(treeview_handle)

        # wrap the json as a string inside another json mimicking progress URL
        return {Constants.VERIFICATION_PROGRESS: json.dumps(treeview_str)}


def download_report_file(report_url: str, filename: str) -> bool:
    """
    Copy a file from the "Reports" folder to the current working dir with the same file name
    url to specific report file is of the form https://<server>/output/NNN/MMM/<filename>?anonymousKey=PPPPP
    (i.e. in the report url add the filename before ?anonymousKey)
    False is returned if the file was not found or if the download did not succeed
    """
    time.sleep(5)  # make sure the file was written
    url = report_url.replace('?', f'/{filename}?')
    try:
        response = requests.get(url)
    except requests.exceptions.RequestException as e:
        logging.debug(f"request for {url} failed: {e}")
        return False
    if response.status_code == 200:
        with open(filename, 'wb') as file:
            file.write(response.content)
        logging.debug(f"{filename} downloaded from {report_url}")
    elif response.status_code == 404:  # file not found is not an error
        mutation_logger.debug(f"{filename} not found in {report_url}")
        return False
    else:
        mutation_logger.debug(f"Failed to download {filename} from {report_url}. Status code: {response.status_code}")
        return False
    return True


class WebFetcher(ReportFetcher):
    def __init__(self, _web_utils: WebUtils, debug: bool = False):
        self.web_utils = _web_utils
        self.verification_report_path_pattern = re.compile(r"^\/output\/\d+\/[0-9a-fA-F]*(\/)?$")
        self.job_status_path_pattern = re.compile(r"^\/jobStatus\/\d+\/[0-9a-fA-F]*(\/)?$")
        self.debug = debug

    @staticmethod
    def get_url_path(url: str) -> Optional[str]:
        parsed_url = urllib3.util.parse_url(url)
        return parsed_url.path

    def get_resource_url(self, url: str, keyword: str) -> str:
        url_path = self.get_url_path(url)
        if not url_path:
            raise Util.CertoraUserInputError(f"Invalid URL was provided: {url}")

        # we check both the status page and the verification report for
        # backward compatibility
        if re.match(self.verification_report_path_pattern, url_path):
            resource_url = url.replace(Constants.OUTPUT, keyword)
        elif re.match(self.job_status_path_pattern, url_path):
            resource_url = url.replace(Constants.JOB_STATUS, keyword)
        else:
            raise Util.CertoraUserInputError(f"Unknown URL was provided: {url}")
        return resource_url

    def get_output_json(self, url: str) -> Optional[Dict[str, Any]]:
        output_url = self.get_resource_url(url, Constants.OUTPUT)
        return self.parse_response(
            get_file_url_from_orig_url(output_url, Constants.OUTPUTJSON),
            Constants.OUTPUTJSON
        )

    def get_treeview_json(self, url: str) -> Optional[Dict[str, Any]]:
        progress_url = self.get_resource_url(url, Constants.PROGRESS)
        return self.parse_response(progress_url, "treeview.json")

    def get_job_data(self, url: str) -> Optional[Dict[str, Any]]:
        job_data_url = self.get_resource_url(url, Constants.JOB_DATA)
        return self.parse_response(job_data_url, "job data")

    def parse_response(self, url: str, resource: str) -> Optional[Dict[str, Any]]:
        response = self.web_utils.get_response_with_timeout(url)
        if response is None or response.status_code != 200:
            mutation_logger.debug(f"Got bad response code when fetching {resource} "
                                  f"{response.status_code if response else ''}")
            return None
        return response.json()
